package ru.yandex.direct.core.entity.organizations.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record1;
import org.jooq.ResultQuery;
import org.jooq.Row2;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.banner.model.old.OldBanner;
import ru.yandex.direct.core.entity.banner.model.old.OldBannerWithOrganization;
import ru.yandex.direct.core.entity.organization.model.BannerPermalink;
import ru.yandex.direct.core.entity.organization.model.Organization;
import ru.yandex.direct.core.entity.organization.model.OrganizationStatusPublish;
import ru.yandex.direct.core.entity.organization.model.PermalinkAssignType;
import ru.yandex.direct.core.entity.organizations.util.PermalinkUtils;
import ru.yandex.direct.dbschema.ppc.enums.BannersStatusarch;
import ru.yandex.direct.dbschema.ppc.tables.records.BannerPermalinksRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.OrganizationsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.core.entity.organization.model.OrganizationStatusPublish.UNKNOWN;
import static ru.yandex.direct.core.entity.organization.model.PermalinkAssignType.MANUAL;
import static ru.yandex.direct.dbschema.ppc.enums.BannerPermalinksPermalinkAssignType.auto;
import static ru.yandex.direct.dbschema.ppc.enums.BannerPermalinksPermalinkAssignType.manual;
import static ru.yandex.direct.dbschema.ppc.tables.BannerPermalinks.BANNER_PERMALINKS;
import static ru.yandex.direct.dbschema.ppc.tables.BannerPhones.BANNER_PHONES;
import static ru.yandex.direct.dbschema.ppc.tables.Banners.BANNERS;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignPermalinks.CAMPAIGN_PERMALINKS;
import static ru.yandex.direct.dbschema.ppc.tables.CampaignPhones.CAMPAIGN_PHONES;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Organizations.ORGANIZATIONS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Репозиторий для работы с организациями из Справочника.
 */
@Repository
@ParametersAreNonnullByDefault
public class OrganizationRepository {
    public static final int BANNER_IDS_CHUNK_SIZE = 2500;

    private final DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<Organization> organizationMapper;
    private final JooqMapperWithSupplier<BannerPermalink> permalinkAssignTypeMapper;
    private final Set<Field<?>> fieldsToRead;
    private final Set<Field<?>> permalinkAssignFieldsToRead;

    @Autowired
    public OrganizationRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;

        organizationMapper = OrganizationRepositoryMapperProvider.createOrganizationMapper();
        permalinkAssignTypeMapper = OrganizationRepositoryMapperProvider.createBannerPermalinkMapper();
        fieldsToRead = organizationMapper.getFieldsToRead();
        permalinkAssignFieldsToRead = permalinkAssignTypeMapper.getFieldsToRead();
        permalinkAssignFieldsToRead.add(BANNER_PERMALINKS.BID);
    }

    /**
     * Получить все организации клиента.
     */
    @Nonnull
    public List<Organization> getAllClientOrganizations(int shard, ClientId clientId) {
        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(ORGANIZATIONS)
                .where(ORGANIZATIONS.CLIENT_ID.eq(clientId.asLong()))
                .fetch(organizationMapper::fromDb);
    }

    /**
     * Выбрать организации с указанными ID пермалинков.
     */
    @Nonnull
    public Map<Long, List<Organization>> getOrganizationsByPermalinkIds(int shard,
                                                                        Collection<Long> permalinkIds) {
        if (permalinkIds.isEmpty()) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(ORGANIZATIONS)
                .where(ORGANIZATIONS.PERMALINK_ID.in(permalinkIds))
                .fetchGroups(ORGANIZATIONS.PERMALINK_ID, organizationMapper::fromDb);
    }

    /**
     * Выбрать организации, привязанные к указанным баннерам.
     */
    @Nonnull
    public Map<Long, Organization> getOrganizationsByBannerIds(int shard,
                                                               Collection<Long> bannerIds) {
        return getOrganizationsByBannerIds(shard, (Long) null, bannerIds);
    }

    @Nonnull
    public Map<Long, List<Long>> getPermalinkIdsByCampaignId(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }

        return getPermalinkIdsByCampaignId(dslContextProvider.ppc(shard), campaignIds);
    }

    @Nonnull
    public Map<Long, List<Long>> getPermalinkIdsByCampaignId(DSLContext context, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }

        return context
                .select(BANNERS.CID, BANNER_PERMALINKS.PERMALINK)
                .from(BANNERS)
                .join(BANNER_PERMALINKS).on(BANNER_PERMALINKS.BID.eq(BANNERS.BID)
                        .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(manual)))
                .where(BANNERS.CID.in(campaignIds))
                .fetchGroups(BANNERS.CID, BANNER_PERMALINKS.PERMALINK);
    }

    /**
     * Выбрать организации, привязанные к указанным баннерам.
     */
    @Nonnull
    public Map<Long, Organization> getOrganizationsByBannerIds(int shard,
                                                               ClientId clientId,
                                                               Collection<Long> bannerIds) {
        return getOrganizationsByBannerIds(shard, clientId.asLong(), bannerIds);
    }

    @Nonnull
    private Map<Long, Organization> getOrganizationsByBannerIds(int shard,
                                                                @Nullable Long clientId,
                                                                Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyMap();
        }

        Set<Field<?>> fieldsToRead = new HashSet<>(this.fieldsToRead);
        fieldsToRead.add(BANNER_PERMALINKS.BID);
        SelectConditionStep<Record> selectStep = dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(BANNERS)
                .join(CAMPAIGNS).on(BANNERS.CID.eq(CAMPAIGNS.CID))
                .join(BANNER_PERMALINKS).on(BANNER_PERMALINKS.BID.eq(BANNERS.BID)
                        .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(manual)))
                .join(ORGANIZATIONS).on(ORGANIZATIONS.PERMALINK_ID.eq(BANNER_PERMALINKS.PERMALINK)
                        .and(ORGANIZATIONS.CLIENT_ID.eq(CAMPAIGNS.CLIENT_ID)))
                .where(BANNERS.BID.in(bannerIds));

        if (clientId != null) {
            selectStep = selectStep.and(ORGANIZATIONS.CLIENT_ID.eq(clientId));
        }

        return selectStep
                .fetchMap(BANNER_PERMALINKS.BID, organizationMapper::fromDb);
    }

    @Nonnull
    public Map<Long, Long> getPermalinkIdsByBannerIds(int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyMap();
        }

        return getPermalinkIdsByBannerIds(dslContextProvider.ppc(shard), bannerIds);
    }

    @Nonnull
    public Map<Long, Long> getPermalinkIdsByBannerIds(DSLContext context, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyMap();
        }

        Map<Long, Long> result = new HashMap<>();
        for (var chunkedBannerIds : Iterables.partition(bannerIds, BANNER_IDS_CHUNK_SIZE)) {
            result.putAll(
                    context
                            .select(BANNER_PERMALINKS.BID, BANNER_PERMALINKS.PERMALINK)
                            .from(BANNERS)
                            .join(BANNER_PERMALINKS).on(BANNER_PERMALINKS.BID.eq(BANNERS.BID)
                                    .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(manual)))
                            .where(BANNERS.BID.in(chunkedBannerIds))
                            .fetchMap(BANNER_PERMALINKS.BID, BANNER_PERMALINKS.PERMALINK)
            );
        }
        return result;
    }

    @Nonnull
    public Map<Long, Long> getDefaultPermalinkIdsByCampaignId(int shard, Collection<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }
        Map<Long, Long> result = new HashMap<>();
        for (var chunkedCampaignIds : Iterables.partition(campaignIds, BANNER_IDS_CHUNK_SIZE)) {
            result.putAll(
                    dslContextProvider.ppc(shard)
                            .select(CAMPAIGN_PERMALINKS.CID, CAMPAIGN_PERMALINKS.PERMALINK_ID)
                            .from(CAMPAIGN_PERMALINKS)
                            .where(CAMPAIGN_PERMALINKS.CID.in(chunkedCampaignIds))
                            .fetchMap(CAMPAIGN_PERMALINKS.CID, CAMPAIGN_PERMALINKS.PERMALINK_ID)
            );
        }
        return result;
    }

    /**
     * Добавляет в базу организации и проставляет их связку с баннером.
     */
    public void addOrUpdateAndLinkOrganizations(int shard, Map<Long, Organization> orgsByBannerId) {
        if (orgsByBannerId.isEmpty()) {
            return;
        }

        dslContextProvider.ppcTransaction(shard, conf -> addOrUpdateAndLinkOrganizations(conf.dsl(), orgsByBannerId));
    }

    /**
     * Добавляет в базу организации и проставляет их связку с баннером.
     */
    public void addOrUpdateAndLinkOrganizations(DSLContext context, Map<Long, Organization> orgsByBannerId) {
        if (orgsByBannerId.isEmpty()) {
            return;
        }

        addOrUpdateOrganizations(context, orgsByBannerId.values());
        linkOrganizationsToBanners(context,
                EntryStream.of(orgsByBannerId)
                        .mapValues(Organization::getPermalinkId)
                        .toMap());
    }

    /**
     * Добавляет/обновляет переданные организации.
     */
    public void addOrUpdateOrganizations(int shard, Collection<Organization> organizations) {
        if (organizations.isEmpty()) {
            return;
        }

        addOrUpdateOrganizations(dslContextProvider.ppc(shard), organizations);
    }

    /**
     * Добавляет/обновляет переданные организации.
     */
    public void addOrUpdateOrganizations(DSLContext context, Collection<Organization> organizations) {
        if (organizations.isEmpty()) {
            return;
        }

        // Ситуация ненормальная, но возможно где-то в справочнике что-то пошло не так.
        StreamEx.of(organizations)
                .filter(org -> org.getStatusPublish() == null)
                .forEach(org -> org.withStatusPublish(UNKNOWN));

        InsertHelper<OrganizationsRecord> insertHelper =
                new InsertHelper<>(context, ORGANIZATIONS);

        insertHelper.addAll(organizationMapper, organizations);

        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(ORGANIZATIONS.CHAIN_ID, MySQLDSL.values(ORGANIZATIONS.CHAIN_ID));
            insertHelper.execute();
        }
    }

    /**
     * Создает связки ID баннера с ID пермалинка
     */
    public void linkOrganizationsToBanners(int shard, Map<Long, Long> permalinkIdByBannerId) {
        linkOrganizationsToBanners(shard, permalinkIdByBannerId, MANUAL);
    }

    /**
     * Создает связки ID баннера с ID пермалинка
     */
    public void linkOrganizationsToBanners(int shard, Map<Long, Long> permalinkIdByBannerId,
                                           PermalinkAssignType assignType) {
        if (permalinkIdByBannerId.isEmpty()) {
            return;
        }

        linkOrganizationsToBanners(dslContextProvider.ppc(shard), permalinkIdByBannerId, assignType);
    }

    @Deprecated
    public void linkOrganizationsToBanners(DSLContext context, Collection<? extends OldBannerWithOrganization> banners,
                                           Set<Long> bannerIdsWithEnabledPreferVCardOverPermalink) {
        if (banners.isEmpty()) {
            return;
        }

        Map<Long, Long> permalinkIdByBannerId = StreamEx.of(banners)
                .mapToEntry(OldBanner::getId, OldBannerWithOrganization::getPermalinkId)
                .nonNullValues()
                .toMap();

        Set<Long> bannerIdsWithVCardPreferFlag = StreamEx.of(banners)
                .filter(PermalinkUtils::calculatePreferVCardOverPermalink)
                .map(OldBanner::getId)
                .filter(bannerIdsWithEnabledPreferVCardOverPermalink::contains)
                .toSet();

        linkOrganizationsToBanners(context, permalinkIdByBannerId, bannerIdsWithVCardPreferFlag);
    }

    /**
     * Создает связки ID баннера с ID пермалинка
     */
    public void linkOrganizationsToBanners(DSLContext context, Map<Long, Long> permalinkIdByBannerId) {
        linkOrganizationsToBanners(context, permalinkIdByBannerId, emptySet());
    }

    /**
     * Создает связки ID баннера с ID пермалинка
     */
    public void linkOrganizationsToBanners(DSLContext context, Map<Long, Long> permalinkIdByBannerId,
                                           Set<Long> bannerIdsWithVCardPreferFlag) {
        linkOrganizationsToBanners(context, permalinkIdByBannerId, bannerIdsWithVCardPreferFlag, MANUAL);
    }

    /**
     * Создает связки ID баннера с ID пермалинка
     */
    void linkOrganizationsToBanners(DSLContext context, Map<Long, Long> permalinkIdByBannerId,
                                    PermalinkAssignType assignType) {
        linkOrganizationsToBanners(context, permalinkIdByBannerId, emptySet(), assignType);
    }

    /**
     * Создает связки ID баннера с ID пермалинка
     */
    void linkOrganizationsToBanners(DSLContext context, Map<Long, Long> permalinkIdByBannerId,
                                    Set<Long> bannerIdsWithVCardPreferFlag,
                                    PermalinkAssignType assignType) {
        if (permalinkIdByBannerId.isEmpty()) {
            return;
        }

        var step = context.insertInto(BANNER_PERMALINKS)
                .columns(BANNER_PERMALINKS.BID, BANNER_PERMALINKS.PERMALINK, BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE,
                        BANNER_PERMALINKS.PREFER_VCARD_OVER_PERMALINK);

        permalinkIdByBannerId.forEach((bannerId, permalinkId) ->
                step.values(bannerId, permalinkId, PermalinkAssignType.toSource(assignType),
                        bannerIdsWithVCardPreferFlag.contains(bannerId) ? 1L : 0L));

        if (assignType == MANUAL) {
            step.onDuplicateKeyUpdate()
                    .set(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE, manual)
                    // Для типа manual поддерживаем значение поля is_change_to_manual_rejected равным 0.
                    // Явно проставляем полю 0, т.к. в нём может храниться 1 в случае ручного добавления
                    // той организации, которая ранее была добавлена автоматически и отклонена.
                    .set(BANNER_PERMALINKS.IS_CHANGE_TO_MANUAL_REJECTED, 0L)
                    .execute();
        } else {
            step
                    .onDuplicateKeyIgnore()
                    .execute();
        }

        var now = LocalDateTime.now();
        context.update(BANNERS)
                .set(BANNERS.LAST_CHANGE, now)
                .where(BANNERS.BID.in(permalinkIdByBannerId.keySet()))
                .execute();
    }

    /**
     * Удаляет созданные вручную связки с пермалинками для указанных групп
     */
    public void unlinkOrganizationsFromBanners(int shard, Collection<Long> adGroupIds) {
        if (adGroupIds.isEmpty()) {
            return;
        }

        unlinkOrganizationsFromBanners(dslContextProvider.ppc(shard), adGroupIds);
    }

    /**
     * Удаляет созданные вручную связки с пермалинками для указанных баннеров
     */
    public void unlinkOrganizationsFromBanners(DSLContext context, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }

        var manualPermalinkBannerIds = getManualOrganizationsBannerIds(context, bannerIds);

        context
                .deleteFrom(BANNER_PERMALINKS)
                .where(BANNER_PERMALINKS.BID.in(manualPermalinkBannerIds))
                .execute();

        context
                .deleteFrom(BANNER_PHONES)
                .where(BANNER_PHONES.BID.in(manualPermalinkBannerIds))
                .execute();

        LocalDateTime now = LocalDateTime.now();
        context.update(BANNERS)
                .set(BANNERS.LAST_CHANGE, now)
                .where(BANNERS.BID.in(manualPermalinkBannerIds))
                .execute();
    }

    public List<Long> getManualOrganizationsBannerIds(DSLContext context, Collection<Long> bannerIds) {
        return context.select(BANNER_PERMALINKS.BID)
                .from(BANNER_PERMALINKS)
                .where(BANNER_PERMALINKS.BID.in(bannerIds))
                .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(manual))
                .fetch(BANNER_PERMALINKS.BID);
    }

    public Map<Long, List<Long>> getBannerPhoneIdsByPermalink(int shard, List<Long> permalinks) {
        return dslContextProvider.ppc(shard)
                .select(BANNER_PERMALINKS.PERMALINK, BANNER_PHONES.CLIENT_PHONE_ID)
                .from(BANNER_PERMALINKS
                        .join(BANNER_PHONES).on(BANNER_PERMALINKS.BID.eq(BANNER_PHONES.BID)
                                .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(manual))))
                .where(BANNER_PERMALINKS.PERMALINK.in(permalinks))
                .fetchGroups(BANNER_PERMALINKS.PERMALINK, BANNER_PHONES.CLIENT_PHONE_ID);
    }

    public Map<Long, List<Long>> getCampaignPhoneIdsByPermalink(int shard, List<Long> permalinks) {
        return dslContextProvider.ppc(shard)
                .select(CAMPAIGN_PERMALINKS.PERMALINK_ID, CAMPAIGN_PHONES.CLIENT_PHONE_ID)
                .from(CAMPAIGN_PERMALINKS
                        .join(CAMPAIGN_PHONES).on(CAMPAIGN_PERMALINKS.CID.eq(CAMPAIGN_PHONES.CID)))
                .where(CAMPAIGN_PERMALINKS.PERMALINK_ID.in(permalinks))
                .fetchGroups(CAMPAIGN_PERMALINKS.PERMALINK_ID, CAMPAIGN_PHONES.CLIENT_PHONE_ID);
    }

    /**
     * Обновляет флаг preferVCardOverPermalink у связи баннер - организация
     */
    public void updateOrganizationsToBannerBinds(DSLContext context,
                                                 Map<Long, Boolean> preferVCardOverPermalinkFlagByBannerIds) {
        if (preferVCardOverPermalinkFlagByBannerIds.isEmpty()) {
            return;
        }

        Set<Long> bannerIdsWithFlag = new HashSet<>();
        Set<Long> bannerIdsWithoutFlag = new HashSet<>();

        preferVCardOverPermalinkFlagByBannerIds.forEach((bannerId, flag) -> {
            if (flag) {
                bannerIdsWithFlag.add(bannerId);
            } else {
                bannerIdsWithoutFlag.add(bannerId);
            }
        });

        if (!bannerIdsWithFlag.isEmpty()) {
            context
                    .update(BANNER_PERMALINKS)
                    .set(BANNER_PERMALINKS.PREFER_VCARD_OVER_PERMALINK, 1L)
                    .where(BANNER_PERMALINKS.BID.in(bannerIdsWithFlag))
                    .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(manual))
                    .execute();
        }

        if (!bannerIdsWithoutFlag.isEmpty()) {
            context
                    .update(BANNER_PERMALINKS)
                    .set(BANNER_PERMALINKS.PREFER_VCARD_OVER_PERMALINK, 0L)
                    .where(BANNER_PERMALINKS.BID.in(bannerIdsWithoutFlag))
                    .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(manual))
                    .execute();
        }
    }

    /**
     * Добавление записей в табличку ppc.banner_permalinks.
     *
     * @param shard   шард
     * @param records записи для добавления
     * @throws IllegalArgumentException Если у записи есть телефон (client_phone_id)
     */
    public int addRecords(int shard, Collection<BannerPermalinksRecord> records) {
        if (records.isEmpty()) {
            return 0;
        }

        var query = dslContextProvider.ppc(shard).dsl().insertQuery(BANNER_PERMALINKS);

        for (BannerPermalinksRecord record : records) {
            query.newRecord();
            query.setRecord(record);
        }

        query.onDuplicateKeyIgnore(true);
        return query.execute();
    }

    /**
     * Удаляет записи из ppc.banner_permalinks по bid, permalink, chain_id, permalink_assign_type
     */
    public int deleteRecords(int shard, Collection<BannerPermalinksRecord> records) {
        if (records.isEmpty()) {
            return 0;
        }

        Condition condition = DSL.falseCondition();
        for (BannerPermalinksRecord record : records) {
            condition = condition.or(BANNER_PERMALINKS.BID.eq(record.getBid())
                    .and(BANNER_PERMALINKS.PERMALINK.eq(record.getPermalink()))
                    .and(BANNER_PERMALINKS.CHAIN_ID.eq(record.getChainId()))
                    .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(record.getPermalinkAssignType())));
        }

        return dslContextProvider.ppc(shard)
                .deleteFrom(BANNER_PERMALINKS)
                .where(condition)
                .execute();
    }

    /**
     * По списку баннеров получает привязанные к ним пермалинки и способы их привязки.
     */
    public Map<Long, List<BannerPermalink>> getBannerPermalinkByBannerIds(
            int shard, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return emptyMap();
        }

        return dslContextProvider.ppc(shard).select(permalinkAssignFieldsToRead)
                .from(BANNER_PERMALINKS)
                .where(BANNER_PERMALINKS.BID.in(bannerIds))
                .fetchGroups(BANNER_PERMALINKS.BID, permalinkAssignTypeMapper::fromDb);
    }

    /**
     * Обновляет статус опубликованности для организаций.
     */
    public void updateOrganizationsStatusPublishByPermalinkIds(int shard,
                                                               Collection<Long> permalinkIds,
                                                               OrganizationStatusPublish statusPublish) {
        updateOrganizationsStatusPublishByPermalinkIds(dslContextProvider.ppc(shard), permalinkIds, statusPublish);
    }

    /**
     * Обновляет статус опубликованности для организаций.
     */
    public void updateOrganizationsStatusPublishByPermalinkIds(DSLContext dslContext,
                                                               Collection<Long> permalinkIds,
                                                               OrganizationStatusPublish statusPublish) {
        if (permalinkIds.isEmpty()) {
            return;
        }

        dslContext
                .update(ORGANIZATIONS)
                .set(ORGANIZATIONS.STATUS_PUBLISH, OrganizationStatusPublish.toSource(statusPublish))
                .where(ORGANIZATIONS.PERMALINK_ID.in(permalinkIds))
                .execute();
    }

    public int deleteOrganizations(
            DSLContext context,
            List<Pair<ClientId, Long>> items) {
        if (items.isEmpty()) {
            return 0;
        }

        Condition condition = DSL.falseCondition();
        for (Pair<ClientId, Long> item : items) {
            condition = condition.or(ORGANIZATIONS.CLIENT_ID.eq(item.getLeft().asLong())
                    .and(ORGANIZATIONS.PERMALINK_ID.eq(item.getRight())));
        }

        return context
                .deleteFrom(ORGANIZATIONS)
                .where(condition)
                .execute();
    }

    private ResultQuery<Record1<Long>> getLinkedBannerIdsByClientIdPermalinkQuery(
            int shard,
            Collection<Pair<ClientId, Long>> clientIdPermalinkIds) {
        return dslContextProvider.ppc(shard)
                .select(BANNER_PERMALINKS.BID)
                .from(BANNER_PERMALINKS)
                .join(ORGANIZATIONS)
                .on(BANNER_PERMALINKS.PERMALINK.eq(ORGANIZATIONS.PERMALINK_ID)
                        .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(manual)))
                .where(row(ORGANIZATIONS.CLIENT_ID, BANNER_PERMALINKS.PERMALINK).in(
                        mapList(clientIdPermalinkIds,
                                clientIdPermalinkId -> row(clientIdPermalinkId.getLeft().asLong(),
                                        clientIdPermalinkId.getRight()))));

    }

    /**
     * Возвращает id баннеров, принадлежащих определенным клиентам и привязанных к определенным организациям.
     */
    public List<Long> getLinkedBannerIdsByClientIdPermalinkIds(int shard,
                                                               Collection<Pair<ClientId, Long>> clientIdPermalinkIds) {
        if (clientIdPermalinkIds.isEmpty()) {
            return emptyList();
        }
        return getLinkedBannerIdsByClientIdPermalinkQuery(shard, clientIdPermalinkIds).fetch(BANNER_PERMALINKS.BID);
    }

    /**
     * Возвращает id баннеров, принадлежащих определенным клиентам и привязанных к определенным организациям.
     */
    public Stream<Long> getLinkedBannerIdsByClientIdPermalinkIdsStream(
            int shard,
            Collection<Pair<ClientId, Long>> clientIdPermalinkIds, int chunkSize) {
        if (clientIdPermalinkIds.isEmpty()) {
            return Stream.empty();
        }

        return getLinkedBannerIdsByClientIdPermalinkQuery(shard, clientIdPermalinkIds)
                .fetchSize(chunkSize)
                .fetchStream()
                .map(r -> r.getValue(BANNER_PERMALINKS.BID));
    }

    public void rejectAutoOrganizations(int shard, Map<Long, Long> permalinkIdsByBannerIds) {
        if (permalinkIdsByBannerIds.isEmpty()) {
            return;
        }

        List<Row2<Long, Long>> rowsToUpdate = EntryStream.of(permalinkIdsByBannerIds).mapKeyValue(DSL::row).toList();
        dslContextProvider.ppc(shard)
                .update(BANNER_PERMALINKS)
                .set(BANNER_PERMALINKS.IS_CHANGE_TO_MANUAL_REJECTED, 1L)
                .where(row(BANNER_PERMALINKS.BID, BANNER_PERMALINKS.PERMALINK).in(rowsToUpdate))
                .and(row(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE).equal(auto))
                .returning(BANNER_PERMALINKS.BID)
                .execute();
    }

    /**
     * Получить limit пермалинков, начиная с lastPermalinkId и lastBannerId для не-архивных баннеров.
     */
    public Map<Long, Long> getPermalinkIdByBannerId(int shard, int limit, Long lastPermalinkId, Long lastBannerId) {
        Condition isAfterLastRead = BANNER_PERMALINKS.PERMALINK.greaterThan(lastPermalinkId)
                .or(BANNER_PERMALINKS.PERMALINK.eq(lastPermalinkId)
                        .and(BANNER_PERMALINKS.BID.greaterThan(lastBannerId)));
        return dslContextProvider.ppc(shard)
                .select(BANNER_PERMALINKS.PERMALINK, BANNER_PERMALINKS.BID, BANNERS.STATUS_ARCH)
                .from(BANNER_PERMALINKS)
                .join(BANNERS)
                .on(BANNERS.BID.eq(BANNER_PERMALINKS.BID))
                .where(isAfterLastRead
                        .and(BANNERS.STATUS_ARCH.eq(BannersStatusarch.No)))
                .and(BANNER_PERMALINKS.PERMALINK_ASSIGN_TYPE.eq(manual))
                .orderBy(BANNER_PERMALINKS.PERMALINK, BANNER_PERMALINKS.BID)
                .limit(limit)
                .fetchMap(BANNER_PERMALINKS.BID, BANNER_PERMALINKS.PERMALINK);
    }
}
