package ru.yandex.direct.core.entity.organizations.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.altay.model.language.LanguageOuterClass;
import ru.yandex.altay.model.language.LanguageOuterClass.Language;
import ru.yandex.direct.core.entity.organization.model.Organization;
import ru.yandex.direct.core.entity.organization.model.OrganizationCreateRequest;
import ru.yandex.direct.core.entity.organizations.OrganizationsClientTranslatableException;
import ru.yandex.direct.core.entity.organizations.repository.OrganizationRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.geosearch.GeosearchClient;
import ru.yandex.direct.organizations.swagger.OrganizationApiInfo;
import ru.yandex.direct.organizations.swagger.OrganizationInfo;
import ru.yandex.direct.organizations.swagger.OrganizationsClient;
import ru.yandex.direct.organizations.swagger.OrganizationsClientException;
import ru.yandex.direct.organizations.swagger.model.Address;
import ru.yandex.direct.organizations.swagger.model.CreateOnlineOrgRequest;
import ru.yandex.direct.organizations.swagger.model.MetrikaData;
import ru.yandex.direct.organizations.swagger.model.PubApiCompaniesData;
import ru.yandex.direct.organizations.swagger.model.PubApiCompany;
import ru.yandex.direct.organizations.swagger.model.PubApiCompanyData;
import ru.yandex.direct.organizations.swagger.model.PubApiExtraCompanyField;
import ru.yandex.direct.organizations.swagger.model.SpravResponse;
import ru.yandex.direct.organizations.swagger.model.UpdateOrganizationRequest;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.constraint.CollectionConstraints;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.constraint.PhoneNumberConstraints;
import ru.yandex.direct.validation.constraint.StringConstraints;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.security.tvm.TvmUtils.retrieveUserTicket;
import static ru.yandex.direct.organizations.swagger.OrganizationInfoConverters.convertMultipleOrganizationsInfo;
import static ru.yandex.direct.organizations.swagger.OrganizationInfoConverters.toCoreStatusPublish;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
@ParametersAreNonnullByDefault
public class OrganizationService {
    private static final Logger logger = LoggerFactory.getLogger(OrganizationService.class);
    private static final List<String> ORGANIZATION_FIELDS_NO_METRIKA_DATA = List.of(
            "names",
            "address",
            "phones",
            "business_profile_link",
            "lk_link",
            "publishing_status",
            "urls",
            "is_online",
            "rubrics",
            "rubric_defs");

    private static final List<String> ORGANIZATION_FIELDS =
            StreamEx.of(ORGANIZATION_FIELDS_NO_METRIKA_DATA).append("metrika_data").toImmutableList();
    public static final String DEFAULT_LANGUAGE = Language.RU.name().toLowerCase();

    private final OrganizationRepository organizationRepository;
    private final OrganizationsClient organizationsClient;
    private final GeosearchClient geosearchClient;
    private final RbacService rbacService;
    private final ShardHelper shardHelper;

    @Autowired
    public OrganizationService(
            OrganizationRepository organizationRepository,
            OrganizationsClient organizationsClient,
            GeosearchClient geosearchClient,
            RbacService rbacService,
            ShardHelper shardHelper) {
        this.organizationRepository = organizationRepository;
        this.organizationsClient = organizationsClient;
        this.geosearchClient = geosearchClient;
        this.rbacService = rbacService;
        this.shardHelper = shardHelper;
    }

    public Result<Long> createOnlineOrganization(
            long creatorUid,
            OrganizationCreateRequest request
    ) {
        ValidationResult<OrganizationCreateRequest, Defect> vr = validateCreateRequest(request);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }
        String tvmUserTicket = retrieveUserTicket();
        SpravResponse<PubApiCompany> response =
                organizationsClient.createOnlineOrg(tvmUserTicket, toCreateOnlineOrgRequest(creatorUid, request));
        return response.fold(
                (success) -> Result.successful(success.getId()),
                (error) -> {
                    logger.warn("Validation error on creating org: {}", JsonUtils.toJson(error));
                    return Result.broken(ValidationResult.failed(request, CommonDefects.invalidValue()));
                });
    }

    private CreateOnlineOrgRequest toCreateOnlineOrgRequest(long creatorUid, OrganizationCreateRequest request) {
        return new CreateOnlineOrgRequest(
                creatorUid,
                request.getName(),
                request.getPhoneFormatted(),
                request.getUrl(),
                request.getServiceAreaGeoIds(),
                request.getRubrics()
        );
    }

    private ValidationResult<OrganizationCreateRequest, Defect>
    validateCreateRequest(OrganizationCreateRequest request) {
        ModelItemValidationBuilder<OrganizationCreateRequest> ivb = ModelItemValidationBuilder.of(request);
        ivb.item(OrganizationCreateRequest.NAME)
                .check(CommonConstraints.notNull())
                .check(StringConstraints.notBlank())
                .check(StringConstraints.minStringLength(3))
                .check(StringConstraints.maxStringLength(255));
        ivb.item(OrganizationCreateRequest.PHONE_FORMATTED)
                .check(CommonConstraints.notNull())
                .check(PhoneNumberConstraints.validPhoneNumber());
        ivb.item(OrganizationCreateRequest.URL)
                .check(StringConstraints.notBlank())
                .check(StringConstraints.validUrl());
        ivb.item(OrganizationCreateRequest.SERVICE_AREA_GEO_IDS)
                .check(CommonConstraints.notNull())
                .check(CollectionConstraints.notEmptyCollection());
        ivb.item(OrganizationCreateRequest.RUBRICS)
                .check(CommonConstraints.notNull())
                .check(CollectionConstraints.notEmptyCollection());
        return ivb.getResult();
    }

    public Result<Long> updateOrganization(
            long permalink,
            UpdateOrganizationRequest request
    ) {
        String tvmUserTicket = retrieveUserTicket();
        organizationsClient.updateOrganization(tvmUserTicket, permalink, request);
        return Result.successful(permalink);
    }

    /**
     * По списку пермалинков возвращает организации клиента со статусом организации
     *
     * @param permalinkIds пермалинки организаций
     * @param clientId     id клиента
     * @return мап (permalinkId -> {@link Organization}), содержит только организации, на которые клиент имеет права.
     * В случае недоступности справочника кидает translatable-исключение
     */
    public Map<Long, Organization> getClientOrganizations(Collection<Long> permalinkIds,
                                                          ClientId clientId)
            throws OrganizationsClientTranslatableException {
        Set<Long> permalinkIdsToQuery = StreamEx.of(permalinkIds)
                .nonNull()
                .toSet();

        if (permalinkIdsToQuery.isEmpty()) {
            return emptyMap();
        }

        List<String> fields = List.of(PubApiExtraCompanyField.PUBLISHING_STATUS.getValue());
        String tvmUserTicket = retrieveUserTicket();
        List<PubApiCompany> companies = queryClientThrow(client -> client.getMultipleOrganizationsInfo(
                permalinkIdsToQuery, fields, DEFAULT_LANGUAGE, tvmUserTicket).getCompanies());
        return StreamEx.of(companies)
                .mapToEntry(
                        PubApiCompany::getId,
                        company -> new Organization()
                                .withPermalinkId(company.getId())
                                .withClientId(clientId)
                                .withStatusPublish(toCoreStatusPublish(company.getPublishingStatus())))
                .toMap();
    }

    /**
     * @return Из списка организаций получает опубликованные.
     * В случае недоступности справочника возвращает пустую коллекцию.
     */
    public Set<Long> getAvailablePermalinkIdsNoThrow(Collection<Long> permalinkIds) {
        return getAvailablePermalinkIds(permalinkIds, false);
    }

    /**
     * @return Из списка организаций получает опубликованные.
     * В случае недоступности справочника кидает translatable-исключение.
     */
    public Set<Long> getAvailablePermalinkIds(Collection<Long> permalinkIds)
            throws OrganizationsClientTranslatableException {
        return getAvailablePermalinkIds(permalinkIds, true);
    }

    /**
     * Из списка организаций получает опубликованные.
     */
    private Set<Long> getAvailablePermalinkIds(Collection<Long> permalinkIds, boolean throwException) {
        var permalinkIdsFiltered = StreamEx.of(permalinkIds).nonNull().toSet();
        if (permalinkIdsFiltered.isEmpty()) {
            return emptySet();
        }
        String tvmUserTicket = retrieveUserTicket();

        var data = queryClient(client -> client.getMultiplePublishedOrganizationsInfo(permalinkIdsFiltered,
                emptyList(), DEFAULT_LANGUAGE, tvmUserTicket), throwException);

        if (data == null) {
            return emptySet();
        }

        return StreamEx.of(data.getCompanies())
                .nonNull()
                .map(PubApiCompany::getId)
                .toSet();
    }

    /**
     * В случае недоступности справочника кидает translatable-исключение.
     */
    @Nonnull
    public List<OrganizationApiInfo> getApiClientOrganizations(ClientId clientId,
                                                               Long chiefUid,
                                                               Language language,
                                                               @Nullable List<Long> permalinkIds)
            throws OrganizationsClientTranslatableException {
        return getApiClientOrganizations(clientId, chiefUid, language, permalinkIds, ORGANIZATION_FIELDS);
    }

    /**
     * В случае недоступности справочника кидает translatable-исключение.
     */
    @Nonnull
    public List<OrganizationApiInfo> getApiClientOrganizationsWithoutMetrikaData(ClientId clientId,
                                                                                 Long chiefUid,
                                                                                 Language language,
                                                                                 @Nullable List<Long> permalinkIds) {
        return getApiClientOrganizations(clientId, chiefUid, language, permalinkIds,
                ORGANIZATION_FIELDS_NO_METRIKA_DATA);
    }

    /**
     * В случае недоступности справочника кидает translatable-исключение.
     */
    @Nonnull
    public List<OrganizationApiInfo> getApiClientOrganizations(ClientId clientId,
                                                               Long chiefUid,
                                                               Language language,
                                                               @Nullable List<Long> permalinkIds,
                                                               Set<PubApiExtraCompanyField> requestedFields) {
        return getApiClientOrganizations(clientId, chiefUid, language, permalinkIds,
                mapSet(requestedFields, PubApiExtraCompanyField::getValue));
    }

    /**
     * В случае недоступности справочника кидает translatable-исключение.
     */
    @Nonnull
    public List<OrganizationApiInfo> getApiClientOrganizations(ClientId clientId,
                                                               Long chiefUid,
                                                               Language language,
                                                               @Nullable List<Long> permalinkIds,
                                                               Collection<String> requestedFields)
            throws OrganizationsClientTranslatableException {
        int shard = shardHelper.getShardByClientId(clientId);
        Set<Long> permalinksInDb = StreamEx
                .of(organizationRepository.getAllClientOrganizations(shard, clientId))
                .map(Organization::getPermalinkId)
                .toSet();

        if (permalinkIds == null) {
            return getAllClientOrganizations(chiefUid, permalinksInDb, language, requestedFields);
        } else {
            return getClientOrganizationsByIds(permalinkIds, language, requestedFields);
        }
    }

    /**
     * Получает для указанных организаций их счетчики метрики.
     * В случае недоступности справочника кидает translatable-исключение.
     */
    public Map<Long, Long> getMetrikaCountersByOrganizationsIds(Language language, Collection<Long> permalinkIds)
            throws OrganizationsClientTranslatableException {
        if (permalinkIds.isEmpty()) {
            return emptyMap();
        }
        String tvmUserTicket = retrieveUserTicket();
        List<MetrikaData> metrikaDataList =
                queryClientThrow(client -> client.getOrganizationsCountersData(Set.copyOf(permalinkIds),
                        language.name().toLowerCase(), tvmUserTicket));
        return StreamEx.of(metrikaDataList)
                .mapToEntry(MetrikaData::getPermalink, MetrikaData::getCounter)
                .removeValues(StringUtils::isEmpty)
                .mapValues(Long::valueOf)
                .toMap();
    }

    /**
     * В случае недоступности справочника кидает translatable-исключение.
     */
    @Nonnull
    public Map<Long, Long> getPermalinkIdsByBannerIds(ClientId clientId, Collection<Long> bannerIds)
            throws OrganizationsClientTranslatableException {
        if (bannerIds.isEmpty()) {
            return emptyMap();
        }
        int shard = shardHelper.getShardByClientId(clientId);
        return organizationRepository.getPermalinkIdsByBannerIds(shard, bannerIds);
    }

    /**
     * В случае недоступности справочника кидает translatable-исключение.
     */
    @Nonnull
    public Map<Long, List<Long>> getPermalinkIdsByCampaignId(ClientId clientId, Collection<Long> campaignIds)
            throws OrganizationsClientTranslatableException {
        if (campaignIds.isEmpty()) {
            return emptyMap();
        }
        int shard = shardHelper.getShardByClientId(clientId);
        return organizationRepository.getPermalinkIdsByCampaignId(shard, campaignIds);
    }

    /**
     * Возвращает организации по их ID.
     * В случае недоступности справочника кидает translatable-исключение.
     * {@link OrganizationApiInfo}
     */
    public List<OrganizationApiInfo> getClientOrganizationsByIds(Collection<Long> permalinkIds,
                                                                 Language language)
            throws OrganizationsClientTranslatableException {
        return getClientOrganizationsByIds(permalinkIds, language, ORGANIZATION_FIELDS);
    }

    /**
     * Возвращает организации по их ID.
     * В случае недоступности справочника кидает translatable-исключение.
     * {@link OrganizationApiInfo}
     */
    public List<OrganizationApiInfo> getClientOrganizationsByIds(Collection<Long> permalinkIds,
                                                                 Language language,
                                                                 Collection<String> requestedFields)
            throws OrganizationsClientTranslatableException {
        Set<Long> permalinkIdsFiltered = filterAndMapToSet(permalinkIds, Objects::nonNull, identity());
        if (permalinkIdsFiltered.isEmpty()) {
            return emptyList();
        }

        String lang = language.name().toLowerCase();
        String tvmUserTicket = retrieveUserTicket();
        PubApiCompaniesData companies = queryClientThrow(client -> client.getMultipleOrganizationsInfo(
                permalinkIdsFiltered, requestedFields, lang, tvmUserTicket));
        if (companies == null) {
            return emptyList();
        }
        List<OrganizationApiInfo> orgs = convertMultipleOrganizationsInfo(OrganizationApiInfo::new, companies, lang);
        return orgs;
    }

    /**
     * Возвращает список всех организаций, пренадлежащих либо пренадлежавших когда-либо клиенту.
     * <p/>
     * Возвращает склеенные пермалинки для всех организаций, на которые есть права.
     * <p/>
     * Для компаний, на которые нет прав, проставляет {@code false} в поле {@code accessible} в
     * {@link OrganizationApiInfo}.
     * <p/>
     * В случае недоступности справочника кидает translatable-исключение.
     *
     * @param chiefUid       UID главного представителя клиента
     * @param permalinksInDb Пермалинки клиента, содержащиеся в табличке ppc.organizations
     * @param language       Язык запроса
     */
    private List<OrganizationApiInfo> getAllClientOrganizations(Long chiefUid,
                                                                Set<Long> permalinksInDb,
                                                                Language language,
                                                                Collection<String> requestedFields)
            throws OrganizationsClientTranslatableException {
        List<OrganizationApiInfo> result = new ArrayList<>();
        Set<Long> permalinksLeft = new HashSet<>(permalinksInDb);

        // Организации, принадлежащие пользователю
        String lang = language.name().toLowerCase();
        String tvmUserTicket = retrieveUserTicket();
        PubApiCompaniesData ownedOrganizationsRaw = queryClientThrow(client -> client.getMultipleOwnOrganizationsInfo(
                List.of(chiefUid), requestedFields, lang, tvmUserTicket));
        List<OrganizationApiInfo> ownedOrganizations = convertMultipleOrganizationsInfo(OrganizationApiInfo::new,
                ownedOrganizationsRaw, lang);
        result.addAll(ownedOrganizations);
        Set<Long> ownedPermalinkIds = listToSet(result, Organization::getPermalinkId);
        permalinksLeft.removeIf(ownedPermalinkIds::contains);

        // Склеенные организации
        if (!permalinksLeft.isEmpty()) {
            Map<Long, Set<Long>> mergedPermalinksByHeadPermalink = geosearchClient.getMergedPermalinks(permalinksLeft);
            for (OrganizationApiInfo info : result) {
                Set<Long> merged = mergedPermalinksByHeadPermalink.get(info.getPermalinkId());
                if (merged != null) {
                    info.withMergedPermalinks(merged);
                    permalinksLeft.removeIf(merged::contains);
                }
            }
        }

        // Организации, которые не принадлежат клиенту
        if (!permalinksLeft.isEmpty()) {
            PubApiCompaniesData orgs = queryClientThrow(client -> client.getMultipleOrganizationsInfo(
                    permalinksLeft, requestedFields, lang, tvmUserTicket));
            List<OrganizationApiInfo> notOwnedOrgs =
                    convertMultipleOrganizationsInfo(OrganizationApiInfo::new, orgs, lang);
            result.addAll(notOwnedOrgs);
        }

        return result;
    }

    /**
     * В случае недоступности справочника кидает translatable-исключение.
     */
    public List<OrganizationInfo> getOrganizationsInfo(List<Long> permalinkIds,
                                                       LanguageOuterClass.Language language,
                                                       Set<PubApiExtraCompanyField> requestedFields)
            throws OrganizationsClientTranslatableException {
        return queryClientThrow(client ->
                client.getOrganizationsInfo(permalinkIds, language, retrieveUserTicket(),
                        mapSet(requestedFields, PubApiExtraCompanyField::getValue))
        );
    }

    /**
     * В случае недоступности справочника кидает translatable-исключение.
     */
    public List<OrganizationInfo> getOrganizationsInfoByText(String text,
                                                             LanguageOuterClass.Language language,
                                                             Set<PubApiExtraCompanyField> requestedFields)
            throws OrganizationsClientTranslatableException {
        return queryClientThrow(client ->
                client.getMultipleOrganizationsInfoByText(text, language, retrieveUserTicket(), requestedFields)
        );
    }

    /**
     * Возвращает сет доступных permalinkId для редактирования
     */
    public Set<Long> operatorEditableOrganizations(long operatorUid, Collection<Long> permalinkIds) {
        var permalinkIdToUids = queryClient(client -> client.getOrganizationsUidsWithModifyPermission(
                permalinkIds, Set.of(operatorUid), DEFAULT_LANGUAGE, retrieveUserTicket()), false);
        if (permalinkIdToUids == null) {
            return Set.of();
        }

        return EntryStream.of(permalinkIdToUids)
                .filterValues(uidsWithPerm -> uidsWithPerm != null && uidsWithPerm.contains(operatorUid))
                .keys()
                .toSet();
    }


    /**
     * Возвращает мапу permalinkId -> доступна ли организация для хотя бы одному предствителю.
     * Если организация не существует, её id не будет в возвращаемом значении.
     * В случае недоступности справочника возвращает пустую мапу.
     */
    public Map<Long, Boolean> hasAccessNoThrow(ClientId clientId, Collection<Long> permalinkIds) {
        return hasAccess(clientId, permalinkIds, false);
    }

    /**
     * Возвращает мапу permalinkId -> доступна ли организация для хотя бы одному предствителю.
     * Если организация не существует, её id не будет в возвращаемом значении.
     * В случае недоступности справочника кидает translatable-исключение.
     */
    public Map<Long, Boolean> hasAccess(ClientId clientId, Collection<Long> permalinkIds)
            throws OrganizationsClientTranslatableException {
        return hasAccess(clientId, permalinkIds, true);
    }

    /**
     * Возвращает мапу permalinkId -> доступна ли организация для хотя бы одному предствителю.
     * Если организация не существует, её id не будет в возвращаемом значении.
     */
    private Map<Long, Boolean> hasAccess(ClientId clientId, Collection<Long> permalinkIds, boolean throwException) {

        Collection<Long> allReps = rbacService.getClientRepresentativesUids(clientId);

        var permalinkIdToUids = queryClient(client -> client.getOrganizationsUidsWithModifyPermission(
                permalinkIds, allReps, DEFAULT_LANGUAGE, retrieveUserTicket()), throwException);
        if (permalinkIdToUids == null) {
            return emptyMap();
        }

        return EntryStream.of(permalinkIdToUids)
                .mapValues(uidsWithPerm -> !Collections.disjoint(allReps, uidsWithPerm))
                .toMap();
    }

    /**
     * В случае недоступности справочника кидает translatable-исключение.
     */
    public Map<Long, List<Long>> getBannerPhoneIdsByPermalink(ClientId clientId, List<Long> permalinkIds)
            throws OrganizationsClientTranslatableException {
        int shard = shardHelper.getShardByClientId(clientId);
        Map<Long, List<Long>> phoneIdsByPermalink =
                organizationRepository.getBannerPhoneIdsByPermalink(shard, permalinkIds);
        phoneIdsByPermalink.values().removeIf(Objects::isNull);
        return phoneIdsByPermalink;
    }

    /**
     * В случае недоступности справочника кидает translatable-исключение.
     */
    public Map<Long, List<Long>> getCampaignPhoneIdsByPermalink(ClientId clientId, List<Long> permalinkIds)
            throws OrganizationsClientTranslatableException {
        int shard = shardHelper.getShardByClientId(clientId);
        Map<Long, List<Long>> phoneIdsByPermalink =
                organizationRepository.getCampaignPhoneIdsByPermalink(shard, permalinkIds);
        phoneIdsByPermalink.values().removeIf(Objects::isNull);
        return phoneIdsByPermalink;
    }

    /**
     * Получаем регион организации.
     */
    public Long getRegion(Long permalinkId)
            throws OrganizationsClientTranslatableException {
        if (permalinkId == null) {
            return null;
        }
        var company = queryClientThrow(client -> client.getSingleOrganizationInfo(
                permalinkId, List.of(PubApiExtraCompanyField.ADDRESS.getValue()), DEFAULT_LANGUAGE,
                retrieveUserTicket()));
        return Optional.ofNullable(company)
                .map(PubApiCompanyData::getAddress)
                .map(Address::getGeoId)
                .orElse(null);
    }

    /**
     * Обертка над {@link OrganizationsClient}, которая транслирует исключение клиента в понятное фронту.
     *
     * @throws OrganizationsClientTranslatableException Бросается при недоступности Справочника.
     */
    private <T> T queryClientThrow(Function<OrganizationsClient, T> query)
            throws OrganizationsClientTranslatableException {
        return queryClient(query, true);
    }


    /**
     * Обертка над {@link OrganizationsClient}, которая транслирует исключение клиента в понятное фронту.
     *
     * @param throwException если true, то бросает OrganizationsClientTranslatableException при недоступности
     *                       справочника, иначе возвращает null
     */
    private <T> T queryClient(Function<OrganizationsClient, T> query,
                              boolean throwException)
            throws OrganizationsClientTranslatableException {
        try {
            return query.apply(organizationsClient);
        } catch (OrganizationsClientException e) {
            logger.error("Failed to get organizations info", e);
            if (throwException) {
                throw new OrganizationsClientTranslatableException();
            } else {
                return null;
            }
        }
    }
}
