package ru.yandex.direct.core.entity.outdoor.repository;


import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Record2;
import org.jooq.Result;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.outdoor.model.OutdoorOperator;
import ru.yandex.direct.dbschema.ppcdict.tables.records.OutdoorOperatorsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.Collections.singleton;
import static ru.yandex.direct.dbschema.ppcdict.tables.OutdoorOperators.OUTDOOR_OPERATORS;
import static ru.yandex.direct.dbschema.ppcdict.tables.Placements.PLACEMENTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class OutdoorOperatorRepository {

    private static final JooqMapperWithSupplier<OutdoorOperator> MAPPER = createMapper();

    private final DslContextProvider dslContextProvider;

    @Autowired
    public OutdoorOperatorRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    public OutdoorOperator getByLogin(String login) {
        return getByLogins(singleton(login))
                .get(login);
    }

    public Map<String, OutdoorOperator> getByLogins(Collection<String> logins) {
        return dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(OUTDOOR_OPERATORS)
                .where(OUTDOOR_OPERATORS.LOGIN.in(logins))
                .fetchMap(OUTDOOR_OPERATORS.LOGIN, MAPPER::fromDb);
    }

    /**
     * Добавить или обновить outdoor операторов
     *
     * @param operators список сохраняемых операторов
     */
    public void addOrUpdateOperators(List<OutdoorOperator> operators) {
        InsertHelper<OutdoorOperatorsRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppcdict(), OUTDOOR_OPERATORS);
        insertHelper.addAll(MAPPER, operators);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(OUTDOOR_OPERATORS.NAME, MySQLDSL.values(OUTDOOR_OPERATORS.NAME));
        }
        insertHelper.executeIfRecordsAdded();
    }

    /**
     * Достает всех операторов, отсортированных по логину
     *
     * @param limitOffset смещение и лимит
     */
    public List<OutdoorOperator> getAllOperators(LimitOffset limitOffset) {
        List<OutdoorOperator> operators = dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(OUTDOOR_OPERATORS)
                .orderBy(OUTDOOR_OPERATORS.LOGIN)
                .offset(limitOffset.offset())
                .limit(limitOffset.limit())
                .fetch(MAPPER::fromDb);
        return operators;
    }

    public Map<Long, String> getPageOperators(Collection<Long> pages) {
        Result<Record2<Long, String>> queryRes = dslContextProvider.ppcdict()
                .select(PLACEMENTS.PAGE_ID, DSL.nvl(OUTDOOR_OPERATORS.NAME, PLACEMENTS.OWNER_LOGIN))
                .from(PLACEMENTS)
                .leftJoin(OUTDOOR_OPERATORS).on(PLACEMENTS.OWNER_LOGIN.eq(OUTDOOR_OPERATORS.LOGIN))
                .where(PLACEMENTS.PAGE_ID.in(pages)).fetch();

        Map<Long, String> res = new HashMap<>();

        queryRes.forEach(r -> res.put(r.value1(), r.value2()));

        return res;
    }

    private static JooqMapperWithSupplier<OutdoorOperator> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(OutdoorOperator::new)
                .map(property(OutdoorOperator.LOGIN, OUTDOOR_OPERATORS.LOGIN))
                .map(property(OutdoorOperator.NAME, OUTDOOR_OPERATORS.NAME))
                .build();
    }
}
