package ru.yandex.direct.core.entity.outdoor.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.outdoor.model.PlacementsOutdoorData;
import ru.yandex.direct.dbschema.ppcdict.enums.PlacementsPageType;
import ru.yandex.direct.dbschema.ppcdict.tables.records.PlacementsOutdoorDataRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static ru.yandex.direct.dbschema.ppcdict.tables.Placements.PLACEMENTS;
import static ru.yandex.direct.dbschema.ppcdict.tables.PlacementsOutdoorData.PLACEMENTS_OUTDOOR_DATA;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
@ParametersAreNonnullByDefault
public class PlacementsOutdoorDataRepository {

    private static final JooqMapperWithSupplier<PlacementsOutdoorData> MAPPER = createMapper();
    private final DslContextProvider dslContextProvider;

    @Autowired
    public PlacementsOutdoorDataRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Добавить или обновить outdoor операторов
     *
     * @param placementsOutdoorDataList список сохраняемых объектов placements_outdoor_data
     */
    public void addOrUpdate(List<PlacementsOutdoorData> placementsOutdoorDataList) {
        InsertHelper<PlacementsOutdoorDataRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppcdict(), PLACEMENTS_OUTDOOR_DATA);
        insertHelper.addAll(MAPPER, placementsOutdoorDataList);
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(PLACEMENTS_OUTDOOR_DATA.INTERNAL_NAME, MySQLDSL.values(PLACEMENTS_OUTDOOR_DATA.INTERNAL_NAME));
        }
        insertHelper.executeIfRecordsAdded();
    }

    /**
     * Получить мапу internalName -> [pageIds]
     * Возвращаются только значения:
     *    есть в таблице PLACEMENTS
     *    не удалены (is_deleted = 0)
     *
     * @return internalName -> [pageIds]
     */
    public Map<String, List<Long>> getInternalNameToPageIds() {
        return dslContextProvider.ppcdict()
                .select(PLACEMENTS_OUTDOOR_DATA.INTERNAL_NAME, PLACEMENTS_OUTDOOR_DATA.PAGE_ID)
                .from(PLACEMENTS_OUTDOOR_DATA)
                .join(PLACEMENTS).using(PLACEMENTS.PAGE_ID)
                .where(PLACEMENTS.IS_DELETED.eq(RepositoryUtils.FALSE))
                .fetchGroups(PLACEMENTS_OUTDOOR_DATA.INTERNAL_NAME, PLACEMENTS_OUTDOOR_DATA.PAGE_ID);
    }

    /**
     * Достает все данные из placements_outdoor_data, отсортированных по id
     *
     * @param limitOffset смещение и лимит
     */
    public List<PlacementsOutdoorData> getAll(LimitOffset limitOffset) {
        return dslContextProvider.ppcdict()
                .select(MAPPER.getFieldsToRead())
                .from(PLACEMENTS_OUTDOOR_DATA)
                .orderBy(PLACEMENTS_OUTDOOR_DATA.PAGE_ID)
                .offset(limitOffset.offset())
                .limit(limitOffset.limit())
                .fetch(MAPPER::fromDb);
    }

    public Map<Long, String> getPlacementIdToInternalName(Collection<Long> ids) {
        return dslContextProvider.ppcdict()
                .selectDistinct(PLACEMENTS_OUTDOOR_DATA.PAGE_ID, PLACEMENTS_OUTDOOR_DATA.INTERNAL_NAME)
                .from(PLACEMENTS_OUTDOOR_DATA)
                .where(PLACEMENTS_OUTDOOR_DATA.PAGE_ID.in(ids))
                .fetchMap(PLACEMENTS_OUTDOOR_DATA.PAGE_ID, PLACEMENTS_OUTDOOR_DATA.INTERNAL_NAME);
    }

    public Map<String, List<String>> getLoginToInternalNames(Collection<String> logins) {
        if (logins.isEmpty()) {
            return Collections.emptyMap();
        }
        return dslContextProvider.ppcdict()
                .selectDistinct(PLACEMENTS.OWNER_LOGIN, PLACEMENTS_OUTDOOR_DATA.INTERNAL_NAME)
                .from(PLACEMENTS_OUTDOOR_DATA)
                .join(PLACEMENTS).using(PLACEMENTS.PAGE_ID)
                .where(PLACEMENTS.PAGE_TYPE.eq(PlacementsPageType.outdoor).and(PLACEMENTS.OWNER_LOGIN.in(logins)))
                .fetchGroups(PLACEMENTS.OWNER_LOGIN, PLACEMENTS_OUTDOOR_DATA.INTERNAL_NAME);
    }

    /**
     * Удалить дополнительную информацию о площадках outdoor
     */
    public void delete(List<Long> pageIds) {
        dslContextProvider.ppcdict()
                .deleteFrom(PLACEMENTS_OUTDOOR_DATA)
                .where(PLACEMENTS_OUTDOOR_DATA.PAGE_ID.in(pageIds))
                .execute();
    }

    private static JooqMapperWithSupplier<PlacementsOutdoorData> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(PlacementsOutdoorData::new)
                .map(property(PlacementsOutdoorData.PAGE_ID, PLACEMENTS_OUTDOOR_DATA.PAGE_ID))
                .map(property(PlacementsOutdoorData.INTERNAL_NAME, PLACEMENTS_OUTDOOR_DATA.INTERNAL_NAME))
                .build();
    }
}
