package ru.yandex.direct.core.entity.page.repository;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterables;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.pages.model.Page;
import ru.yandex.direct.core.entity.pages.model.PageOption;
import ru.yandex.direct.dbschema.ppcdict.tables.records.PagesRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.convertibleEnumSet;
import static ru.yandex.direct.dbschema.ppcdict.tables.Pages.PAGES;
import static ru.yandex.direct.dbutil.SqlUtils.setField;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

/**
 * Репозиторий для работы с информацией о площадках {@link Page}
 */
@Repository
@ParametersAreNonnullByDefault
public class PageRepository {

    private static final int UPDATE_CHUNK_SIZE = 2000;

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<Page> pageMapper;

    @Autowired
    public PageRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.pageMapper = createPageMapper();
    }

    /**
     * Возвращает набор Page из таблицы PAGES по comapId {@param pagesId}
     */
    public Set<Page> getPagesByIds(Collection<Long> pagesId) {
        if (pagesId.isEmpty()) {
            return Collections.emptySet();
        }
        return dslContextProvider.ppcdict()
                .select(pageMapper.getFieldsToRead())
                .from(PAGES)
                .where(PAGES.PAGE_ID.in(pagesId))
                .fetchSet(pageMapper::fromDb);
    }

    /**
     * Возвращает список всех площадок для внутренней рекламы
     */
    public List<Page> getAllInternalAdPages() {
        return dslContextProvider.ppcdict()
                .select(pageMapper.getFieldsToRead())
                .from(PAGES)
                .where(setField(PAGES.OPTIONS).contains(PageOption.INTERNAL_AD.getTypedValue()))
                .fetch(pageMapper::fromDb);
    }

    /**
     * Добавление новых и изменение существующих записей из {@code pages} списка в PAGES таблицу
     *
     * @implNote чанкует запросы к базе по {@link #UPDATE_CHUNK_SIZE} штук
     */
    public void updatePages(Collection<Page> pages) {
        if (pages.isEmpty()) {
            return;
        }
        Iterable<List<Page>> parts = Iterables.partition(pages, UPDATE_CHUNK_SIZE);
        for (List<Page> part : parts) {
            InsertHelper<PagesRecord> helper =
                    new InsertHelper<>(dslContextProvider.ppcdict(), PAGES);
            helper.addAll(pageMapper, part);
            if (helper.hasAddedRecords()) {
                helper.onDuplicateKeyUpdate()
                        .set(PAGES.ORIG_PAGE_ID, MySQLDSL.values(PAGES.ORIG_PAGE_ID))
                        .set(PAGES.DOMAIN, MySQLDSL.values(PAGES.DOMAIN))
                        .set(PAGES.DESCR, MySQLDSL.values(PAGES.DESCR))
                        .set(PAGES.TARGET_TYPE, MySQLDSL.values(PAGES.TARGET_TYPE))
                        .set(PAGES.NAME, MySQLDSL.values(PAGES.NAME))
                        .set(PAGES.GROUP_NICK, MySQLDSL.values(PAGES.GROUP_NICK))
                        .set(PAGES.SORTING, MySQLDSL.values(PAGES.SORTING))
                        .set(PAGES.OPTIONS, MySQLDSL.values(PAGES.OPTIONS));
            }
            helper.executeIfRecordsAdded();
        }
    }

    private JooqMapperWithSupplier<Page> createPageMapper() {
        return JooqMapperWithSupplierBuilder.builder(Page::new)
                .map(property(Page.ID, PAGES.PAGE_ID))
                .map(property(Page.ORIG_PAGE_ID, PAGES.ORIG_PAGE_ID))
                .map(property(Page.DOMAIN, PAGES.DOMAIN))
                .map(property(Page.DESCRIPTION, PAGES.DESCR))
                .map(convertibleProperty(Page.TARGET_TYPE, PAGES.TARGET_TYPE,
                        RepositoryUtils::shortFromLong, RepositoryUtils::shortToLong))
                .map(property(Page.NAME, PAGES.NAME))
                .map(property(Page.GROUP_NICK, PAGES.GROUP_NICK))
                .map(convertibleProperty(Page.SORTING, PAGES.SORTING,
                        RepositoryUtils::shortFromLong, RepositoryUtils::shortToLong))
                .map(convertibleEnumSet(Page.OPTIONS, PAGES.OPTIONS,
                        PageOption::fromTypedValue, PageOption::getTypedValue))
                .build();
    }
}
