package ru.yandex.direct.core.entity.page.service;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Suppliers;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.page.repository.PageRepository;
import ru.yandex.direct.core.entity.pages.model.Page;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.utils.JsonUtils;

/**
 * Сервис для работы с информацией о площадках {@link Page}.
 */
@Service
@ParametersAreNonnullByDefault
public class PageService {

    private static final String INTERNAL_PAGES_DOMAINS_FILE = "classpath:///pages/internal-pages-domains.json";
    private static final int CACHE_RELOAD_INTERVAL_IN_HOURS = 1;

    private final Set<String> internalPagesDomains;
    private final Supplier<List<Page>> internalAdPages;

    @Autowired
    public PageService(PageRepository pageRepository) {
        this.internalPagesDomains = readInternalPagesDomainsFile();
        this.internalAdPages = Suppliers.memoizeWithExpiration(pageRepository::getAllInternalAdPages,
                CACHE_RELOAD_INTERVAL_IN_HOURS, TimeUnit.HOURS);
    }

    /**
     * Возвращает список всех площадок для внутренней рекламы
     * Используется кэш с обновлением в {@link #CACHE_RELOAD_INTERVAL_IN_HOURS}
     */
    public List<Page> getAllInternalAdPages() {
        return internalAdPages.get();
    }

    /**
     * Возвращает список доменов внутренних площадок из исходного списка доменов.
     *
     * @param domains исходный список доменов
     * @return список доменов внутренних площадок
     */
    public List<String> getInternalPagesDomains(Collection<String> domains) {
        return StreamEx.of(domains)
                .filter(domain -> internalPagesDomains.contains(domain.toLowerCase()))
                .toList();
    }

    public boolean isInternalPageDomain(String domain) {
        if (domain == null) {
            return false;
        }

        return internalPagesDomains.contains(domain);
    }

    private Set<String> readInternalPagesDomainsFile() {
        String json = LiveResourceFactory.get(INTERNAL_PAGES_DOMAINS_FILE).getContent();
        return JsonUtils.fromJson(json, Set.class);
    }
}
