package ru.yandex.direct.core.entity.payment.service;

import java.time.LocalDateTime;
import java.util.List;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.balance.client.model.response.GetCardBindingURLResponse;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.WalletRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes;
import ru.yandex.direct.core.entity.payment.model.AutopayParams;
import ru.yandex.direct.core.entity.payment.model.TurnOnAutopayJobParams;
import ru.yandex.direct.core.entity.payment.repository.AutopaySettingsRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbqueue.repository.DbQueueRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.regions.Region;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.singleton;
import static ru.yandex.direct.regions.Region.GLOBAL_REGION_ID;

@Service
public class AutopayService {

    private static final Logger logger = LoggerFactory.getLogger(AutopayService.class);

    private final WalletRepository walletRepository;
    private final RbacService rbacService;
    private final ClientService clientService;
    private final CampaignRepository campaignRepository;
    private final BalanceService balanceService;
    private final DbQueueRepository dbQueueRepository;
    private final AutopaySettingsRepository autopaySettingsRepository;

    @Autowired
    public AutopayService(WalletRepository walletRepository, RbacService rbacService, ClientService clientService,
                          CampaignRepository campaignRepository, BalanceService balanceService,
                          DbQueueRepository dbQueueRepository, AutopaySettingsRepository autopaySettingsRepository) {
        this.walletRepository = walletRepository;
        this.rbacService = rbacService;
        this.clientService = clientService;
        this.campaignRepository = campaignRepository;
        this.balanceService = balanceService;
        this.dbQueueRepository = dbQueueRepository;
        this.autopaySettingsRepository = autopaySettingsRepository;
    }

    /**
     * Получить ссылку на привязку карты и добавить задание на включение автоплатежа после привязки
     * Возращает ссылку на форму привязки
     */
    public String getBindingUrlAndStartAutopayJob(User user, int shard, Long walletCid, CurrencyCode currencyCode,
                                                  AutopayParams autopayParams, boolean isMobileForm,
                                                  @Nullable String redirectUrl) {
        GetCardBindingURLResponse response = balanceService.getCardBinding(user.getUid(), currencyCode,
                redirectUrl, isMobileForm);

        String cardBindingUrl = response.getBindingUrl();

        LocalDateTime autopayLastChange = walletRepository.getAutopayLastChange(shard, walletCid);

        TurnOnAutopayJobParams jobParams = new TurnOnAutopayJobParams()
                .withPurchaseToken(response.getPurchaseToken())
                .withAutopayParams(autopayParams)
                .withActualTime(autopayLastChange);

        Long jobId = dbQueueRepository.insertJob(shard, DbQueueJobTypes.TURN_ON_AUTOPAY, user.getClientId(),
                user.getUid(), jobParams)
                .getId();
        logger.info("Added turn_on_autopay job with id {}", jobId);

        return cardBindingUrl;

    }

    /**
     * Проверяем, можно ли включить автоплатеж и включаем если да
     */
    public void turnOnAutopay(int shard, Long uid, ClientId clientId, AutopayParams autopayParams) {
        if (canUseAutopay(shard, uid, clientId)) {

            List<String> userCards = balanceService.getUserCardIds(uid);
            checkState(userCards.contains(autopayParams.getCardId()), "card is not owned by user");

            Currency clientCurrency = clientService.getWorkCurrency(clientId);
            Long walletCid = walletRepository.getActualClientWalletId(shard, clientId, clientCurrency.getCode());

            checkNotNull(walletCid);
            checkNotNull(autopayParams.getCardId());
            checkNotNull(autopayParams.getPaymentSum());
            checkNotNull(autopayParams.getRemainingSum());
            checkNotNull(autopayParams.getPaymentType());
            checkNotNull(autopayParams.getPersonId());

            walletRepository.turnOnAutopay(shard, walletCid, uid, autopayParams);
        } else {
            logger.error("client can't use autopay");
        }
    }

    /**
     * Выключаем автоплатеж
     */
    public void removeAutopay(int shard, ClientId clientId) {
        Currency clientCurrency = clientService.getWorkCurrency(clientId);
        Long walletCid = walletRepository.getActualClientWalletId(shard, clientId, clientCurrency.getCode());

        if (walletCid != null) {
            walletRepository.deleteInactiveAutoPays(shard, singleton(walletCid));
        }
    }

    public boolean canUseAutopay(int shard, Long uid, ClientId clientId) {
        Currency clientCurrency = clientService.getWorkCurrency(clientId);

        if (!clientCurrency.getCode().equals(CurrencyCode.RUB)) {
            return false;
        }

        Long clientRegionId = clientService.getCountryRegionIdByClientId(clientId).orElse(GLOBAL_REGION_ID);

        if (clientRegionId != Region.RUSSIA_REGION_ID) {
            return false;
        }

        boolean hasAgency = campaignRepository.clientHasWebEditBaseAgencyCampaigns(shard, clientId);

        if (hasAgency) {
            return false;
        }

        return !rbacService.getUidRole(uid).equals(RbacRole.AGENCY);
    }

    public boolean isLastChangeNotAfter(int shard, ClientId clientId, @Nullable LocalDateTime actualTime) {

        if (actualTime == null) {
            return true;
        }

        Currency clientCurrency = clientService.getWorkCurrency(clientId);
        Long walletCid = walletRepository.getActualClientWalletId(shard, clientId, clientCurrency.getCode());
        LocalDateTime autopayLastChange = walletRepository.getAutopayLastChange(shard, walletCid);

        if (autopayLastChange != null) {
            return !autopayLastChange.isAfter(actualTime);
        } else {
            return true;
        }
    }

    /**
     * Получить число попыток автопополнения
     */
    @Nullable
    public Long getAutopayTriesNum(int shard, Long walletCid) {
        Boolean isAutopayOn = walletRepository.isAutopayEnabled(shard, walletCid);

        if (!isAutopayOn) {
            return null;
        }

        return autopaySettingsRepository.getAutopaySettings(shard, walletCid).getTriesNum();
    }
}
