package ru.yandex.direct.core.entity.payment.service;

import java.math.BigDecimal;
import java.util.List;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.balance.client.model.response.GetCardBindingURLResponse;
import ru.yandex.direct.core.entity.campaign.repository.WalletRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.payment.model.AutopayParams;
import ru.yandex.direct.core.entity.payment.model.AutopaySettingsPaymethodType;
import ru.yandex.direct.core.entity.payment.model.BillingData;
import ru.yandex.direct.core.entity.payment.model.CardInfo;
import ru.yandex.direct.core.entity.promocodes.service.PromocodeHelper;
import ru.yandex.direct.core.entity.promocodes.service.PromocodeValidationContainer;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.service.integration.balance.BalanceService;
import ru.yandex.direct.core.service.integration.balance.model.CreateAndPayRequestResult;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.intapi.client.IntApiClient;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Preconditions.checkState;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isTrue;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Service
public class PaymentService {

    private final WalletRepository walletRepository;
    private final ShardHelper shardHelper;
    private final BalanceService balanceService;
    private final IntApiClient intApiClient;
    private final AutopayService autopayService;
    private final FeatureService featureService;
    private final PromocodeHelper promocodeHelper;
    private final PaymentValidationService paymentValidationService;
    private final ClientService clientService;

    @Autowired
    public PaymentService(ShardHelper shardHelper, BalanceService balanceService, WalletRepository walletRepository,
                          IntApiClient intApiClient, AutopayService autopayService, FeatureService featureService,
                          PromocodeHelper promocodeHelper, PaymentValidationService paymentValidationService,
                          ClientService clientService) {
        this.shardHelper = shardHelper;
        this.balanceService = balanceService;
        this.walletRepository = walletRepository;
        this.intApiClient = intApiClient;
        this.autopayService = autopayService;
        this.featureService = featureService;
        this.promocodeHelper = promocodeHelper;
        this.paymentValidationService = paymentValidationService;
        this.clientService = clientService;
    }

    public static class EnableAutopayIfPossibleAndGetUrlsResult {
        public String paymentUrl;
        public String cardBindingUrl;
        public BigDecimal amount;
        public ValidationResult<?, Defect> validation;

        public boolean hasErrors() {
            return validation != null && validation.hasAnyErrors();
        }
    }
    /**
     * Получаем ссылку на форму для оплаты картой и если в {@param autopayParams} переданы суммы для включения
     * автопополнения и карту не передали, то еще отдаем ссылку на привязку карты.
     * Если карту передали, то на нее включаем автопополнение и ссылку на привязку не отдаем.
     * Кидаем исключение:
     * - Если не смогли получить клиентский кошелек
     * - Для юрика с не единственным юридическим плательщиком
     * - Для невалидной оплаты (валидируем через ручку PrepareAndValidatePayCamp)
     */
    public EnableAutopayIfPossibleAndGetUrlsResult enableAutopayIfPossibleAndGetUrls(
            User subjectUser,
            Long operatorUID,
            CurrencyCode clientCurrency,
            BigDecimal sum,
            boolean isLegalPerson,
            AutopayParams autopayParams,
            String promocode,
            PromocodeValidationContainer promocodeValidationContainer,
            boolean isMobileForm,
            @Nullable String successRedirectUrl) {

        var result = new EnableAutopayIfPossibleAndGetUrlsResult();

        var validator = ItemValidationBuilder.of(result, Defect.class);

        ClientId clientId = subjectUser.getClientId();
        Long uid = subjectUser.getUid();
        int shard = shardHelper.getShardByClientIdStrictly(subjectUser.getClientId());

        Long walletCid = walletRepository.getActualClientWalletId(shard, clientId, clientCurrency);

        validator.item(walletCid, "wallet").check(notNull());
        result.validation = validator.getResult();
        if (result.hasErrors()) {
            return result;
        }

        var perlPaymentCheck = intApiClient.validatePayCamp(uid, walletCid, operatorUID, sum);
        validator.item(perlPaymentCheck, "payment").check(isTrue());
        result.validation = validator.getResult();
        if (result.hasErrors()) {
            return result;
        }

        Long personId = balanceService.getOrCreatePerson(subjectUser, operatorUID, isLegalPerson);
        validator.item(personId, "personId").check(notNull());
        result.validation = validator.getResult();
        if (result.hasErrors()) {
            return result;
        }

        boolean paymentBeforeModerationEnabled = featureService.isEnabledForClientId(
                clientId, FeatureName.PAYMENT_BEFORE_MODERATION);

        boolean denyPromocode  = promocode != null && promocodeHelper.doDenyPromocode(promocodeValidationContainer);
        CreateAndPayRequestResult paymentResult = balanceService.createAndPayRequest(
                uid, clientId, walletCid, clientCurrency, sum, personId, successRedirectUrl,
                paymentBeforeModerationEnabled, promocode, denyPromocode);

        result.paymentUrl = paymentResult.getPaymentUrl(isMobileForm);
        result.amount = paymentResult.getAmount();
        result.validation = paymentResult.getPaymentUrlValidationResult();

        if (result.hasErrors()) {
            return result;
        }

        autopayParams.setPersonId(personId);
        result.cardBindingUrl = turnOnAutopay(subjectUser, shard, walletCid, clientCurrency, autopayParams,
                isMobileForm, successRedirectUrl);

        return result;
    }


    public static class SetUpAutopayParams {
        public Boolean isLegalPerson;
        public BigDecimal paymentSum;
        public BigDecimal remainingSum;
        public Boolean isMobile;
        public @Nullable String cardId;
    }

    public static class SetUpAutopayResult {
        public String bindingUrl;
        public ValidationResult<AutopayParams, Defect> validationResult;
    }

    /**
     * Получить ссылку на привязку карты и добавить задание на включение автоплатежа после привязки
     */
    public SetUpAutopayResult setUpAutopay(User user, Long operatorUid, SetUpAutopayParams input) {
        var result = new SetUpAutopayResult();
        ClientId clientId = user.getClientId();
        CurrencyCode currency = clientService.getWorkCurrency(clientId).getCode();

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Long walletCid = walletRepository.getActualClientWalletId(shard, clientId, currency);
        checkNotNull(walletCid, "no client wallet");

        Long personIdToPay = getPersonId(user, operatorUid, input.isLegalPerson);

        AutopayParams autopayParams = new AutopayParams()
                    .withCardId(input.cardId)
                    .withPaymentSum(input.paymentSum)
                    .withRemainingSum(input.remainingSum)
                    .withPaymentType(AutopaySettingsPaymethodType.CARD)
                    .withPersonId(personIdToPay);

        result.validationResult = paymentValidationService.validateAutopayInput(autopayParams, currency);

        if (!result.validationResult.hasAnyErrors()) {
            result.bindingUrl = turnOnAutopay(
                    user, shard, walletCid, currency, autopayParams, input.isMobile, null);
        }

        return result;
    }

    private Long getPersonId(User user, Long operatorUid, boolean isLegalPerson) {
        Long personIdToPay = balanceService.getOrCreatePerson(user, operatorUid, isLegalPerson);
        checkNotNull(personIdToPay, "no person to pay");
        return personIdToPay;
    }

    /**
     * Получаем ссылку на форму для привязки карты
     * Кидаем исключение:
     * - Если не смогли получить клиентский кошелек
     * - Для юрика с не единственным юридическим плательщиком
     */
    public String getBindingForm(User subjectUser, Long operatorUID,
                                                  CurrencyCode clientCurrency,
                                                  boolean isLegalPerson,
                                                  boolean isMobileForm,
                                                  String successRedirectUrl) {

        ClientId clientId = subjectUser.getClientId();
        int shard = shardHelper.getShardByClientIdStrictly(subjectUser.getClientId());

        Long walletCid = walletRepository.getActualClientWalletId(shard, clientId, clientCurrency);
        checkState(walletCid != null, "no client wallet");

        Long personIdToPay = balanceService.getOrCreatePerson(subjectUser, operatorUID, isLegalPerson);
        checkState(personIdToPay != null, "no person to pay");

        GetCardBindingURLResponse response = balanceService.getCardBinding(subjectUser.getUid(), clientCurrency,
                successRedirectUrl, isMobileForm);

        return response.getBindingUrl();
    }

    /**
     * Подключить автоплатеж сразу или получить ссылку на привязку карты и добавить задание на включение автоплатежа
     * после привязки
     * Возращает ссылку на форму привязки, если понадобилось привязывать
     */
    @Nullable
    public String turnOnAutopay(
            User user, int shard, Long walletCid, CurrencyCode currencyCode, AutopayParams autopayParams,
            boolean isMobileForm, @Nullable String redirectUrl) {

        if (autopayParams.getPaymentSum() == null || autopayParams.getRemainingSum() == null) {
            return null;
        }

        if (isNotBlank(autopayParams.getCardId())) {
            autopayService.turnOnAutopay(shard, user.getUid(), user.getClientId(), autopayParams);
            return null;
        }

        return autopayService.getBindingUrlAndStartAutopayJob(user, shard, walletCid, currencyCode, autopayParams,
                isMobileForm, redirectUrl);

    }

    public BillingData getBillingData(User subjectUser, Long operatorUid) {
        List<CardInfo> userCards = balanceService.getUserCards(subjectUser.getUid());
        boolean hasSingleLegalPerson = balanceService.getOrCreatePerson(subjectUser, operatorUid, true) != null;

        return new BillingData()
                .withBoundCards(userCards)
                .withHasSingleLegalPerson(hasSingleLegalPerson);
    }
}
