package ru.yandex.direct.core.entity.payment.service;

import java.math.BigDecimal;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.core.entity.payment.model.AutopayParams;
import ru.yandex.direct.core.entity.payment.model.PaymentParams;
import ru.yandex.direct.core.entity.promocodes.service.PromocodeHelper;
import ru.yandex.direct.core.entity.promocodes.service.PromocodeValidationContainer;
import ru.yandex.direct.core.validation.defects.Defects;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.currency.MoneyUtils;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.ppcproperty.model.PpcPropertyEnum.NEW_CARD_PAY_LIMIT;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@Service
public class PaymentValidationService {

    private final PromocodeHelper promocodeHelper;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    @Autowired
    public PaymentValidationService(PromocodeHelper promocodeHelper,
                                    PpcPropertiesSupport ppcPropertiesSupport) {
        this.promocodeHelper = promocodeHelper;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    public ValidationResult<PaymentParams, Defect> validate(
            PaymentParams params,
            CurrencyCode clientCurrency,
            PromocodeValidationContainer promocodeValidationContainer,
            boolean useExtendedCodes) {
        var validator = ItemValidationBuilder.of(params, Defect.class);

        validator.item(clientCurrency, "client currency")
                .check(fromPredicate(CurrencyCode.RUB::equals, invalidValue()));

        var currency = Currencies.getCurrency(clientCurrency);

        validator.item(params.getSum(), "sum")
                .check(notLessThan(currency.getMinPay()));

        boolean useNewAutopayCardLimit = ppcPropertiesSupport.find(NEW_CARD_PAY_LIMIT.getName())
                .orElse("0").equals("1");
        BigDecimal maxAutopayCard = MoneyUtils
                .getMaxAutopayCard(currency, useNewAutopayCardLimit);

        validator.item(params.getPaymentSum(), "payment sum")
                .check(notLessThan(currency.getMinAutopay()))
                .check(notGreaterThan(maxAutopayCard));

        validator.item(params.getRemainingSum(), "remaining sum")
                .check(notLessThan(BigDecimal.ZERO))
                .check(notGreaterThan(currency.getMaxAutopayRemaining()));

        validator.item(params.getPromocode(), "promocode")
                .check(useExtendedCodes
                                ? t -> promocodeHelper.checkPromocode(t, promocodeValidationContainer)
                                : fromPredicate(
                                        t -> promocodeHelper.isApplicablePromocode(t, promocodeValidationContainer),
                                        Defects.promocodeDomainOrClientDoesNotMatch()),
                        When.isValid());
        return validator.getResult();
    }


    public ValidationResult<AutopayParams, Defect> validateAutopayInput(AutopayParams input,
                                                                        CurrencyCode clientCurrency) {
        Currency currency = Currencies.getCurrency(clientCurrency);
        ItemValidationBuilder<AutopayParams, Defect> validator = ItemValidationBuilder.of(input, Defect.class);

        boolean useNewAutopayCardLimit = ppcPropertiesSupport.find(NEW_CARD_PAY_LIMIT.getName())
                .orElse("0").equals("1");

        BigDecimal maxAutopayCard = MoneyUtils
                .getMaxAutopayCard(currency, useNewAutopayCardLimit);

        validator.item(input.getPaymentSum(), AutopayParams.PAYMENT_SUM.name())
                .check(notLessThan(currency.getMinAutopay()))
                .check(notGreaterThan(maxAutopayCard));

        validator.item(input.getRemainingSum(), AutopayParams.REMAINING_SUM.name())
                .check(notLessThan(BigDecimal.ZERO))
                .check(notGreaterThan(currency.getMaxAutopayRemaining()));

        return validator.getResult();
    }

}
