package ru.yandex.direct.core.entity.performancefilter.container;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterBaseStatus;
import ru.yandex.direct.core.entity.performancefilter.model.TargetFunnel;
import ru.yandex.direct.dbschema.ppc.enums.BidsPerformanceTargetFunnel;
import ru.yandex.direct.dbutil.model.BusinessIdAndShopId;
import ru.yandex.direct.dbutil.model.ClientId;

import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

/**
 * Контейнер для фильтров, применяемых при отборе PerformanceFilter
 */
public class PerformanceFiltersQueryFilter {

    private final ClientId clientId;
    private final List<Long> campaignIds;
    private final List<Long> adGroupIds;
    private final List<BusinessIdAndShopId> businessIdsAndShopIds;
    private final List<Long> perfFilterIds;
    private final boolean withDeleted;
    private final boolean withNotSynced;
    private final boolean withSuspended;
    private final boolean withSynced;
    private final Set<PerformanceFilterBaseStatus> baseStatuses;
    private final Set<Integer> autobudgetPriorities;
    private final Set<BidsPerformanceTargetFunnel> targetFunnels;
    private final String nameContains;
    private final String nameNotContains;
    private final BigDecimal minPriceCpc;
    private final BigDecimal maxPriceCpc;
    private final BigDecimal minPriceCpa;
    private final BigDecimal maxPriceCpa;

    private PerformanceFiltersQueryFilter(Builder builder) {
        clientId = builder.clientId;
        campaignIds = builder.campaignIds;
        adGroupIds = builder.adGroupIds;
        perfFilterIds = builder.perfFilterIds;
        businessIdsAndShopIds = builder.businessIdsAndShopIds;
        withDeleted = builder.withDeleted;
        withNotSynced = builder.withNotSynced;
        withSuspended = builder.withSuspended;
        withSynced = builder.withSynced;
        baseStatuses = builder.baseStatuses;
        autobudgetPriorities = builder.autobudgetPriorities;
        targetFunnels = builder.targetFunnels;
        nameContains = builder.nameContains;
        nameNotContains = builder.nameNotContains;
        minPriceCpc = builder.minPriceCpc;
        maxPriceCpc = builder.maxPriceCpc;
        minPriceCpa = builder.minPriceCpa;
        maxPriceCpa = builder.maxPriceCpa;
    }

    @Nullable
    public ClientId getClientId() {
        return clientId;
    }

    @Nullable
    public List<Long> getCampaignIds() {
        return campaignIds;
    }

    @Nullable
    public List<Long> getAdGroupIds() {
        return adGroupIds;
    }

    @Nullable
    public List<BusinessIdAndShopId> getBusinessIdsAndShopIds() {
        return businessIdsAndShopIds;
    }

    @Nullable
    public List<Long> getPerfFilterIds() {
        return perfFilterIds;
    }

    public boolean isWithDeleted() {
        return withDeleted;
    }

    public boolean isWithNotSynced() {
        return withNotSynced;
    }

    public boolean isWithSuspended() {
        return withSuspended;
    }

    public boolean isWithSynced() {
        return withSynced;
    }

    @Nullable
    public Set<PerformanceFilterBaseStatus> getBaseStatuses() {
        return baseStatuses;
    }

    @Nullable
    public Set<Integer> getAutobudgetPriorities() {
        return autobudgetPriorities;
    }

    @Nullable
    public Set<BidsPerformanceTargetFunnel> getTargetFunnels() {
        return targetFunnels;
    }

    @Nullable
    public String getNameContains() {
        return nameContains;
    }

    @Nullable
    public String getNameNotContains() {
        return nameNotContains;
    }

    @Nullable
    public BigDecimal getMinPriceCpc() {
        return minPriceCpc;
    }

    @Nullable
    public BigDecimal getMaxPriceCpc() {
        return maxPriceCpc;
    }

    @Nullable
    public BigDecimal getMinPriceCpa() {
        return minPriceCpa;
    }

    @Nullable
    public BigDecimal getMaxPriceCpa() {
        return maxPriceCpa;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static class Builder {

        private ClientId clientId = null;
        private List<Long> campaignIds = null;
        private List<Long> adGroupIds = null;
        private List<Long> perfFilterIds = null;
        private List<BusinessIdAndShopId> businessIdsAndShopIds = null;
        private boolean withDeleted = true;
        private boolean withNotSynced = true;
        private boolean withSuspended = true;
        private boolean withSynced = true;
        private Set<PerformanceFilterBaseStatus> baseStatuses;
        private Set<Integer> autobudgetPriorities;
        private Set<BidsPerformanceTargetFunnel> targetFunnels;
        private String nameContains;
        private String nameNotContains;
        private BigDecimal minPriceCpc = null;
        private BigDecimal maxPriceCpc = null;
        private BigDecimal minPriceCpa = null;
        private BigDecimal maxPriceCpa = null;

        private Builder() {
        }

        public Builder withClientId(@Nullable ClientId clientId) {
            this.clientId = clientId;
            return this;
        }

        public Builder withCampaignIds(@Nullable Collection<Long> campaignIds) {
            this.campaignIds = (campaignIds == null) ? null : List.copyOf(campaignIds);
            return this;
        }

        public Builder withAdGroupIds(@Nullable Collection<Long> adGroupIds) {
            this.adGroupIds = (adGroupIds == null) ? null : List.copyOf(adGroupIds);
            return this;
        }

        public Builder withBusinessIdsAndShopIds(@Nullable Collection<BusinessIdAndShopId> businessIdsAndShopIds) {
            this.businessIdsAndShopIds = (businessIdsAndShopIds == null) ? null : List.copyOf(businessIdsAndShopIds);
            return this;
        }

        public Builder withPerfFilterIds(@Nullable Collection<Long> perfFilterIds) {
            this.perfFilterIds = (perfFilterIds == null) ? null : List.copyOf(perfFilterIds);
            return this;
        }

        public Builder withoutDeleted() {
            this.withDeleted = false;
            return this;
        }

        public Builder withoutNotSynced() {
            this.withNotSynced = false;
            return this;
        }

        public Builder withoutSuspended() {
            this.withSuspended = false;
            return this;
        }

        public Builder withoutSynced() {
            this.withSynced = false;
            return this;
        }

        public Builder withBaseStatuses(@Nullable Set<PerformanceFilterBaseStatus> baseStatuses) {
            this.baseStatuses = (baseStatuses == null) ? null : Set.copyOf(baseStatuses);
            return this;
        }

        public Builder withAutobudgetPriorities(@Nullable Set<Integer> autobudgetPriorities) {
            this.autobudgetPriorities = autobudgetPriorities;
            return this;
        }

        public Builder withTargetFunnels(@Nullable Set<TargetFunnel> targetFunnels) {
            this.targetFunnels = mapSet(targetFunnels, TargetFunnel::toSource);
            return this;
        }

        public Builder withNameContains(@Nullable String nameContains) {
            this.nameContains = nameContains;
            return this;
        }

        public Builder withNameNotContains(@Nullable String nameNotContains) {
            this.nameNotContains = nameNotContains;
            return this;
        }

        public Builder withMinPriceCpc(@Nullable BigDecimal minPriceCpc) {
            this.minPriceCpc = minPriceCpc;
            return this;
        }

        public Builder withMaxPriceCpc(@Nullable BigDecimal maxPriceCpc) {
            this.maxPriceCpc = maxPriceCpc;
            return this;
        }

        public Builder withMinPriceCpa(@Nullable BigDecimal minPriceCpa) {
            this.minPriceCpa = minPriceCpa;
            return this;
        }

        public Builder withMaxPriceCpa(@Nullable BigDecimal maxPriceCpa) {
            this.maxPriceCpa = maxPriceCpa;
            return this;
        }

        public PerformanceFiltersQueryFilter build() {
            return new PerformanceFiltersQueryFilter(this);
        }
    }
}
