package ru.yandex.direct.core.entity.performancefilter.repository;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.type.TypeReference;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Record;
import org.jooq.SelectJoinStep;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.performancefilter.container.PerformanceFiltersQueryFilter;
import ru.yandex.direct.core.entity.performancefilter.model.NowOptimizingBy;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterBaseStatus;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterTab;
import ru.yandex.direct.core.entity.performancefilter.model.TargetFunnel;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatParser;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatSerializer;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.dbschema.ppc.enums.BidsPerformanceFromTab;
import ru.yandex.direct.dbschema.ppc.enums.BidsPerformanceStatusbssynced;
import ru.yandex.direct.dbschema.ppc.tables.records.BidsPerformanceRecord;
import ru.yandex.direct.dbutil.model.BusinessIdAndShopId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.dbschema.ppc.tables.AdgroupsPerformance.ADGROUPS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.tables.BidsPerformance.BIDS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.tables.CampSecondaryOptions.CAMP_SECONDARY_OPTIONS;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.Feeds.FEEDS;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class PerformanceFilterRepository {

    private static final String PERF_FILTER_FROM_TAB = "perf_filter:from_tab";

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    public final JooqMapperWithSupplier<PerformanceFilter> jooqMapper;
    private final PerformanceFilterConditionDBFormatParser performanceFilterConditionDBFormatParser;
    private final PerformanceFilterStorage performanceFilterStorage;

    @Autowired
    public PerformanceFilterRepository(DslContextProvider dslContextProvider,
                                       ShardHelper shardHelper,
                                       PerformanceFilterStorage performanceFilterStorage) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.performanceFilterStorage = performanceFilterStorage;
        this.performanceFilterConditionDBFormatParser = PerformanceFilterConditionDBFormatParser.INSTANCE;
        jooqMapper = createMapper();
    }

    private static JooqMapperWithSupplier<PerformanceFilter> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(PerformanceFilter::new)
                .map(property(PerformanceFilter.ID, BIDS_PERFORMANCE.PERF_FILTER_ID))
                .map(property(PerformanceFilter.PID, BIDS_PERFORMANCE.PID))
                .map(property(PerformanceFilter.NAME, BIDS_PERFORMANCE.NAME))
                .map(property(PerformanceFilter.PRICE_CPC, BIDS_PERFORMANCE.PRICE_CPC))
                .map(property(PerformanceFilter.PRICE_CPA, BIDS_PERFORMANCE.PRICE_CPA))
                .map(integerProperty(PerformanceFilter.AUTOBUDGET_PRIORITY, BIDS_PERFORMANCE.AUTOBUDGET_PRIORITY))
                .map(convertibleProperty(PerformanceFilter.TARGET_FUNNEL, BIDS_PERFORMANCE.TARGET_FUNNEL,
                        TargetFunnel::fromSource, TargetFunnel::toSource))
                .map(property(PerformanceFilter.FEED_ID, FEEDS.FEED_ID))
                .map(convertibleProperty(PerformanceFilter.FEED_TYPE, FEEDS.FEED_TYPE,
                        FeedType::fromTypedValue,
                        v -> v == null ? null : v.getTypedValue()
                ))
                .map(convertibleProperty(PerformanceFilter.BUSINESS_TYPE, FEEDS.BUSINESS_TYPE,
                        BusinessType::fromSource,
                        BusinessType::toSource)
                )
                .readProperty(PerformanceFilter.SOURCE, fromField(FEEDS.SOURCE).by(Source::fromSource))
                .map(convertibleProperty(PerformanceFilter.NOW_OPTIMIZING_BY,
                        BIDS_PERFORMANCE.NOW_OPTIMIZING_BY,
                        NowOptimizingBy::fromSource, NowOptimizingBy::toSource))
                .map(property(PerformanceFilter.LAST_CHANGE, BIDS_PERFORMANCE.LAST_CHANGE))
                .map(convertibleProperty(PerformanceFilter.CONDITIONS, BIDS_PERFORMANCE.CONDITION_JSON,
                        //условия не конвертируем в маппере, т.к. нужен контекст из PerformanceFilterStorage
                        v -> null,
                        PerformanceFilterRepository::toDb))
                .map(convertibleProperty(PerformanceFilter.STATUS_BS_SYNCED, BIDS_PERFORMANCE.STATUS_BS_SYNCED,
                        PerformanceFilterRepository::statusBsSyncedFromDb,
                        PerformanceFilterRepository::statusBsSyncedToDb))
                .map(booleanProperty(PerformanceFilter.IS_SUSPENDED, BIDS_PERFORMANCE.IS_SUSPENDED))
                .map(booleanProperty(PerformanceFilter.IS_DELETED, BIDS_PERFORMANCE.IS_DELETED))
                .map(property(PerformanceFilter.RET_COND_ID, BIDS_PERFORMANCE.RET_COND_ID))
                .map(convertibleProperty(PerformanceFilter.TAB, BIDS_PERFORMANCE.FROM_TAB,
                        PerformanceFilterTab::fromSource,
                        v -> v == null ? BidsPerformanceFromTab.condition : PerformanceFilterTab.toSource(v))
                )
                .build();
    }

    public static BidsPerformanceStatusbssynced statusBsSyncedToDb(StatusBsSynced statusBsSynced) {
        return BidsPerformanceStatusbssynced.valueOf(statusBsSynced.toDbFormat());
    }

    public static StatusBsSynced statusBsSyncedFromDb(BidsPerformanceStatusbssynced statusBsSynced) {
        return StatusBsSynced.valueOfDbFormat(statusBsSynced.getLiteral());
    }

    public List<PerformanceFilter> getFilters(int shard, PerformanceFiltersQueryFilter queryFilter) {
        return joinAllBidsPerformanceRelatedTables(dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(BIDS_PERFORMANCE))
                .where(getCondition(queryFilter))
                .fetch(this::fromDb);
    }

    public Map<BusinessIdAndShopId, List<PerformanceFilter>> getFiltersByBusinessIdAndShopId(
            int shard, PerformanceFiltersQueryFilter queryFilter) {
        return joinAllBidsPerformanceRelatedTables(dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .select(FEEDS.MARKET_BUSINESS_ID, FEEDS.MARKET_SHOP_ID)
                .from(BIDS_PERFORMANCE))
                .where(getCondition(queryFilter))
                .fetchGroups(r -> BusinessIdAndShopId.ofNullable(r.get(FEEDS.MARKET_BUSINESS_ID),
                        r.get(FEEDS.MARKET_SHOP_ID)), this::fromDb);
    }

    private SelectJoinStep<Record> joinAllBidsPerformanceRelatedTables(SelectJoinStep<Record> step) {
        return step
                .join(PHRASES).on(PHRASES.PID.eq(BIDS_PERFORMANCE.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .join(ADGROUPS_PERFORMANCE).on(ADGROUPS_PERFORMANCE.PID.eq(PHRASES.PID))
                .join(FEEDS).on(FEEDS.FEED_ID.eq(ADGROUPS_PERFORMANCE.FEED_ID));
    }

    private static Condition getCondition(PerformanceFiltersQueryFilter queryFilter) {
        Condition condition = DSL.noCondition();
        if (queryFilter.getPerfFilterIds() != null) {
            condition = condition.and(BIDS_PERFORMANCE.PERF_FILTER_ID.in(queryFilter.getPerfFilterIds()));
        }
        if (queryFilter.getBusinessIdsAndShopIds() != null) {
            condition = condition.and(row(FEEDS.MARKET_BUSINESS_ID, FEEDS.MARKET_SHOP_ID).in(
                    listToSet(queryFilter.getBusinessIdsAndShopIds(), id -> row(id.getBusinessId(), id.getShopId()))));
        }
        if (queryFilter.getAdGroupIds() != null) {
            condition = condition.and(BIDS_PERFORMANCE.PID.in(queryFilter.getAdGroupIds()));
        }
        if (queryFilter.getCampaignIds() != null) {
            condition = condition.and(PHRASES.CID.in(queryFilter.getCampaignIds()));
        }
        if (queryFilter.getClientId() != null) {
            condition = condition.and(CAMPAIGNS.CLIENT_ID.eq(queryFilter.getClientId().asLong()));
        }
        if (!queryFilter.isWithDeleted()) {
            condition = condition.and(BIDS_PERFORMANCE.IS_DELETED.eq(RepositoryUtils.FALSE));
        }
        if (!queryFilter.isWithNotSynced()) {
            condition = condition.and(BIDS_PERFORMANCE.STATUS_BS_SYNCED.eq(BidsPerformanceStatusbssynced.Yes));
        }
        if (!queryFilter.isWithSuspended()) {
            condition = condition.and(BIDS_PERFORMANCE.IS_SUSPENDED.eq(RepositoryUtils.FALSE));
        }
        if (!queryFilter.isWithSynced()) {
            condition = condition.and(BIDS_PERFORMANCE.STATUS_BS_SYNCED.ne(BidsPerformanceStatusbssynced.Yes));
        }
        if (isNotEmpty(queryFilter.getBaseStatuses())) {
            condition = condition.and(getConditionFromBaseStatuses(queryFilter.getBaseStatuses()));
        }
        if (queryFilter.getNameContains() != null) {
            condition = condition.and(BIDS_PERFORMANCE.NAME.contains(queryFilter.getNameContains()));
        }
        if (queryFilter.getNameNotContains() != null) {
            condition = condition.and(BIDS_PERFORMANCE.NAME.notContains(queryFilter.getNameNotContains()));
        }
        if (queryFilter.getTargetFunnels() != null) {
            condition = condition.and(BIDS_PERFORMANCE.TARGET_FUNNEL.in(queryFilter.getTargetFunnels()));
        }
        if (queryFilter.getMinPriceCpc() != null) {
            condition = condition.and(BIDS_PERFORMANCE.PRICE_CPC.ge(queryFilter.getMinPriceCpc()));
        }
        if (queryFilter.getMaxPriceCpc() != null) {
            condition = condition.and(BIDS_PERFORMANCE.PRICE_CPC.le(queryFilter.getMaxPriceCpc()));
        }
        if (queryFilter.getMinPriceCpa() != null) {
            condition = condition.and(BIDS_PERFORMANCE.PRICE_CPA.ge(queryFilter.getMinPriceCpa()));
        }
        if (queryFilter.getMaxPriceCpa() != null) {
            condition = condition.and(BIDS_PERFORMANCE.PRICE_CPA.le(queryFilter.getMaxPriceCpa()));
        }
        if (queryFilter.getAutobudgetPriorities() != null) {
            condition = condition.and(BIDS_PERFORMANCE.AUTOBUDGET_PRIORITY.in(queryFilter.getAutobudgetPriorities()));
        }
        return condition;
    }

    private static Condition getConditionFromBaseStatuses(Collection<PerformanceFilterBaseStatus> statuses) {
        return statuses.stream()
                .map(PerformanceFilterRepository::buildFromBaseStatus)
                .reduce(Condition::or)
                .orElse(DSL.trueCondition());
    }

    private static Condition buildFromBaseStatus(PerformanceFilterBaseStatus status) {
        switch (status) {
            case ACTIVE:
                return BIDS_PERFORMANCE.IS_SUSPENDED.eq(RepositoryUtils.FALSE);
            case SUSPENDED:
                return BIDS_PERFORMANCE.IS_SUSPENDED.eq(RepositoryUtils.TRUE);
            default:
                throw new IllegalArgumentException("Unknown status: " + status);
        }
    }

    private void fillPerformanceFilterTabs(int shard, List<PerformanceFilter> filters) {
        Set<Long> filterIds = listToSet(filters, PerformanceFilter::getId);
        List<String> options = dslContextProvider.ppc(shard)
                .selectDistinct(CAMP_SECONDARY_OPTIONS.OPTIONS)
                .from(BIDS_PERFORMANCE)
                .join(PHRASES).on(PHRASES.PID.eq(BIDS_PERFORMANCE.PID))
                .join(CAMP_SECONDARY_OPTIONS).on(CAMP_SECONDARY_OPTIONS.CID.eq(PHRASES.CID))
                .where(BIDS_PERFORMANCE.PERF_FILTER_ID.in(filterIds))
                .and(CAMP_SECONDARY_OPTIONS.KEY.eq(PERF_FILTER_FROM_TAB))
                .fetch(CAMP_SECONDARY_OPTIONS.OPTIONS);

        Map<Long, PerformanceFilterTab> tabByFilterId = StreamEx.of(options)
                .flatMapToEntry(PerformanceFilterRepository::fromJsonFilterTabs)
                .mapValues(PerformanceFilterTab::fromSource)
                .nonNullValues()
                .distinctKeys()
                .toMap();

        for (PerformanceFilter f : filters) {
            Long id = f.getId();
            PerformanceFilterTab tab = tabByFilterId.get(id);
            f.setTab(tab);
        }
    }

    public Map<Long, List<Long>> getFilterIdsByRetargetingConditionIds(int shard, Collection<Long> retCondIds) {
        return dslContextProvider.ppc(shard)
                .select(BIDS_PERFORMANCE.RET_COND_ID, BIDS_PERFORMANCE.PERF_FILTER_ID)
                .from(BIDS_PERFORMANCE)
                .where(BIDS_PERFORMANCE.RET_COND_ID.in(retCondIds))
                .fetchGroups(BIDS_PERFORMANCE.RET_COND_ID, BIDS_PERFORMANCE.PERF_FILTER_ID);
    }

    public Map<Long, List<Long>> getFilterIdsByAdGroupIds(int shard, Collection<Long> agGroupsIds) {
        return dslContextProvider.ppc(shard)
                .select(BIDS_PERFORMANCE.PID, BIDS_PERFORMANCE.PERF_FILTER_ID)
                .from(BIDS_PERFORMANCE)
                .where(BIDS_PERFORMANCE.PID.in(agGroupsIds))
                .and(BIDS_PERFORMANCE.IS_DELETED.eq(RepositoryUtils.FALSE))
                .fetchGroups(BIDS_PERFORMANCE.PID, BIDS_PERFORMANCE.PERF_FILTER_ID);
    }

    public Map<Long, List<PerformanceFilter>> getFiltersByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        PerformanceFiltersQueryFilter queryFilter = PerformanceFiltersQueryFilter.newBuilder()
                .withAdGroupIds(adGroupIds)
                .build();
        List<PerformanceFilter> filters = getFilters(shard, queryFilter);
        return StreamEx.of(filters).groupingBy(PerformanceFilter::getPid);
    }

    public Map<Long, List<PerformanceFilter>> getNotDeletedFiltersByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        PerformanceFiltersQueryFilter queryFilter = PerformanceFiltersQueryFilter.newBuilder()
                .withAdGroupIds(adGroupIds)
                .withoutDeleted()
                .build();
        List<PerformanceFilter> filters = getFilters(shard, queryFilter);
        return StreamEx.of(filters).groupingBy(PerformanceFilter::getPid);
    }

    public List<Long> addPerformanceFilters(int shard, List<PerformanceFilter> performanceFilters) {
        if (isEmpty(performanceFilters)) {
            return emptyList();
        }

        generatePerformanceFilterIds(performanceFilters);
        addToBidsPerformanceTable(shard, performanceFilters);

        return mapList(performanceFilters, PerformanceFilter::getId);
    }

    private void addToBidsPerformanceTable(int shard, List<PerformanceFilter> performanceFilters) {
        new InsertHelper<>(dslContextProvider.ppc(shard), BIDS_PERFORMANCE)
                .addAll(jooqMapper, performanceFilters)
                .executeIfRecordsAdded();
    }

    private void generatePerformanceFilterIds(List<PerformanceFilter> performanceFilters) {
        Iterator<Long> idsIterator = shardHelper.generatePerformanceFilterIds(performanceFilters.size()).iterator();
        performanceFilters.forEach(filter -> filter.setId(idsIterator.next()));
    }

    public List<PerformanceFilter> getFiltersById(int shard, Collection<Long> filterIds) {
        if (isEmpty(filterIds)) {
            return emptyList();
        }
        List<PerformanceFilter> filters = dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(BIDS_PERFORMANCE)
                .join(ADGROUPS_PERFORMANCE).on(ADGROUPS_PERFORMANCE.PID.eq(BIDS_PERFORMANCE.PID))
                .join(FEEDS).on(FEEDS.FEED_ID.eq(ADGROUPS_PERFORMANCE.FEED_ID))
                .where(BIDS_PERFORMANCE.PERF_FILTER_ID.in(filterIds))
                .fetch(this::fromDb);
        return filters;
    }

    public void update(int shard, List<AppliedChanges<PerformanceFilter>> appliedChanges) {
        if (isEmpty(appliedChanges)) {
            return;
        }
        JooqUpdateBuilder<BidsPerformanceRecord, PerformanceFilter> updateBuilder =
                new JooqUpdateBuilder<>(BIDS_PERFORMANCE.PERF_FILTER_ID, appliedChanges);
        updateBuilder.processProperty(PerformanceFilter.ID, BIDS_PERFORMANCE.PERF_FILTER_ID);
        updateBuilder.processProperty(PerformanceFilter.PID, BIDS_PERFORMANCE.PID);
        updateBuilder.processProperty(PerformanceFilter.NAME, BIDS_PERFORMANCE.NAME);
        updateBuilder.processProperty(PerformanceFilter.PRICE_CPC, BIDS_PERFORMANCE.PRICE_CPC);
        updateBuilder.processProperty(PerformanceFilter.PRICE_CPA, BIDS_PERFORMANCE.PRICE_CPA);
        updateBuilder.processProperty(PerformanceFilter.AUTOBUDGET_PRIORITY, BIDS_PERFORMANCE.AUTOBUDGET_PRIORITY,
                RepositoryUtils::intToLong);
        updateBuilder.processProperty(PerformanceFilter.TARGET_FUNNEL, BIDS_PERFORMANCE.TARGET_FUNNEL,
                TargetFunnel::toSource);
        updateBuilder.processProperty(PerformanceFilter.NOW_OPTIMIZING_BY, BIDS_PERFORMANCE.NOW_OPTIMIZING_BY,
                NowOptimizingBy::toSource);
        updateBuilder.processProperty(PerformanceFilter.LAST_CHANGE, BIDS_PERFORMANCE.LAST_CHANGE);
        updateBuilder.processProperty(PerformanceFilter.CONDITIONS, BIDS_PERFORMANCE.CONDITION_JSON,
                PerformanceFilterRepository::toDb);
        updateBuilder.processProperty(PerformanceFilter.STATUS_BS_SYNCED, BIDS_PERFORMANCE.STATUS_BS_SYNCED,
                PerformanceFilterRepository::statusBsSyncedToDb);
        updateBuilder.processProperty(PerformanceFilter.IS_SUSPENDED, BIDS_PERFORMANCE.IS_SUSPENDED,
                RepositoryUtils::booleanToLong);
        updateBuilder.processProperty(PerformanceFilter.IS_DELETED, BIDS_PERFORMANCE.IS_DELETED,
                RepositoryUtils::booleanToLong);
        updateBuilder.processProperty(PerformanceFilter.RET_COND_ID, BIDS_PERFORMANCE.RET_COND_ID);
        updateBuilder.processProperty(PerformanceFilter.TAB, BIDS_PERFORMANCE.FROM_TAB, PerformanceFilterTab::toSource);
        dslContextProvider.ppc(shard)
                .update(BIDS_PERFORMANCE)
                .set(updateBuilder.getValues())
                .where(BIDS_PERFORMANCE.PERF_FILTER_ID.in(updateBuilder.getChangedIds()))
                .execute();
    }

    public PerformanceFilter fromDb(Record record) {
        PerformanceFilter filter = jooqMapper.fromDb(record);
        String conditionJson = record.get(BIDS_PERFORMANCE.CONDITION_JSON);
        FilterSchema filterSchema = performanceFilterStorage.getFilterSchema(filter);
        List<PerformanceFilterCondition> conditions =
                performanceFilterConditionDBFormatParser.parse(filterSchema, conditionJson);
        filter.withConditions(conditions);
        return filter;
    }

    /**
     * Обновление только поля condition_json, метод для проведения эксперимента (см. DIRECT-104046)
     */
    public void updateConditionJsonField(int shard, Long perfFilterId, String conditionJson) {
        dslContextProvider.ppc(shard)
                .update(BIDS_PERFORMANCE)
                .set(BIDS_PERFORMANCE.CONDITION_JSON, conditionJson)
                .where(BIDS_PERFORMANCE.PERF_FILTER_ID.eq(perfFilterId))
                .execute();
    }

    public void resetBidsPerformanceBsStatusAndPriority(int shard, Collection<Long> campaignIds) {
        dslContextProvider.ppc(shard)
                .update(BIDS_PERFORMANCE.join(PHRASES).using(BIDS_PERFORMANCE.PID))
                .set(BIDS_PERFORMANCE.STATUS_BS_SYNCED, BidsPerformanceStatusbssynced.No)
                .set(BIDS_PERFORMANCE.AUTOBUDGET_PRIORITY, 3L)
                .where(BIDS_PERFORMANCE.AUTOBUDGET_PRIORITY.isNull())
                .and(PHRASES.CID.in(campaignIds))
                .execute();
    }

    public void resetBidsPerformanceBsStatus(int shard, Collection<Long> campaignIds) {
        dslContextProvider.ppc(shard)
                .update(BIDS_PERFORMANCE.join(PHRASES).using(BIDS_PERFORMANCE.PID))
                .set(BIDS_PERFORMANCE.STATUS_BS_SYNCED, BidsPerformanceStatusbssynced.No)
                .where(PHRASES.CID.in(campaignIds))
                .execute();
    }

    public List<PerformanceFilterCondition> parseConditionJson(PerformanceFilter filter, String conditionJson) {
        FilterSchema filterSchema = performanceFilterStorage.getFilterSchema(filter);
        return performanceFilterConditionDBFormatParser.parse(filterSchema, conditionJson);
    }

    private static String toDb(List<PerformanceFilterCondition> conditions) {
        return PerformanceFilterConditionDBFormatSerializer.INSTANCE.serialize(conditions);
    }

    private static Map<Long, BidsPerformanceFromTab> fromJsonFilterTabs(@Nullable String filterTabsJson) {
        if (filterTabsJson == null) {
            return emptyMap();
        }
        Map<Long, String> filterTabs = JsonUtils.fromJson(filterTabsJson, new TypeReference<>() {
        });
        Map<Long, BidsPerformanceFromTab> result = EntryStream.of(filterTabs)
                .mapValues(x -> fromString(x))
                .toMap();

        return result;
    }

    private static String toJsonFilterTabs(Map<Long, BidsPerformanceFromTab> filterTabs) {
        SortedMap<String, String> sortedFilterTabs = EntryStream.of(filterTabs)
                .mapKeys(Object::toString)
                .mapValues(BidsPerformanceFromTab::getLiteral)
                .toCustomMap(TreeMap::new);
        return JsonUtils.toJson(sortedFilterTabs);
    }

    private static BidsPerformanceFromTab fromString(String text) {
        for (BidsPerformanceFromTab b : BidsPerformanceFromTab.values()) {
            if (b.getLiteral().equalsIgnoreCase(text)) {
                return b;
            }
        }
        return null;
    }
}
