package ru.yandex.direct.core.entity.performancefilter.schema.parser;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.List;

import com.fasterxml.jackson.core.type.TypeReference;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.performancefilter.container.DecimalRange;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition.PARSED_VALUE;
import static ru.yandex.direct.core.entity.performancefilter.schema.parser.OneNumberListParser.DEFAULT_PRECISION;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

public class DecimalRangeListParser extends ObjectListParser<DecimalRange> {

    public static final int RANGE_MAX_ITEM_COUNT = 10;

    // диапазон чисел
    public static final DecimalRangeListParser INTEGER_RANGE = new DecimalRangeListParser.Builder()
            .withPrecision(0)
            .build();

    // для "price" precision должно быть 2
    public static final DecimalRangeListParser PRICE_RANGE = new Builder()
            .withPrecision(2)
            .build();

    private static final TypeReference<List<String>> TYPE_REFERENCE = new TypeReference<>() {
    };

    private final int precision;

    protected DecimalRangeListParser(int minItemCount, int maxItemCount,
                                     int precision) {
        super(minItemCount, maxItemCount);
        this.precision = precision;
    }

    public int getPrecision() {
        return precision;
    }

    @Override
    protected List<DecimalRange> parseInternal(String value) {
        List<DecimalRange> parsed = StreamEx.of(JsonUtils.fromJson(value, TYPE_REFERENCE))
                .map(DecimalRange::new)
                .toList();
        for (DecimalRange range : parsed) {
            if (range.getLeft() != null) {
                range.withLeft(roundDown(range.getLeft(), precision));
            }
            if (range.getRight() != null) {
                range.withRight(roundDown(range.getRight(), precision));
            }
        }
        return parsed;
    }

    private static BigDecimal roundDown(BigDecimal val, int scale) {
        return val.setScale(scale, RoundingMode.DOWN);
    }

    @Override
    public ValidationResult<PerformanceFilterCondition, Defect> validate(
            PerformanceFilterCondition<List<DecimalRange>> value) {
        ModelItemValidationBuilder<PerformanceFilterCondition> vb = ModelItemValidationBuilder.of(value);
        vb.checkBy(super::validate);
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        vb.list(value.getParsedValue(), PARSED_VALUE)
                .checkEachBy(diapasonValidator(), When.isValid());
        return vb.getResult();
    }

    private Validator<DecimalRange, Defect> diapasonValidator() {
        return diapason -> {
            ItemValidationBuilder<DecimalRange, Defect> vb = ItemValidationBuilder.of(diapason);
            vb.check(hasAtLeastOneBound());
            vb.check(leftBoundCorrect(), When.valueIs(d -> d.getLeft() != null));
            vb.check(rightBoundCorrect(), When.valueIs(d -> d.getRight() != null));
            vb.check(leftLessThanRight(), When.valueIs(d -> d.getLeft() != null && d.getRight() != null));
            return vb.getResult();
        };
    }

    private Constraint<DecimalRange, Defect> hasAtLeastOneBound() {
        return Constraint.fromPredicate(d -> d.getLeft() != null || d.getRight() != null, invalidValue());
    }

    private Constraint<DecimalRange, Defect> leftBoundCorrect() {
        return Constraint.fromPredicate(d -> d.getLeft().compareTo(BigDecimal.ZERO) >= 0, invalidValue());
    }

    private Constraint<DecimalRange, Defect> rightBoundCorrect() {
        return Constraint.fromPredicate(d -> d.getRight().compareTo(BigDecimal.ZERO) >= 0, invalidValue());
    }

    private Constraint<DecimalRange, Defect> leftLessThanRight() {
        return Constraint.fromPredicate(d -> d.getLeft().compareTo(d.getRight()) < 0, invalidValue());
    }

    public static class Builder {
        private int minItemCount = MIN_ITEM_COUNT;
        private int maxItemCount = RANGE_MAX_ITEM_COUNT;
        private int precision = DEFAULT_PRECISION;

        public Builder withMinItemCount(int minItemCount) {
            this.minItemCount = minItemCount;
            return this;
        }

        public Builder withMaxItemCount(int maxItemCount) {
            this.maxItemCount = maxItemCount;
            return this;
        }

        public Builder withPrecision(int precision) {
            this.precision = precision;
            return this;
        }

        public DecimalRangeListParser build() {
            return new DecimalRangeListParser(minItemCount, maxItemCount, precision);
        }
    }
}
