package ru.yandex.direct.core.entity.performancefilter.schema.parser;

import java.util.List;

import com.fasterxml.jackson.core.type.TypeReference;

import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;

public class LongListParser extends ObjectListParser<Long> {
    private static final TypeReference<List<Long>> TYPE_REFERENCE = new TypeReference<List<Long>>() {
    };

    private final long minValue;
    private final long maxValue;

    public static final LongListParser INSTANCE = new Builder().build();

    protected LongListParser(int minItemCount, int maxItemCount, long minValue, long maxValue) {
        super(minItemCount, maxItemCount);
        this.minValue = minValue;
        this.maxValue = maxValue;
    }

    public double getMinValue() {
        return minValue;
    }

    public double getMaxValue() {
        return maxValue;
    }

    @Override
    protected List<Long> parseInternal(String value) {
        return JsonUtils.fromJson(value, TYPE_REFERENCE);
    }

    @Override
    public ValidationResult<PerformanceFilterCondition, Defect> validate(PerformanceFilterCondition<List<Long>> value) {
        ModelItemValidationBuilder<PerformanceFilterCondition> vb = ModelItemValidationBuilder.of(value);
        vb.checkBy(super::validate);
        vb.list(value.getParsedValue(), PerformanceFilterCondition.PARSED_VALUE)
                .checkEach(notLessThan(minValue), When.isValid())
                .checkEach(notGreaterThan(maxValue), When.isValid());
        return vb.getResult();
    }

    public static class Builder {
        private int minItemCount = MIN_ITEM_COUNT;
        private int maxItemCount = MAX_ITEM_COUNT;
        private long minValue = 0L;
        private long maxValue = Long.MAX_VALUE;

        public Builder withMinItemCount(int minItemCount) {
            this.minItemCount = minItemCount;
            return this;
        }

        public Builder withMaxItemCount(int maxItemCount) {
            this.maxItemCount = maxItemCount;
            return this;
        }

        public Builder withMinValue(long minValue) {
            this.minValue = minValue;
            return this;
        }

        public Builder withMaxValue(long maxValue) {
            this.maxValue = maxValue;
            return this;
        }

        public LongListParser build() {
            return new LongListParser(minItemCount, maxItemCount, minValue, maxValue);
        }
    }
}
