package ru.yandex.direct.core.entity.performancefilter.schema.parser;

import java.util.List;
import java.util.Optional;

import com.fasterxml.jackson.core.type.TypeReference;

import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.utils.NumberUtils;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;

public class NumberListParser extends ObjectListParser<Double> {
    private static final TypeReference<List<Double>> TYPE_REFERENCE = new TypeReference<List<Double>>() {
    };

    static final int DEFAULT_PRECISION = 2;

    private final double minValue;
    private final double maxValue;

    protected final int precision;

    public static final NumberListParser INSTANCE = new Builder().build();

    protected NumberListParser(int minItemCount, int maxItemCount,
                               double minValue, double maxValue, int precision) {
        super(minItemCount, maxItemCount);
        this.minValue = minValue;
        this.maxValue = maxValue;
        this.precision = precision;
    }

    public double getMinValue() {
        return minValue;
    }

    public double getMaxValue() {
        return maxValue;
    }

    public int getPrecision() {
        return precision;
    }

    @Override
    protected List<Double> parseInternal(String value) {
        List<Double> parsed = JsonUtils.fromJson(value, TYPE_REFERENCE);
        for (int i = 0; i < parsed.size(); i++) {
            double val = Optional.ofNullable(parsed.get(i))
                    .orElse(Double.NaN);
            parsed.set(i, NumberUtils.roundDown(val, precision));
        }
        return parsed;
    }

    @Override
    public ValidationResult<PerformanceFilterCondition, Defect> validate(
            PerformanceFilterCondition<List<Double>> value) {
        ModelItemValidationBuilder<PerformanceFilterCondition> vb = ModelItemValidationBuilder.of(value);
        vb.checkBy(super::validate);
        vb.list(value.getParsedValue(), PerformanceFilterCondition.PARSED_VALUE)
                .checkEach(notLessThan(minValue), When.isValid())
                .checkEach(notGreaterThan(maxValue), When.isValid());
        return vb.getResult();
    }

    public static class Builder {
        private int minItemCount = MIN_ITEM_COUNT;
        private int maxItemCount = MAX_ITEM_COUNT;
        private int precision = DEFAULT_PRECISION;

        private double minValue = 0;
        private double maxValue = Long.MAX_VALUE;

        public Builder withMinItemCount(int minItemCount) {
            this.minItemCount = minItemCount;
            return this;
        }

        public Builder withMaxItemCount(int maxItemCount) {
            this.maxItemCount = maxItemCount;
            return this;
        }

        public Builder withMinValue(long minValue) {
            this.minValue = minValue;
            return this;
        }

        public Builder withMaxValue(long maxValue) {
            this.maxValue = maxValue;
            return this;
        }

        public Builder withPrecision(int precision) {
            this.precision = precision;
            return this;
        }

        public NumberListParser build() {
            return new NumberListParser(minItemCount, maxItemCount, minValue,
                    maxValue, precision);
        }
    }

}
