package ru.yandex.direct.core.entity.performancefilter.schema.parser;

import java.util.List;

import com.fasterxml.jackson.core.type.TypeReference;

import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.utils.JsonUtils;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.StringConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition.PARSED_VALUE;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.listSize;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

public class StringListParser extends ObjectListParser<String> {
    public static final StringListParser INSTANCE = new Builder().build();

    static final int MIN_STRING_LENGTH = 1;
    static final int MAX_STRING_LENGTH = 175;

    private final int minStringLength;
    private final int maxStringLength;

    private static final TypeReference<List<String>> TYPE_REFERENCE = new TypeReference<List<String>>() {
    };

    private StringListParser(int minStringLength, int maxStringLength, int minItemCount, int maxItemCount) {
        super(minItemCount, maxItemCount);
        this.minStringLength = minStringLength;
        this.maxStringLength = maxStringLength;
    }

    public int getMinStringLength() {
        return minStringLength;
    }

    public int getMaxStringLength() {
        return maxStringLength;
    }

    @Override
    public ValidationResult<PerformanceFilterCondition, Defect> validate(
            PerformanceFilterCondition<List<String>> value) {
        ModelItemValidationBuilder<PerformanceFilterCondition> vb = ModelItemValidationBuilder.of(value);
        vb.checkBy(super::validate);
        if (vb.getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        vb.list(value.getParsedValue(), PARSED_VALUE)
                .check(notNull())
                .check(listSize(minItemCount, maxItemCount), When.isValid())
                .checkEach(StringConstraints.minStringLength(minStringLength), When.isValid())
                .checkEach(StringConstraints.maxStringLength(maxStringLength), When.isValid());
        return vb.getResult();
    }

    @Override
    protected List<String> parseInternal(String value) {
        return JsonUtils.fromJson(value, TYPE_REFERENCE);
    }

    public static class Builder {
        protected int minStringLength = MIN_STRING_LENGTH;
        protected int maxStringLength = MAX_STRING_LENGTH;
        protected int minItemCount = MIN_ITEM_COUNT;
        protected int maxItemCount = MAX_ITEM_COUNT;

        public Builder withMinStringLength(int minStringLength) {
            this.minStringLength = minStringLength;
            return this;
        }

        public Builder withMaxStringLength(int maxStringLength) {
            this.maxStringLength = maxStringLength;
            return this;
        }

        public Builder withMinItemCount(int minItemCount) {
            this.minItemCount = minItemCount;
            return this;
        }

        public Builder withMaxItemCount(int maxItemCount) {
            this.maxItemCount = maxItemCount;
            return this;
        }

        public StringListParser build() {
            return new StringListParser(minStringLength, maxStringLength, minItemCount, maxItemCount);
        }
    }
}
