package ru.yandex.direct.core.entity.performancefilter.service;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleModule;

import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule;
import ru.yandex.direct.core.entity.performancefilter.model.Operator;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.schema.parser.AbstractPerformanceConditionValueParser;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.core.entity.performancefilter.model.Operator.CONTAINS;
import static ru.yandex.direct.core.entity.performancefilter.model.Operator.EQUALS;
import static ru.yandex.direct.core.entity.performancefilter.model.Operator.RANGE;
import static ru.yandex.direct.core.entity.performancefilter.schema.compiled.PerformanceDefault.CATEGORY_ID;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class PerformanceFilterConditionDBFormatParser implements PerformanceFilterConditionParser {
    private static final ObjectMapper OBJECT_MAPPER;

    public static final PerformanceFilterConditionDBFormatParser INSTANCE =
            new PerformanceFilterConditionDBFormatParser();

    static {
        OBJECT_MAPPER = JsonUtils.MAPPER.copy();

        SimpleModule module = new SimpleModule();
        module.addDeserializer(String.class, new RawJsonDeserializer());
        OBJECT_MAPPER.registerModule(module);
    }

    private static final String DESERIALIZATION_ERROR = "can not deserialize object";

    public List<PerformanceFilterCondition> parse(FilterSchema filterSchema, String conditions) {
        Map<String, String> map;
        try {
            map = OBJECT_MAPPER.readValue(conditions, new TypeReference<Map<String, String>>() {
            });
        } catch (IOException e) {
            throw new IllegalArgumentException(DESERIALIZATION_ERROR, e);
        }

        List<PerformanceFilterCondition> lst = new ArrayList<>();
        for (Map.Entry<String, String> e : map.entrySet()) {
            PerformanceFilterCondition condition = getFilterCondition(e.getKey(), e.getValue());
            AbstractPerformanceConditionValueParser parser =
                    filterSchema.getParser(condition.getFieldName(), condition.getOperator());
            condition.setParsedValue(parser.parse(condition));
            lst.add(condition);
        }


        return lst;
    }

    public static void setParsedValue(FilterSchema filterSchema,
                                      List<? extends PerformanceFilterCondition> conditions) {
        for (PerformanceFilterCondition condition : conditions) {
            AbstractPerformanceConditionValueParser parser =
                    filterSchema.getParser(condition.getFieldName(), condition.getOperator());
            condition.setParsedValue(parser.parse(condition));
        }
    }

    private PerformanceFilterCondition<?> getFilterCondition(String key, String value) {
        String[] splitKey = key.split(" ", 2);
        String fieldName = splitKey[0];
        Operator operator;
        if (splitKey.length == 1) {
            operator = Operator.EQUALS;
        } else {
            operator = Operator.fromString(splitKey[1]);
        }
        if (CATEGORY_ID.equals(fieldName)) {
            // id категорий могут быть довольно большими и не всегда умещаются в JS'ный number без потери точности
            // т.к. строковое значение мы в итоге отдаем напрямую фронтенду, представляем их в виде строк
            // строки мы умеем парсить обратно в числа
            List<?> values = JsonUtils.fromJsonIgnoringErrors(value, List.class);
            if (values != null) {
                value = JsonUtils.toJson(mapList(values, Object::toString));
            }
        }
        return new PerformanceFilterCondition<>(fieldName, operator, value);
    }

    public List<DynamicFeedRule> parseDynamic(FilterSchema filterSchema, String conditionJson) {
        List<PerformanceFilterCondition> performanceConditions = parse(filterSchema, conditionJson);
        return mapList(performanceConditions, PerformanceFilterConditionDBFormatParser::convertToDynamic);
    }

    public static Boolean isSupportedByTreeTab(List<? extends PerformanceFilterCondition> conditions) {
        return conditions.stream().allMatch(l -> ("categoryId".equals(l.getFieldName()) && l.getOperator() == EQUALS)
                || ("price".equals(l.getFieldName()) && l.getOperator() == RANGE)
                || ("vendor".equals(l.getFieldName()) && l.getOperator() == CONTAINS)
                || ("available".equals(l.getFieldName()) && l.getOperator() == EQUALS));
    }

    private static <V> DynamicFeedRule<V> convertToDynamic(PerformanceFilterCondition<V> performanceCondition) {
        DynamicFeedRule<V> dynamicFeedRule = new DynamicFeedRule<>(
                performanceCondition.getFieldName(),
                performanceCondition.getOperator(),
                performanceCondition.getStringValue()
        );
        dynamicFeedRule.setParsedValue(performanceCondition.getParsedValue());
        return dynamicFeedRule;
    }

    private static class RawJsonDeserializer extends JsonDeserializer<String> {
        @Override
        public String deserialize(JsonParser jp, DeserializationContext ctxt) throws IOException {
            ObjectMapper mapper = (ObjectMapper) jp.getCodec();
            JsonNode node = mapper.readTree(jp);
            return node.isTextual() ? node.asText() : node.toString();
        }
    }
}
