package ru.yandex.direct.core.entity.performancefilter.service;

import java.io.IOException;
import java.math.RoundingMode;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.module.SimpleModule;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule;
import ru.yandex.direct.core.entity.performancefilter.container.DecimalRange;
import ru.yandex.direct.core.entity.performancefilter.container.Exists;
import ru.yandex.direct.core.entity.performancefilter.model.Operator;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.core.entity.performancefilter.container.DecimalRange.SEPARATOR;
import static ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema.AVAILABLE;
import static ru.yandex.direct.core.entity.performancefilter.schema.compiled.PerformanceDefault.PICKUP;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.CommonUtils.safeCast;

public class PerformanceFilterConditionDBFormatSerializer implements PerformanceFilterConditionSerializer {
    private static final ObjectMapper OBJECT_MAPPER;
    private static final DecimalFormat DECIMAL_FORMAT;

    public static final PerformanceFilterConditionDBFormatSerializer INSTANCE =
            new PerformanceFilterConditionDBFormatSerializer();

    static {
        OBJECT_MAPPER = JsonUtils.MAPPER.copy();

        SimpleModule module = new SimpleModule();
        module.addSerializer(DecimalRange.class, new DecimalRangeSerializer());
        module.addSerializer(Exists.class, new ExistsSerializer());
        module.addSerializer(Double.class, new NumberSerializer());
        OBJECT_MAPPER.registerModule(module);

        DECIMAL_FORMAT = new DecimalFormat("#.##");
        DECIMAL_FORMAT.setRoundingMode(RoundingMode.DOWN);
    }

    @Override
    public String serialize(List<? extends PerformanceFilterCondition> conditions) {
        // используем SortedMap для предсказуемости сериализации
        SortedMap<String, Object> map = new TreeMap<>();

        for (PerformanceFilterCondition condition : conditions) {
            if (isExistsWithFalseValue(condition.getParsedValue())) {
                continue;
            }
            String key = condition.getFieldName() + " " + condition.getOperator();
            Object value = condition.getParsedValue();

            // Это преобразование необходимо, потому что со стороны баннерлэнда == означает числовое сравнение
            // "available ==":true  ->   "available":"true"
            if (AVAILABLE.equals(condition.getFieldName())) {
                if ((Operator.EQUALS == condition.getOperator()) && Boolean.TRUE.equals(condition.getParsedValue())) {
                    key = condition.getFieldName();
                    value = "true"; // строка как в перле
                } else if (Operator.NOT_EQUALS == condition.getOperator() &&
                        Boolean.FALSE.equals(condition.getParsedValue())) {
                    value = "false";
                } else {
                    continue;
                }
            }
            // "pickup ==":true     ->   "pickup":"true"
            if (PICKUP.equals(condition.getFieldName())) {
                if (Operator.EQUALS == condition.getOperator()
                        && (Boolean.TRUE.equals(condition.getParsedValue()) || Boolean.FALSE.equals(condition.getParsedValue()))) {
                    key = condition.getFieldName();
                    value = Boolean.TRUE.equals(condition.getParsedValue()) ? "true" : "false";
                } else {
                    continue;
                }
            }
            // "url ==":[...]  ->  "url":[...], а также другие поля со списками со строковыми значениями (DIRECT-126857)
            if (Operator.EQUALS == condition.getOperator() && isListWithStrings(value)) {
                key = condition.getFieldName();
            }
            map.put(key, value);
        }
        try {
            return OBJECT_MAPPER.writeValueAsString(map);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    private static boolean isExistsWithFalseValue(Object parsedValue) {
        if (parsedValue instanceof Exists) {
            return !((Exists) parsedValue).value();
        }
        return false;
    }

    private static boolean isListWithStrings(Object value) {
        List<?> values = nvl(safeCast(value, List.class), emptyList());
        return StreamEx.of(values)
                .select(CharSequence.class)
                .remove(StringUtils::isNumeric)
                .findAny()
                .isPresent();
    }

    /**
     * Сериализация для вычисления condition_hash для динамических условий
     * <p>
     * {"available":"true","categoryId ==":[3813,3816]}  ->
     * ["available",1,{"field":"categoryId","relation":"==","value":[3813,3816]}]
     */
    public String serializeForDynamicConditionHash(List<DynamicFeedRule> conditions) {
        Comparator<DynamicFeedRule> comparator = Comparator
                .<DynamicFeedRule, String>comparing(DynamicFeedRule::getFieldName)
                .thenComparing(c -> c.getOperator().toString());

        List<DynamicFeedRule> sortedConditions = conditions.stream()
                .sorted(comparator)
                .collect(toList());

        List<SortedMap<String, Object>> conditionList = new ArrayList<>();
        boolean available = false;

        for (DynamicFeedRule condition : sortedConditions) {
            if (isExistsWithFalseValue(condition.getParsedValue())) {
                continue;
            }
            if (AVAILABLE.equals(condition.getFieldName()) && Operator.EQUALS == condition.getOperator()) {
                if (Boolean.TRUE.equals(condition.getParsedValue())) {
                    available = true;
                }
                continue;
            }

            SortedMap<String, Object> map = new TreeMap<>();
            map.put("field", condition.getFieldName());
            map.put("relation", condition.getOperator().toString());
            map.put("value", condition.getParsedValue());

            conditionList.add(map);
        }
        List<Object> conditionListWithAvailable = new ArrayList<>();
        if (available) {
            conditionListWithAvailable.addAll(List.of(AVAILABLE, 1));
        }
        conditionListWithAvailable.addAll(conditionList);

        try {
            return OBJECT_MAPPER.writeValueAsString(conditionListWithAvailable);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    private static class DecimalRangeSerializer extends JsonSerializer<DecimalRange> {
        @Override
        public void serialize(DecimalRange value, JsonGenerator gen, SerializerProvider serializers) throws IOException {
            String left = value.getLeft() == null ? "" : value.getLeft().toString();
            String right = value.getRight() == null ? "" : value.getRight().toString();
            gen.writeString(left + SEPARATOR + right);
        }
    }

    private static class ExistsSerializer extends JsonSerializer<Exists> {
        @Override
        public void serialize(Exists value, JsonGenerator gen, SerializerProvider serializers) throws IOException {
            if (value.value()) {
                gen.writeString("1");
            }
        }
    }

    private static class NumberSerializer extends JsonSerializer<Double> {
        @Override
        public void serialize(Double value, JsonGenerator gen, SerializerProvider serializers) throws IOException {
            gen.writeNumber(DECIMAL_FORMAT.format(value));
        }
    }
}
