package ru.yandex.direct.core.entity.performancefilter.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.autobudget.service.AutobudgetAlertService;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.performancefilter.container.PerformanceFilterSelectionCriteria;
import ru.yandex.direct.core.entity.performancefilter.container.PerformanceFiltersQueryFilter;
import ru.yandex.direct.core.entity.performancefilter.model.Operator;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterTab;
import ru.yandex.direct.core.entity.performancefilter.repository.PerformanceFilterRepository;
import ru.yandex.direct.dbutil.model.BusinessIdAndShopId;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class PerformanceFilterService implements EntityService<PerformanceFilter, Long> {
    private final PerformanceFilterValidationService validationService;
    private final AutobudgetAlertService autobudgetAlertService;
    private final PerformanceFilterRepository performanceFilterRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final ShardHelper shardHelper;
    private final LogPriceService logPriceService;
    private final ClientService clientService;
    private final FeatureService featureService;
    private final PpcPropertiesSupport ppcPropertiesSupport;


    public PerformanceFilterService(
            PerformanceFilterValidationService performanceFilterValidationService,
            AutobudgetAlertService autobudgetAlertService,
            PerformanceFilterRepository performanceFilterRepository,
            AdGroupRepository adGroupRepository,
            BannerCommonRepository bannerCommonRepository,
            ShardHelper shardHelper,
            LogPriceService logPriceService,
            ClientService clientService,
            FeatureService featureService,
            PpcPropertiesSupport ppcPropertiesSupport) {
        this.validationService = performanceFilterValidationService;
        this.autobudgetAlertService = autobudgetAlertService;
        this.performanceFilterRepository = performanceFilterRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.shardHelper = shardHelper;
        this.logPriceService = logPriceService;
        this.clientService = clientService;
        this.featureService = featureService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    private static ModelChanges<PerformanceFilter> toModelChanges(PerformanceFilter filter) {
        ModelChanges<PerformanceFilter> modelChanges = new ModelChanges<>(filter.getId(), PerformanceFilter.class);
        modelChanges.processNotNull(filter.getPid(), PerformanceFilter.PID);
        modelChanges.processNotNull(filter.getName(), PerformanceFilter.NAME);
        modelChanges.processNotNull(filter.getPriceCpc(), PerformanceFilter.PRICE_CPC);
        modelChanges.processNotNull(filter.getPriceCpa(), PerformanceFilter.PRICE_CPA);
        modelChanges.processNotNull(filter.getAutobudgetPriority(), PerformanceFilter.AUTOBUDGET_PRIORITY);
        modelChanges.processNotNull(filter.getTargetFunnel(), PerformanceFilter.TARGET_FUNNEL);
        modelChanges.processNotNull(filter.getNowOptimizingBy(), PerformanceFilter.NOW_OPTIMIZING_BY);
        modelChanges.processNotNull(filter.getLastChange(), PerformanceFilter.LAST_CHANGE);
        modelChanges.processNotNull(filter.getConditions(), PerformanceFilter.CONDITIONS);
        modelChanges.processNotNull(filter.getStatusBsSynced(), PerformanceFilter.STATUS_BS_SYNCED);
        modelChanges.processNotNull(filter.getIsSuspended(), PerformanceFilter.IS_SUSPENDED);
        modelChanges.processNotNull(filter.getIsDeleted(), PerformanceFilter.IS_DELETED);
        modelChanges.processNotNull(filter.getRetCondId(), PerformanceFilter.RET_COND_ID);
        modelChanges.processNotNull(filter.getTab(), PerformanceFilter.TAB);
        return modelChanges;
    }

    // Не стоит использовать этот метод для внешних доступов, в нём нет валидации на правомерность доступа к фильтру.
    public Map<Long, List<PerformanceFilter>> getPerformanceFilters(ClientId clientId, List<Long> adGroupIds) {
        int shard = shardHelper.getShardByClientId(clientId);
        Map<Long, List<PerformanceFilter>> filtersByAdGroupIds =
                performanceFilterRepository.getFiltersByAdGroupIds(shard, adGroupIds);
        filtersByAdGroupIds.replaceAll((k, v) -> mapList(v, this::removeDefaultFilterConditions));
        return filtersByAdGroupIds;
    }

    public List<PerformanceFilter> getPerformanceFilters(ClientId clientId, PerformanceFiltersQueryFilter queryFilter) {
        int shard = shardHelper.getShardByClientId(clientId);
        return mapList(performanceFilterRepository.getFilters(shard, queryFilter), this::removeDefaultFilterConditions);
    }

    public Map<BusinessIdAndShopId, List<PerformanceFilter>> getPerformanceFiltersByBusinessIdAndShopId(
            ClientId clientId, PerformanceFiltersQueryFilter queryFilter) {
        int shard = shardHelper.getShardByClientId(clientId);
        Map<BusinessIdAndShopId, List<PerformanceFilter>> filtersByBusinessIdAndShopId =
                performanceFilterRepository.getFiltersByBusinessIdAndShopId(shard, queryFilter);
        filtersByBusinessIdAndShopId.replaceAll((k, v) -> mapList(v, this::removeDefaultFilterConditions));
        return filtersByBusinessIdAndShopId;
    }

    @Override
    public List<PerformanceFilter> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        int shard = shardHelper.getShardByClientId(clientId);
        return mapList(performanceFilterRepository.getFiltersById(shard, ids), this::removeDefaultFilterConditions);
    }

    @Override
    public MassResult<Long> add(
            ClientId clientId, Long operatorUid, List<PerformanceFilter> entities, Applicability applicability) {
        PerformanceFiltersAddOperation addOperation = createPartialAddOperation(clientId, operatorUid, entities);
        return addOperation.prepareAndApply();
    }

    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer,
                                 List<PerformanceFilter> entities, Applicability applicability) {
        return add(copyContainer.getClientIdTo(), copyContainer.getOperatorUid(), entities, applicability);
    }

    public PerformanceFiltersAddOperation createPartialAddOperation(ClientId clientId,
                                                                    long operatorUid,
                                                                    List<PerformanceFilter> performanceFilters) {
        return createAddOperation(clientId, operatorUid, performanceFilters, Applicability.PARTIAL);
    }

    private PerformanceFiltersAddOperation createAddOperation(ClientId clientId,
                                                              long operatorUid,
                                                              List<PerformanceFilter> performanceFilters,
                                                              Applicability applicability) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return new PerformanceFiltersAddOperation(applicability,
                performanceFilters,
                performanceFilterRepository,
                adGroupRepository,
                bannerCommonRepository,
                validationService,
                autobudgetAlertService,
                logPriceService,
                clientService,
                ppcPropertiesSupport,
                shard,
                clientId,
                operatorUid);
    }

    public MassResult<Long> updatePerformanceFilters(ClientId clientId, long operatorUid,
                                                     Collection<PerformanceFilter> performanceFilters) {
        PerformanceFiltersUpdateOperation updateOperation =
                createUpdateOperation(clientId, operatorUid, performanceFilters, Applicability.PARTIAL);
        return updateOperation.prepareAndApply();
    }

    public MassResult<Long> updatePerformanceFilters(ClientId clientId, long operatorUid,
                                                     List<ModelChanges<PerformanceFilter>> performanceFilters,
                                                     Applicability applicability) {
        PerformanceFiltersUpdateOperation updateOperation =
                createUpdateOperation(clientId, operatorUid, performanceFilters, applicability);
        return updateOperation.prepareAndApply();
    }

    PerformanceFiltersUpdateOperation createUpdateOperation(
            ClientId clientId,
            long operatorUid,
            Collection<PerformanceFilter> performanceFilters,
            Applicability applicability) {
        List<ModelChanges<PerformanceFilter>> modelChanges =
                mapList(performanceFilters, PerformanceFilterService::toModelChanges);
        return createUpdateOperation(clientId, operatorUid, modelChanges, applicability);
    }

    private PerformanceFiltersUpdateOperation createUpdateOperation(
            ClientId clientId,
            long operatorUid,
            List<ModelChanges<PerformanceFilter>> modelChanges,
            Applicability applicability) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return new PerformanceFiltersUpdateOperation(applicability, modelChanges, performanceFilterRepository,
                adGroupRepository, bannerCommonRepository, validationService, logPriceService, clientService,
                featureService, ppcPropertiesSupport, shard, clientId, operatorUid);
    }
    /**
     * Обновление только поля condition_json, метод для проведения эксперимента (см. DIRECT-104046)
     * Пока что это эксперементальный метод, не стоит его использовать для продакшен-процессов, у него другое поведение!
     */
    public void updateConditionJsonField(ClientId clientId, Long adGroupId, Long perfFilterId, String conditionJson) {
        int shard = shardHelper.getShardByClientId(clientId);
        performanceFilterRepository.updateConditionJsonField(shard, perfFilterId, conditionJson);
        AdGroup adGroup = adGroupRepository.getAdGroups(shard, singletonList(adGroupId)).get(0);
        if (!adGroup.getStatusModerate().equals(StatusModerate.NEW)) {
            adGroupRepository.setStatusBlGeneratedForPerformanceAdGroups(shard, singletonList(adGroupId));
        }
    }

    public void updatePerformanceFilterTab(ClientId clientId, Long filterId, PerformanceFilterTab tab) {
        int shard = shardHelper.getShardByClientId(clientId);
        ModelChanges<PerformanceFilter> changes = new ModelChanges<>(filterId, PerformanceFilter.class);
        changes.processNotNull(tab, PerformanceFilter.TAB);
        PerformanceFilter performanceFilter = new PerformanceFilter().withId(filterId);
        performanceFilterRepository.update(shard, singletonList(changes.applyTo(performanceFilter)));
    }

    public List<PerformanceFilterCondition> parseConditionJson(PerformanceFilter filter, String conditionJson) {
        if (conditionJson.trim().equals("{}")) {
            return emptyList();
        }
        return performanceFilterRepository.parseConditionJson(filter, conditionJson);
    }

    public MassResult<Long> deletePerformanceFilters(ClientId clientId, long operatorUid, List<Long> deleteFilterIds) {
        List<PerformanceFilter> performanceFilters = StreamEx.of(deleteFilterIds)
                .map(id -> new PerformanceFilter().withId(id).withIsDeleted(true))
                .toList();
        return updatePerformanceFilters(clientId, operatorUid, performanceFilters);
    }

    public List<PerformanceFilter> getPerfFiltersBySelectionCriteria(ClientId clientId, Long operatorUid,
                                                                     PerformanceFilterSelectionCriteria selectionCriteria) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<PerformanceFilter> filters = new ArrayList<>();
        filters.addAll(getByFilterIds(shard, selectionCriteria));
        filters.addAll(getByAdGroupIds(shard, selectionCriteria));
        filters.addAll(getByCampaignIds(shard, selectionCriteria));

        ValidationResult<List<PerformanceFilter>, Defect> validationResult =
                validationService.validateReadAccess(clientId, operatorUid, filters);
        List<PerformanceFilter> validFilters = ValidationResult.getValidItems(validationResult);

        LimitOffset limitOffset = nvl(selectionCriteria.getLimitOffset(), LimitOffset.maxLimited());
        return StreamEx.of(validFilters)
                .distinct(PerformanceFilter::getPerfFilterId)
                .sortedBy(PerformanceFilter::getPerfFilterId)
                .skip(limitOffset.offset())
                .limit(limitOffset.limit())
                .toList();
    }

    private List<PerformanceFilter> getByFilterIds(int shard, PerformanceFilterSelectionCriteria selectionCriteria) {
        return getPerformanceFilters(shard, selectionCriteria, PerformanceFilterSelectionCriteria::getPerfFilterIds,
                PerformanceFiltersQueryFilter.Builder::withPerfFilterIds, true);
    }

    private List<PerformanceFilter> getByAdGroupIds(int shard, PerformanceFilterSelectionCriteria selectionCriteria) {
        return getPerformanceFilters(shard, selectionCriteria, PerformanceFilterSelectionCriteria::getAdGroupIds,
                PerformanceFiltersQueryFilter.Builder::withAdGroupIds, false);
    }

    private List<PerformanceFilter> getByCampaignIds(int shard, PerformanceFilterSelectionCriteria selectionCriteria) {
        return getPerformanceFilters(shard, selectionCriteria, PerformanceFilterSelectionCriteria::getCampaignIds,
                PerformanceFiltersQueryFilter.Builder::withCampaignIds, false);
    }

    private List<PerformanceFilter> getPerformanceFilters(int shard,
                                                          PerformanceFilterSelectionCriteria selectionCriteria,
                                                          GetIds getIds, SetIds setIds, boolean ignoreFilterStates) {
        List<Long> ids = getIds.apply(selectionCriteria);
        if (isEmpty(ids)) {
            return emptyList();
        }
        PerformanceFiltersQueryFilter.Builder builder = PerformanceFiltersQueryFilter.newBuilder();
        setIds.apply(builder, ids);
        if (!ignoreFilterStates) {
            if (!selectionCriteria.isWithDeleted()) {
                builder.withoutDeleted();
            }
            if (!selectionCriteria.isWithNotSynced()) {
                builder.withoutNotSynced();
            }
            if (!selectionCriteria.isWithSuspended()) {
                builder.withoutSuspended();
            }
            if (!selectionCriteria.isWithSynced()) {
                builder.withoutSynced();
            }
        }
        PerformanceFiltersQueryFilter queryFilter = builder.build();
        return mapList(performanceFilterRepository.getFilters(shard, queryFilter), this::removeDefaultFilterConditions);
    }

    // Убираем из ответа дефолтное условие для фильтров групп с source=SITE (DIRECT-161247) и
    // для фильтров ТК (DIRECT-166949)
    private PerformanceFilter removeDefaultFilterConditions(PerformanceFilter filter) {
        filter.withConditions(filterList(filter.getConditions(),
                c -> !(c.getFieldName().equals("available") && c.getOperator() == Operator.NOT_EQUALS)));

        return (filter.getSource() != Source.SITE) ? filter : filter.withConditions(
                filterList(filter.getConditions(), c -> !c.getFieldName().equals("available")));
    }

    @FunctionalInterface
    private interface GetIds {
        List<Long> apply(PerformanceFilterSelectionCriteria selectionCriteria);
    }

    @FunctionalInterface
    private interface SetIds {
        PerformanceFiltersQueryFilter.Builder apply(PerformanceFiltersQueryFilter.Builder builder, List<Long> ids);
    }

}
