package ru.yandex.direct.core.entity.performancefilter.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.campaign.service.CampaignService;
import ru.yandex.direct.core.entity.performancefilter.container.PerformanceFilterSelectionCriteria;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.utils.SetBidItemSearcher;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.MappingPathConverter;
import ru.yandex.direct.validation.result.Path;
import ru.yandex.direct.validation.result.PathConverter;
import ru.yandex.direct.validation.result.PathHelper;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static org.apache.commons.lang3.ObjectUtils.firstNonNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
public class PerformanceFilterSetBidsService {

    private static final PathConverter TO_SET_BID_ITEM_WITH_ID_CONVERTER;
    private static final PathConverter TO_SET_BID_ITEM_WITH_AD_GROUP_ID_CONVERTER;
    private static final PathConverter TO_SET_BID_ITEM_WITH_CAMPAIGN_ID_CONVERTER;

    static {
        Map<String, String> strategyFieldPairs = EntryStream.of(
                PerformanceFilter.PRICE_CPC, SetBidItem.PRICE_SEARCH,
                PerformanceFilter.PRICE_CPA, SetBidItem.PRICE_CONTEXT,
                PerformanceFilter.AUTOBUDGET_PRIORITY, SetBidItem.AUTOBUDGET_PRIORITY)
                .mapKeys(ModelProperty::name)
                .mapValues(ModelProperty::name)
                .toMap();

        TO_SET_BID_ITEM_WITH_ID_CONVERTER = getToSetBidItemConverter(strategyFieldPairs, SetBidItem.ID);
        TO_SET_BID_ITEM_WITH_AD_GROUP_ID_CONVERTER = getToSetBidItemConverter(strategyFieldPairs,
                SetBidItem.AD_GROUP_ID);
        TO_SET_BID_ITEM_WITH_CAMPAIGN_ID_CONVERTER = getToSetBidItemConverter(strategyFieldPairs,
                SetBidItem.CAMPAIGN_ID);
    }

    private final PerformanceFilterValidationService validationService;
    private final PerformanceFilterService performanceFilterService;
    private final CampaignService campaignService;

    public PerformanceFilterSetBidsService(PerformanceFilterValidationService validationService,
                                           PerformanceFilterService performanceFilterService,
                                           CampaignService campaignService) {
        this.validationService = validationService;
        this.performanceFilterService = performanceFilterService;
        this.campaignService = campaignService;
    }

    private static void applyDefects(ValidationResult<SetBidItem, Defect> vr, List<DefectInfo<Defect>> defectInfos,
                                     DefectSetter<Defect> defectSetter) {
        SetBidItem item = vr.getValue();
        PathConverter pathConverter = getPathConverter(item);
        List<DefectInfo<Defect>> convertedDefects = StreamEx.of(defectInfos)
                .map(d -> d.convertPath(pathConverter))
                .distinct(d -> String.join(" : ", d.getPath().getFieldName(), d.getDefect().defectId().getCode()))
                .toList();

        for (DefectInfo<Defect> defectInfo : convertedDefects) {
            Path path = defectInfo.getPath();
            String fieldName = nvl(path.getFieldName(), "");
            PathNode.Field pathNode = PathHelper.field(fieldName);
            Object value;
            switch (fieldName) {
                case "autobudgetPriority":
                    value = item.getAutobudgetPriority();
                    break;
                case "priceContext":
                    value = item.getPriceContext();
                    break;
                case "priceSearch":
                    value = item.getPriceSearch();
                    break;
                default:
                    value = firstNonNull(item.getId(), item.getAdGroupId(), item.getCampaignId());
            }
            ValidationResult<?, Defect> subResult = vr.getOrCreateSubValidationResultWithoutCast(pathNode, value);
            defectSetter.apply(subResult, defectInfo.getDefect());
        }
    }

    private static PathConverter getPathConverter(SetBidItem item) {
        if (item.getId() != null) {
            return TO_SET_BID_ITEM_WITH_ID_CONVERTER;
        }
        if (item.getAdGroupId() != null) {
            return TO_SET_BID_ITEM_WITH_AD_GROUP_ID_CONVERTER;
        }
        if (item.getCampaignId() != null) {
            return TO_SET_BID_ITEM_WITH_CAMPAIGN_ID_CONVERTER;
        }
        // Здесь не должны оказаться
        throw new IllegalStateException();
    }

    private static PathConverter getToSetBidItemConverter(
            Map<String, String> strategyFieldPairs, ModelProperty<SetBidItem, Long> idField) {
        String idFieldName = idField.name();
        var perfFilterStrategyFields = strategyFieldPairs.keySet();
        Map<String, String> toSetBidItemWithIdDict = StreamEx.of(PerformanceFilter.allModelProperties())
                .map(ModelProperty::name)
                .remove(perfFilterStrategyFields::contains)
                .mapToEntry(n -> idFieldName)
                .append(strategyFieldPairs).toMap();
        return MappingPathConverter.builder(SetBidItem.class, "SetBidItem")
                .add(toSetBidItemWithIdDict)
                .build();
    }

    // Нет отдельного валидатора под SetBidItem для смарт фильтров, вместо него в методе сделано перекладывание
    // результатов валидации изменённых по SetBidItem фильтров в результаты валидации SetBidItem. Причина: в операциях
    // со смарт-фильтровами достаточно много бизнес-логики, при этом она не стоит на месте, а регулярно дополнятеся.
    // Поэтому, кажется, перекладывание результатов валидации меньшее зло, чем разнесение и дублирование
    // бизнес-логики смарт-фильтров по нескольким местам.
    public MassResult<SetBidItem> setBids(ClientId clientId, Long operatorUid, List<SetBidItem> setBidItems) {
        if (setBidItems.isEmpty()) {
            return MassResult.emptyMassAction();
        }
        PerformanceFilterSelectionCriteria selectionCriteria = new PerformanceFilterSelectionCriteria()
                .withPerfFilterIds(getSetBidIds(setBidItems, SetBidItem::getId))
                .withAdGroupIds(getSetBidIds(setBidItems, SetBidItem::getAdGroupId))
                .withCampaignIds(getSetBidIds(setBidItems, SetBidItem::getCampaignId))
                .withoutDeleted();
        List<PerformanceFilter> allFilters =
                performanceFilterService.getPerfFiltersBySelectionCriteria(clientId, operatorUid, selectionCriteria);
        Map<Long, Long> campaignIdByPerfFilterId = campaignService.getCampaignIdByPerfFilterId(clientId, allFilters);
        SetBidItemSearcher bidSearcher = new SetBidItemSearcher(setBidItems);
        Map<SetBidItem, List<PerformanceFilter>> perfFiltersBySetBidItem = new HashMap<>();
        for (var filter : allFilters) {
            SetBidItem setBidItem = getSetBidItem(filter, campaignIdByPerfFilterId, bidSearcher);
            applyBids(filter, setBidItem);
            List<PerformanceFilter> setBidFilters =
                    perfFiltersBySetBidItem.computeIfAbsent(setBidItem, k -> new ArrayList<>());
            setBidFilters.add(filter);
        }

        // Каждый SetBids должен либо примениться ко всем своим фильтрам, либо не применяться ни к одному из них.
        // Поэтому, сначала валидируем получившиеся после применения фильтры, потом отсеиваем все фильтры тех
        // SetBids, у которых хотя бы один фильтр не прошёл валидацию.
        ValidationResult<List<PerformanceFilter>, Defect> validationResult =
                validationService.validate(clientId, operatorUid, allFilters);
        HashMap<PerformanceFilter, List<DefectInfo<Defect>>> notValidItemsWithDefects =
                ValidationResult.getNotValidItemsWithDefects(validationResult, false);
        Set<PerformanceFilter> notValidFilters = notValidItemsWithDefects.keySet();
        Set<SetBidItem> notValidSetBidItems = mapSet(notValidFilters, filter -> getSetBidItem(filter,
                campaignIdByPerfFilterId, bidSearcher));
        Set<PerformanceFilter> excludedFilters = StreamEx.of(notValidSetBidItems)
                .map(perfFiltersBySetBidItem::get)
                .flatMap(StreamEx::of)
                .toSet();

        // Применяем изменения
        List<PerformanceFilter> applyFilters = filterList(allFilters, f -> !excludedFilters.contains(f));
        MassResult<Long> longMassResult =
                performanceFilterService.createUpdateOperation(clientId, operatorUid, applyFilters, Applicability.FULL)
                        .prepareAndApply();
        Map<Long, ? extends ValidationResult<?, Defect>> applyValResultByPerfFilterId =
                StreamEx.of(longMassResult.getResult())
                        .mapToEntry(Result::getResult, Result::getValidationResult)
                        .filterValues(Objects::nonNull)
                        .toMap();

        // Конвертируем результаты валидации смарт-фильтров в валидацию SetBidsItems
        ListValidationBuilder<SetBidItem, Defect> lvb = ListValidationBuilder.of(setBidItems);
        lvb.checkEachBy(item ->
                getSetBidItemDefectValidationResult(item, perfFiltersBySetBidItem, notValidItemsWithDefects,
                        notValidSetBidItems, applyValResultByPerfFilterId));
        ValidationResult<List<SetBidItem>, Defect> vr = lvb.getResult();
        return MassResult.successfulMassAction(setBidItems, vr);
    }

    private ValidationResult<SetBidItem, Defect> getSetBidItemDefectValidationResult(
            SetBidItem item,
            Map<SetBidItem, List<PerformanceFilter>> perfFiltersBySetBidItem,
            HashMap<PerformanceFilter, List<DefectInfo<Defect>>> notValidItemsWithDefects,
            Set<SetBidItem> notValidSetBidItems,
            Map<Long, ? extends ValidationResult<?, Defect>> applyValResultByPerfFilterId) {
        ValidationResult<SetBidItem, Defect> vr = new ValidationResult<>(item);
        List<PerformanceFilter> setBidsFilters = perfFiltersBySetBidItem.get(item);

        if (isEmpty(setBidsFilters)) {
            vr.addWarning(CommonDefects.objectNotFound());
        } else if (notValidSetBidItems.contains(item)) {
            List<DefectInfo<Defect>> errorDefects = StreamEx.of(setBidsFilters)
                    .map(notValidItemsWithDefects::get)
                    .filter(Objects::nonNull)
                    .flatMap(StreamEx::of)
                    .toList();
            applyDefects(vr, errorDefects, ValidationResult::addError);
        } else {
            List<? extends ValidationResult<?, Defect>> itemApplyValResults = StreamEx.of(setBidsFilters)
                    .map(PerformanceFilter::getPerfFilterId)
                    .map(applyValResultByPerfFilterId::get)
                    .filter(Objects::nonNull)
                    .toList();
            List<DefectInfo<Defect>> errorDefects = StreamEx.of(itemApplyValResults)
                    .filter(ValidationResult::hasAnyErrors)
                    .map(ValidationResult::flattenErrors)
                    .flatMap(StreamEx::of)
                    .toList();
            applyDefects(vr, errorDefects, ValidationResult::addError);
            List<DefectInfo<Defect>> warningDefects = StreamEx.of(itemApplyValResults)
                    .filter(ValidationResult::hasAnyWarnings)
                    .map(ValidationResult::flattenWarnings)
                    .flatMap(StreamEx::of)
                    .toList();
            applyDefects(vr, warningDefects, ValidationResult::addWarning);
        }
        return vr;
    }

    private SetBidItem getSetBidItem(PerformanceFilter filter, Map<Long, Long> campaignIdByPerfFilterId,
                                     SetBidItemSearcher bidSearcher) {
        Long perfFilterId = filter.getId();
        Long campaignId = campaignIdByPerfFilterId.get(perfFilterId);
        return bidSearcher.get(perfFilterId, filter.getPid(), campaignId);
    }

    private List<Long> getSetBidIds(List<SetBidItem> items, SetBidIdGetter idGetter) {
        return items.stream()
                .map(idGetter::apply)
                .filter(Objects::nonNull)
                .collect(toList());
    }

    private void applyBids(PerformanceFilter filter, SetBidItem setBidItem) {
        filter.withPriceCpc(setBidItem.getPriceSearch());
        filter.withPriceCpa(setBidItem.getPriceContext());
        filter.withAutobudgetPriority(setBidItem.getAutobudgetPriority());
    }

    @FunctionalInterface
    private interface DefectSetter<D> {
        void apply(ValidationResult<?, D> vr, D defect);
    }

    @FunctionalInterface
    private interface SetBidIdGetter {
        Long apply(SetBidItem setBidItem);
    }

}
