package ru.yandex.direct.core.entity.performancefilter.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.validation.constraints.NotNull;

import org.apache.commons.collections4.map.MultiKeyMap;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;

@Service
@ParametersAreNonnullByDefault
public class PerformanceFilterStorage {
    // Схемы фильтров в зависимости от BusinessType и FeedType для фидов с source не SITE
    private final MultiKeyMap<Object, FilterSchema> filterSchemas;

    // Схемы фильтров в зависимости от BusinessType для фидов с source == SITE
    private final Map<BusinessType, FilterSchema> siteFilterSchemas;

    public PerformanceFilterStorage(List<FilterSchema> schemaList) {
        filterSchemas = new MultiKeyMap<>();
        siteFilterSchemas = new EnumMap<>(BusinessType.class);
        for (FilterSchema filterSchema : schemaList) {
            if (filterSchema.isSiteSource()) {
                // схема фильтров для смарта по сайту не зависит от FeedType
                siteFilterSchemas.put(filterSchema.businessType(), filterSchema);
            } else {
                filterSchemas.put(filterSchema.businessType(), filterSchema.feedType(), filterSchema);
            }
        }
    }

    @NotNull
    public FilterSchema getFilterSchema(BusinessType businessType, FeedType feedType) {
        FilterSchema filterSchema = filterSchemas.get(businessType, feedType);
        return filterSchema == null ? defaultSchema() : filterSchema;
    }

    @NotNull
    public FilterSchema getFilterSchema(BusinessType businessType, @Nullable FeedType feedType,
                                        @Nullable Source source) {
        FilterSchema filterSchema = getFilterSchemaOrNull(businessType, feedType, source);
        if (filterSchema != null) {
            return filterSchema;
        } else if (source == Source.SITE) {
            return defaultSiteSchema();
        } else {
            return defaultSchema();
        }
    }

    @Nullable
    public FilterSchema getFilterSchemaOrNull(BusinessType businessType, FeedType feedType) {
        return filterSchemas.get(businessType, feedType);
    }

    @Nullable
    public FilterSchema getFilterSchemaOrNull(BusinessType businessType, @Nullable FeedType feedType,
                                              @Nullable Source source) {
        if (source == Source.SITE) {
            return siteFilterSchemas.get(businessType);
        }
        return filterSchemas.get(businessType, feedType);
    }

    @NotNull
    public FilterSchema getFilterSchema(PerformanceFilter performanceFilter) {
        return getFilterSchema(performanceFilter.getBusinessType(), performanceFilter.getFeedType(),
                performanceFilter.getSource());
    }

    @NotNull
    public FilterSchema getFilterSchema(DynamicFeedAdTarget dynamicFeedAdTarget) {
        return getFilterSchema(dynamicFeedAdTarget.getBusinessType(), dynamicFeedAdTarget.getFeedType());
    }

    public Collection<FilterSchema> getAllSchemas() {
        List<FilterSchema> allSchemas = new ArrayList<>(filterSchemas.size() + siteFilterSchemas.size());
        allSchemas.addAll(filterSchemas.values());
        allSchemas.addAll(siteFilterSchemas.values());
        return allSchemas;
    }

    private FilterSchema defaultSchema() {
        return filterSchemas.get(BusinessType.RETAIL, FeedType.YANDEX_MARKET);
    }

    private FilterSchema defaultSiteSchema() {
        return siteFilterSchemas.get(BusinessType.RETAIL);
    }
}
