package ru.yandex.direct.core.entity.performancefilter.service;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects;
import ru.yandex.direct.core.entity.bids.validation.AutobudgetValidator;
import ru.yandex.direct.core.entity.bids.validation.PriceValidator;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.model.StrategyName;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.AccessDefectPresets;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessValidator;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.repository.PerformanceFilterRepository;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.utils.PerformanceFilterUtils;
import ru.yandex.direct.core.entity.performancefilter.validation.FilterConditionsValidator;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupDefects.inconsistentAdGroupType;
import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.charsAreAllowed;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPC_PER_CAMP;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_AVG_CPC_PER_FILTER;
import static ru.yandex.direct.core.entity.campaign.model.StrategyName.AUTOBUDGET_ROI;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignTypeNotSupported;
import static ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterDefects.filterCountIsTooLarge;
import static ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterDefects.inconsistentCampaignStrategy;
import static ru.yandex.direct.core.validation.defects.RightsDefects.noRights;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;
import static ru.yandex.direct.validation.result.PathHelper.field;

@Service
@ParametersAreNonnullByDefault
public class PerformanceFilterValidationService {
    private static final Map<ModelProperty<? super PerformanceFilter, ?>, Set<StrategyName>> ALLOWED_STRATEGIES_BY_PROPERTY
            = ImmutableMap.of(
            PerformanceFilter.PRICE_CPC, ImmutableSet.of(AUTOBUDGET_AVG_CPC_PER_FILTER, AUTOBUDGET_AVG_CPC_PER_CAMP),

            PerformanceFilter.PRICE_CPA, ImmutableSet.of(AUTOBUDGET_AVG_CPA_PER_CAMP, AUTOBUDGET_AVG_CPA_PER_FILTER),

            PerformanceFilter.AUTOBUDGET_PRIORITY, ImmutableSet.of(AUTOBUDGET_ROI));

    private static final int MAX_FILTER_NAME_LENGTH = 100;
    /**
     * Максимальное количество фильтров (помеченные как удалённые не в счёт), которые могут быть в одной группе
     * объявлений.
     */
    static final int MAX_FILTERS_COUNT = 50;

    private static final CampaignAccessDefects ACCESS_DEFECTS =
            AccessDefectPresets.DEFAULT_DEFECTS.toBuilder()
                    .withTypeNotAllowable(AdGroupDefects.notFound())
                    .withNotVisible(AdGroupDefects.notFound())
                    .withTypeNotSupported(campaignTypeNotSupported())
                    .withNoRights(noRights())
                    .build();

    private final ShardHelper shardHelper;
    private final ClientService clientService;
    private final AdGroupRepository adGroupRepository;
    private final CampaignRepository campaignRepository;
    private final PerformanceFilterRepository performanceFilterRepository;
    private final PerformanceFilterStorage filterSchemaServiceStorage;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;

    PerformanceFilterValidationService(ShardHelper shardHelper,
                                       ClientService clientService,
                                       AdGroupRepository adGroupRepository,
                                       CampaignRepository campaignRepository,
                                       PerformanceFilterRepository performanceFilterRepository,
                                       PerformanceFilterStorage filterSchemaServiceStorage,
                                       CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory) {
        this.shardHelper = shardHelper;
        this.clientService = clientService;
        this.adGroupRepository = adGroupRepository;
        this.campaignRepository = campaignRepository;
        this.performanceFilterRepository = performanceFilterRepository;
        this.filterSchemaServiceStorage = filterSchemaServiceStorage;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
    }

    public ValidationResult<List<ModelChanges<PerformanceFilter>>, Defect> validateModelChanges(ClientId clientId,
                                                                                                List<ModelChanges<PerformanceFilter>> modelChanges) {
        Set<Long> filterIds = listToSet(modelChanges, ModelChanges::getId);
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<PerformanceFilter> actualFilters = performanceFilterRepository.getFiltersById(shard, filterIds);
        Set<Long> existingIds = listToSet(actualFilters, PerformanceFilter::getId);
        Map<Long, DbStrategy> strategyByFilterIds = campaignRepository.getStrategyByFilterIds(shard, filterIds);
        ListValidationBuilder<ModelChanges<PerformanceFilter>, Defect> lvb = ListValidationBuilder.of(modelChanges);
        lvb.checkEachBy(mc -> validateModel(mc, existingIds, strategyByFilterIds));
        return lvb.getResult();
    }

    private ValidationResult<ModelChanges<PerformanceFilter>, Defect> validateModel(ModelChanges<PerformanceFilter> mc,
                                                                                    Set<Long> existingIds,
                                                                                    Map<Long, DbStrategy> strategyByFilterIds) {
        ValidationResult<ModelChanges<PerformanceFilter>, Defect> modelChangesVr = new ValidationResult<>(mc);
        Long filterId = mc.getId();
        if (!existingIds.contains(filterId)) {
            ValidationResult<?, Defect> fieldVr =
                    modelChangesVr.getOrCreateSubValidationResult(field(PerformanceFilter.ID), filterId);
            fieldVr.addError(CommonDefects.objectNotFound());
            return modelChangesVr;
        }
        DbStrategy dbStrategy = strategyByFilterIds.get(filterId);
        checkState(dbStrategy != null && dbStrategy.getStrategyName() != null, "StrategyName not found to " +
                "PerformanceFilterId=%s", filterId);
        StrategyName strategyName = dbStrategy.getStrategyName();
        for (Map.Entry<ModelProperty<? super PerformanceFilter, ?>, Set<StrategyName>> entry :
                ALLOWED_STRATEGIES_BY_PROPERTY.entrySet()) {
            ModelProperty<? super PerformanceFilter, ?> modelProperty = entry.getKey();
            Object propertyValue = mc.getPropIfChanged(modelProperty);
            Set<StrategyName> allowedStrategies = entry.getValue();
            if (isNotEmpty(propertyValue) && !allowedStrategies.contains(strategyName)) {
                ValidationResult<?, Defect> fieldVr =
                        modelChangesVr.getOrCreateSubValidationResult(field(modelProperty),
                                mc.getChangedProp(modelProperty));
                fieldVr.addWarning(inconsistentCampaignStrategy());
            }
        }
        return modelChangesVr;
    }

    public ValidationResult<List<PerformanceFilter>, Defect> validate(ClientId clientId,
                                                                      long operatorUid,
                                                                      List<PerformanceFilter> filters) {
        PerformanceFilterValidationContainer container = initValidationContainer(clientId, filters);
        Set<Long> adGroupIds = StreamEx.of(filters).map(PerformanceFilter::getPid).nonNull().toSet();
        CampaignSubObjectAccessChecker checker =
                campaignSubObjectAccessCheckerFactory.newAdGroupChecker(operatorUid, clientId, adGroupIds);
        CampaignSubObjectAccessValidator adGroupAccessValidator = checker.createValidator(
                CampaignAccessType.READ_WRITE, ACCESS_DEFECTS);
        ListValidationBuilder<PerformanceFilter, Defect> lvb = ListValidationBuilder.of(filters);
        Map<Long, List<PerformanceFilter>> filtersByAdGroupId = StreamEx.of(filters)
                .groupingBy(PerformanceFilter::getPid);
        lvb.checkEachBy(filter -> validateDuplicateInNewFilters(filter, filtersByAdGroupId))
                .checkEachBy(filter -> validateFilter(filter, container, adGroupAccessValidator), When.isValid());
        return lvb.getResult();
    }

    private ValidationResult<PerformanceFilter, Defect> validateFilter(PerformanceFilter performanceFilter,
                                                                       PerformanceFilterValidationContainer container,
                                                                       Validator<Long, Defect> adGroupAccessValidator) {
        ModelItemValidationBuilder<PerformanceFilter> vb = ModelItemValidationBuilder.of(performanceFilter);
        FilterSchema filterSchema = filterSchemaServiceStorage.getFilterSchema(performanceFilter);
        Map<Long, AdGroupSimple> adGroups = container.getAdGroupsById();
        vb.item(PerformanceFilter.PID)
                .check(inSet(adGroups.keySet()), AdGroupDefects.notFound())
                .checkBy(adGroupAccessValidator, When.isValid())
                .checkBy(adGroupId -> validateAdGroupType(adGroups, adGroupId));

        vb.list(PerformanceFilter.CONDITIONS)
                .checkBy(new FilterConditionsValidator(filterSchema, performanceFilter.getTab()));

        vb.item(PerformanceFilter.NAME)
                .check(notNull())
                .check(notBlank())
                .check(charsAreAllowed())
                .check(maxStringLength(MAX_FILTER_NAME_LENGTH));
        vb.item(PerformanceFilter.TARGET_FUNNEL)
                .check(notNull());
        Currency currency = container.getClientWorkCurrency();
        //Ставки могут отсутствовать — тогда будут браться со стратегии.
        vb.item(PerformanceFilter.PRICE_CPC)
                .checkBy(new PriceValidator(currency, AdGroupType.PERFORMANCE),
                        When.isTrue(isNotEmpty(performanceFilter.getPriceCpc())));
        vb.item(PerformanceFilter.PRICE_CPA)
                .checkBy(new PriceValidator(currency, AdGroupType.PERFORMANCE),
                        When.isTrue(isNotEmpty(performanceFilter.getPriceCpa())));

        // Если группа не корректна, то для проверки бизнес-логики нет необходимых данных или они тоже некорректны,
        // поэтому в этом случае завершаем валидацию.
        if (vb.item(PerformanceFilter.PID).getResult().hasAnyErrors()) {
            return vb.getResult();
        }
        Campaign campaign = container.getCampaignsByAdGroupId().get(performanceFilter.getPid());
        DbStrategy strategy = campaign.getStrategy();
        if (Objects.equals(strategy.getStrategyName(), AUTOBUDGET_ROI)) {
            vb.item(PerformanceFilter.AUTOBUDGET_PRIORITY)
                    .check(notNull())
                    .checkBy(new AutobudgetValidator());
        }

        Map<Long, List<PerformanceFilter>> oldFiltersByAdGroupId = container.getOldFiltersByAdGroupId();
        Map<Long, PerformanceFilter> oldFiltersById = container.getOldFiltersById();
        vb.checkBy(filter -> validateFiltersCount(filter, oldFiltersById, oldFiltersByAdGroupId))
                .checkBy(filter -> validateDuplicateInExistingFilters(filter, oldFiltersByAdGroupId), When.isValid());
        return vb.getResult();
    }

    private ValidationResult<Long, Defect> validateAdGroupType(Map<Long, AdGroupSimple> adGroups, Long adGroupId) {
        return Optional.ofNullable(adGroups.get(adGroupId))
                .filter(adGroup -> !adGroup.getType().equals(AdGroupType.PERFORMANCE))
                .map(adGroup -> ValidationResult.failed(adGroupId, (Defect) inconsistentAdGroupType()))
                .orElse(ValidationResult.success(adGroupId));
    }

    private static boolean isNotEmpty(@Nullable Object propertyValue) {
        if (propertyValue == null) {
            return false;
        }
        if (propertyValue instanceof BigDecimal) {
            BigDecimal bd = (BigDecimal) propertyValue;
            return bd.compareTo(BigDecimal.ZERO) != 0;
        }
        return true;
    }

    private ValidationResult<PerformanceFilter, Defect> validateFiltersCount(
            PerformanceFilter filter,
            Map<Long, PerformanceFilter> oldFiltersById,
            Map<Long, List<PerformanceFilter>> filtersByAdGroupId) {
        PerformanceFilter oldFilter = oldFiltersById.get(filter.getId());
        boolean isIncreaseFilterCount = oldFilter == null || (oldFilter.getIsDeleted() && !filter.getIsDeleted());
        List<PerformanceFilter> filtersWithSameAdGroupId = filtersByAdGroupId.get(filter.getPid());
        boolean isTooMuchFilters = filtersWithSameAdGroupId != null &&
                MAX_FILTERS_COUNT <= filtersWithSameAdGroupId.size();
        return isIncreaseFilterCount && isTooMuchFilters
                ? ValidationResult.failed(filter, filterCountIsTooLarge(MAX_FILTERS_COUNT))
                : ValidationResult.success(filter);
    }

    private ValidationResult<PerformanceFilter, Defect> validateDuplicateInNewFilters(
            PerformanceFilter filter,
            Map<Long, List<PerformanceFilter>> filtersByAdGroupId) {
        Predicate<PerformanceFilter> notSamePerformanceFilter = f -> f != filter;
        return validateDuplicateFilters(filter, notSamePerformanceFilter, filtersByAdGroupId);
    }

    private ValidationResult<PerformanceFilter, Defect> validateDuplicateInExistingFilters(
            PerformanceFilter filter,
            Map<Long, List<PerformanceFilter>> filtersByAdGroupId) {
        Predicate<PerformanceFilter> notSamePerformanceFilter = f -> !Objects.equals(f.getId(), filter.getId());
        return validateDuplicateFilters(filter, notSamePerformanceFilter, filtersByAdGroupId);
    }

    private ValidationResult<PerformanceFilter, Defect> validateDuplicateFilters(
            PerformanceFilter filter,
            Predicate<PerformanceFilter> notSamePerformanceFilter,
            Map<Long, List<PerformanceFilter>> filtersByAdGroupId) {
        List<PerformanceFilter> filtersWithSameAdGroupId = filtersByAdGroupId.get(filter.getPid());
        // проверка на isDeleted нужна для удаляемых фильтров, поскольку удаление происходит через update
        if (!nvl(filter.getIsDeleted(), false) && filtersWithSameAdGroupId != null) {
            Optional<PerformanceFilter> duplicatedFilter = StreamEx.of(filtersWithSameAdGroupId)
                    .filter(notSamePerformanceFilter)
                    .findAny(f -> PerformanceFilterUtils.isEqual(f, filter));
            if (duplicatedFilter.isPresent()) {
                return ValidationResult.failed(filter, duplicatedObject());
            }
        }
        return ValidationResult.success(filter);
    }

    private PerformanceFilterValidationContainer initValidationContainer(ClientId clientId,
                                                                         List<PerformanceFilter> filters) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Set<Long> filterIds = StreamEx.of(filters)
                .map(PerformanceFilter::getId)
                .nonNull()
                .toSet();
        List<PerformanceFilter> oldFilters = performanceFilterRepository.getFiltersById(shard, filterIds);
        Map<Long, PerformanceFilter> oldFiltersById = listToMap(oldFilters, PerformanceFilter::getId);
        Set<Long> adGroupIds = listToSet(filters, PerformanceFilter::getPid);
        Map<Long, AdGroupSimple> adGroupsById =
                adGroupRepository.getAdGroupSimple(shard, clientId, adGroupIds);
        Map<Long, List<PerformanceFilter>> oldFiltersByAdGroupId =
                performanceFilterRepository.getNotDeletedFiltersByAdGroupIds(shard, adGroupIds);
        Map<Long, Long> campaignIdByAdGroupId = EntryStream.of(adGroupsById)
                .mapValues(AdGroupSimple::getCampaignId)
                .toMap();
        Map<Long, Campaign> campaignsById =
                listToMap(campaignRepository.getCampaigns(shard, campaignIdByAdGroupId.values()), Campaign::getId);
        Map<Long, Campaign> campaignsByAdGroupId = EntryStream.of(campaignIdByAdGroupId)
                .mapValues(campaignsById::get)
                .filterValues(Objects::nonNull)
                .toMap();

        Currency workCurrency = clientService.getWorkCurrency(clientId);
        return new PerformanceFilterValidationContainer()
                .withClientWorkCurrency(workCurrency)
                .withCampaignsByAdGroupId(campaignsByAdGroupId)
                .withAdGroupsById(adGroupsById)
                .withOldFilters(oldFiltersById)
                .withOldFiltersByAdGroupId(oldFiltersByAdGroupId);
    }

    ValidationResult<List<PerformanceFilter>, Defect> validateReadAccess(ClientId clientId,
                                                                         long operatorUid,
                                                                         List<PerformanceFilter> filters) {
        Set<Long> adGroupIds = listToSet(filters, PerformanceFilter::getPid);
        CampaignSubObjectAccessChecker checker =
                campaignSubObjectAccessCheckerFactory.newAdGroupChecker(operatorUid, clientId, adGroupIds);
        CampaignSubObjectAccessValidator adGroupAccessValidator =
                checker.createValidator(CampaignAccessType.READ, ACCESS_DEFECTS);
        ListValidationBuilder<PerformanceFilter, Defect> lvb = ListValidationBuilder.of(filters);
        lvb.checkEachBy(filter -> {
            ModelItemValidationBuilder<PerformanceFilter> vb = ModelItemValidationBuilder.of(filter);
            vb.item(PerformanceFilter.PID).checkBy(adGroupAccessValidator);
            return vb.getResult();
        });
        return lvb.getResult();
    }

}
