package ru.yandex.direct.core.entity.performancefilter.service;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.StreamEx;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.autobudget.service.AutobudgetAlertService;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.performancefilter.model.NowOptimizingBy;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.repository.PerformanceFilterRepository;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.operation.add.SimpleAbstractAddOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.common.db.PpcPropertyNames.ADD_DEFAULT_SITE_FILTER_CONDITION_ENABLED;
import static ru.yandex.direct.core.entity.performancefilter.utils.PerformanceFilterUtils.addDefaultConditionIfSite;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Операция добавления PerformanceFilter.
 */
public class PerformanceFiltersAddOperation extends SimpleAbstractAddOperation<PerformanceFilter, Long> {

    private final PerformanceFilterRepository performanceFilterRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final PerformanceFilterValidationService validationService;
    private final AutobudgetAlertService autobudgetAlertService;
    private final LogPriceService logPriceService;
    private final ClientService clientService;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    private final int shard;
    private final ClientId clientId;
    private final long operatorUid;


    @SuppressWarnings("checkstyle:parameternumber")
    PerformanceFiltersAddOperation(
            Applicability applicability,
            List<PerformanceFilter> performanceFilters,
            PerformanceFilterRepository performanceFilterRepository,
            AdGroupRepository adGroupRepository,
            BannerCommonRepository bannerCommonRepository,
            PerformanceFilterValidationService performanceFilterValidationService,
            AutobudgetAlertService autobudgetAlertService,
            LogPriceService logPriceService,
            ClientService clientService,
            PpcPropertiesSupport ppcPropertiesSupport,
            int shard, ClientId clientId, long operatorUid) {
        super(applicability, performanceFilters);
        this.validationService = performanceFilterValidationService;
        this.autobudgetAlertService = autobudgetAlertService;
        this.shard = shard;
        this.clientId = clientId;
        this.operatorUid = operatorUid;

        this.performanceFilterRepository = performanceFilterRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.logPriceService = logPriceService;
        this.clientService = clientService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    @Override
    protected void validate(ValidationResult<List<PerformanceFilter>, Defect> preValidationResult) {
        ValidationResult<List<PerformanceFilter>, Defect> validationResult =
                validationService.validate(clientId, operatorUid, getModels());
        preValidationResult.merge(validationResult);
    }

    @Override
    protected void onModelsValidated(ModelsValidatedStep<PerformanceFilter> modelsValidatedStep) {
        Collection<PerformanceFilter> performanceFilters = modelsValidatedStep.getValidModelsMap().values();
        if (isEnabled()) {
            performanceFilters = mapList(performanceFilters, filter ->
                    filter.withConditions(addDefaultConditionIfSite(filter.getConditions(), filter.getSource())));
        }
        prepareSystemFields(performanceFilters);
    }

    private boolean isEnabled() {
        return ppcPropertiesSupport.get(ADD_DEFAULT_SITE_FILTER_CONDITION_ENABLED).getOrDefault(false);
    }

    private void prepareSystemFields(Collection<PerformanceFilter> performanceFilters) {
        LocalDateTime now = LocalDateTime.now();
        performanceFilters.forEach(filter -> filter
                .withLastChange(now)
                .withStatusBsSynced(StatusBsSynced.NO)
                .withIsDeleted(false)
                .withNowOptimizingBy(NowOptimizingBy.CPC)
        );
    }

    @Override
    protected void beforeExecution(Map<Integer, PerformanceFilter> validModelsMapToApply) {
        Collection<PerformanceFilter> performanceFilters = validModelsMapToApply.values();
        setMissingPrices(performanceFilters);
    }

    private void setMissingPrices(Collection<PerformanceFilter> performanceFilters) {
        performanceFilters.forEach(filter -> {
            filter.setPriceCpc(nvl(filter.getPriceCpc(), BigDecimal.ZERO));
            filter.setPriceCpa(nvl(filter.getPriceCpa(), BigDecimal.ZERO));
        });
    }

    private void updateAdGroupAndBannerStatuses(Collection<PerformanceFilter> performanceFilters) {
        Set<Long> adGroupIds = listToSet(performanceFilters, PerformanceFilter::getPid);

        adGroupRepository.updateLastChange(shard, adGroupIds);

        Map<Long, AdGroupSimple> adGroupsById = adGroupRepository.getAdGroupSimple(shard, clientId, adGroupIds);

        List<AdGroupSimple> notDraftAdGroups =
                filterList(adGroupsById.values(), adGroup -> adGroup.getStatusModerate() != StatusModerate.NEW);
        List<Long> notDraftAdGroupIds = mapList(notDraftAdGroups, AdGroupSimple::getId);

        // Если фильтр добавляется в группу не-черновик, то сбросим статус синхронизации на группе
        // Также, сбросим статус синхронизации всех баннеров группы
        adGroupRepository.updateStatusBsSynced(shard, notDraftAdGroupIds, StatusBsSynced.NO);
        bannerCommonRepository.updateStatusBsSyncedByAdgroupId(shard, notDraftAdGroupIds, StatusBsSynced.NO);

        Set<Long> adGroupIdsToSetStatusBlGenerated = StreamEx.of(performanceFilters)
                .remove(PerformanceFilter::getIsSuspended)
                .map(PerformanceFilter::getPid)
                .filter(notDraftAdGroupIds::contains)
                .toSet();

        adGroupRepository.setStatusBlGeneratedForPerformanceAdGroups(shard, adGroupIdsToSetStatusBlGenerated);

        // Для кампаний групп-не-черновиков сделаем заморозку предупреждений автобюджета
        Set<Long> campaignIds = listToSet(notDraftAdGroups, AdGroupSimple::getCampaignId);
        autobudgetAlertService.freezeAlertsOnKeywordsChange(clientId, campaignIds);
    }

    private void logPriceChanges(Collection<PerformanceFilter> performanceFilters) {
        Set<Long> adGroupIds = listToSet(performanceFilters, PerformanceFilter::getPid);
        Map<Long, Long> campaignIdsByAdGroupIds = adGroupRepository.getCampaignIdsByAdGroupIds(shard, adGroupIds);

        Currency clientCurrency = clientService.getWorkCurrency(clientId);

        Function<PerformanceFilter, LogPriceData> performanceFilterToLogFn = filter -> new LogPriceData(
                campaignIdsByAdGroupIds.get(filter.getPid()),
                filter.getPid(),
                filter.getId(),
                nvl(filter.getPriceCpa(), BigDecimal.ZERO).doubleValue(),
                nvl(filter.getPriceCpc(), BigDecimal.ZERO).doubleValue(),
                clientCurrency.getCode(),
                LogPriceData.OperationType.PERF_FILTER_CREATE);

        List<LogPriceData> priceDataList = mapList(performanceFilters, performanceFilterToLogFn);
        logPriceService.logPrice(priceDataList, operatorUid);
    }

    @Override
    protected List<Long> execute(List<PerformanceFilter> validModelsToApply) {
        List<Long> performanceFilterIds = performanceFilterRepository.addPerformanceFilters(shard, validModelsToApply);
        updateAdGroupAndBannerStatuses(validModelsToApply);
        logPriceChanges(validModelsToApply);
        return performanceFilterIds;
    }
}
