package ru.yandex.direct.core.entity.performancefilter.service;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.PerformanceAdGroup;
import ru.yandex.direct.core.entity.adgroup.model.StatusBLGenerated;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.performancefilter.container.PerformanceFilterPreUpdateLogic;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.repository.PerformanceFilterRepository;
import ru.yandex.direct.core.entity.performancefilter.utils.PerformanceFilterUtils;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.AppliedChangesValidatedStep;
import ru.yandex.direct.operation.update.ModelChangesValidatedStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectInfo;
import ru.yandex.direct.validation.result.PathNode;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableList;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.common.db.PpcPropertyNames.ADD_DEFAULT_SITE_FILTER_CONDITION_ENABLED;
import static ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterDefects.PerformanceFilterDefectIds.INCONSISTENT_CAMPAIGN_STRATEGY;
import static ru.yandex.direct.core.entity.performancefilter.utils.PerformanceFilterUtils.addDefaultConditionIfSite;
import static ru.yandex.direct.utils.CommonUtils.notEquals;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.ValidationResult.transferSubNodesWithIssues;

public class PerformanceFiltersUpdateOperation extends SimpleAbstractUpdateOperation<PerformanceFilter, Long> {

    private static final Map<ModelProperty<? super PerformanceFilter, ?>, BiFunction<Object, Object, Boolean>> CUSTOM_MODEL_EQUALS =
            ImmutableMap.of(PerformanceFilter.CONDITIONS, PerformanceFilterUtils::validAndEqual);

    //При изменении не забыть поменять список обрабатываемых моделей в #revertStrategyWarnProperties
    private static final List<ModelProperty> STRATEGY_VALIDATE_PROPERTIES = unmodifiableList(
            asList(PerformanceFilter.PRICE_CPA, PerformanceFilter.PRICE_CPC, PerformanceFilter.AUTOBUDGET_PRIORITY));

    private final HashMap<Long, List<ModelProperty>> strategyWarnPropsByFilterId = new HashMap<>();

    private final PerformanceFilterRepository performanceFilterRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final PerformanceFilterValidationService validationService;
    private final LogPriceService logPriceService;
    private final ClientService clientService;
    private final FeatureService featureService;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final int shard;
    private final ClientId clientId;
    private final long operatorUid;

    PerformanceFiltersUpdateOperation(
            Applicability applicability,
            List<ModelChanges<PerformanceFilter>> modelChanges,
            PerformanceFilterRepository performanceFilterRepository,
            AdGroupRepository adGroupRepository,
            BannerCommonRepository bannerCommonRepository,
            PerformanceFilterValidationService validationService,
            LogPriceService logPriceService,
            ClientService clientService,
            FeatureService featureService,
            PpcPropertiesSupport ppcPropertiesSupport,
            int shard,
            ClientId clientId,
            long operatorUid) {
        super(applicability, modelChanges, id -> new PerformanceFilter().withId(id), emptySet(), CUSTOM_MODEL_EQUALS);
        this.performanceFilterRepository = performanceFilterRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.validationService = validationService;
        this.logPriceService = logPriceService;
        this.clientService = clientService;
        this.featureService = featureService;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.shard = shard;
        this.clientId = clientId;
        this.operatorUid = operatorUid;
    }

    private static boolean isNotDraft(AdGroup adGroup) {
        return notEquals(StatusModerate.NEW, adGroup.getStatusModerate());
    }

    @Override
    protected Collection<PerformanceFilter> getModels(Collection<Long> ids) {
        return performanceFilterRepository.getFiltersById(shard, ids);
    }

    @Override
    protected ValidationResult<List<ModelChanges<PerformanceFilter>>, Defect> validateModelChanges(List<ModelChanges<PerformanceFilter>> modelChanges) {
        return validationService.validateModelChanges(clientId, modelChanges);
    }

    @Override
    protected void onModelChangesValidated(ModelChangesValidatedStep<PerformanceFilter> modelChangesValidatedStep) {
        //Если валидация вернула предупреждения типа INCONSISTENT_CAMPAIGN_STRATEGY (значение не соответствует
        // стратегии компании)
        //для полей PRICE_CPA, PRICE_CPC или AUTOBUDGET_PRIORITY, то значения этих полей в базе менять не будем.
        //Для этого в ModelChanges положим старые значения этих полей.
        ValidationResult<List<ModelChanges<PerformanceFilter>>, Defect> modelChangesVr =
                modelChangesValidatedStep.getModelChangesValidationResult();
        Map<PathNode, ValidationResult<?, Defect>> subResults = modelChangesVr.getSubResults();
        Collection<ValidationResult<?, Defect>> values = subResults.values();
        if (values.isEmpty()) {
            return;
        }

        for (ValidationResult<?, Defect> vr : values) {
            List<DefectInfo<Defect>> defectInfos = vr.flattenWarnings();
            for (DefectInfo<Defect> di : defectInfos) {
                if (di.getDefect().defectId() != INCONSISTENT_CAMPAIGN_STRATEGY) {
                    continue;
                }
                String fieldName = di.getPath().getFieldName();
                ModelChanges modelChanges = (ModelChanges) vr.getValue();
                Long id = modelChanges.getId();
                List<ModelProperty> modelProperties =
                        strategyWarnPropsByFilterId.computeIfAbsent(id, k -> new ArrayList<>());
                for (ModelProperty property : STRATEGY_VALIDATE_PROPERTIES) {
                    if (Objects.equals(fieldName, property.name())) {
                        modelProperties.add(property);
                    }
                }
            }
        }
    }

    @Override
    protected ValidationResult<List<PerformanceFilter>, Defect> validateAppliedChanges(
            ValidationResult<List<PerformanceFilter>, Defect> validationResult) {
        ValidationResult<List<PerformanceFilter>, Defect> validate =
                validationService.validate(clientId, operatorUid, validationResult.getValue());
        // не делаем merge, так как value узлов в validate могут отличаться
        transferSubNodesWithIssues(validate, validationResult);
        return validationResult;
    }

    @Override
    protected void onAppliedChangesValidated(AppliedChangesValidatedStep<PerformanceFilter> appliedChangesValidatedStep) {
        if (isEnabled()) {
            var appliedChanges = appliedChangesValidatedStep.getValidAppliedChanges();
            mapList(appliedChanges, c -> {
                c.modify(PerformanceFilter.CONDITIONS, addDefaultConditionIfSite(c.getNewValue(PerformanceFilter.CONDITIONS),
                        c.getNewValue(PerformanceFilter.SOURCE)));
                return c;
            });
        }
    }

    private boolean isEnabled() {
        return ppcPropertiesSupport.get(ADD_DEFAULT_SITE_FILTER_CONDITION_ENABLED).getOrDefault(false);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<PerformanceFilter>> applicableAppliedChanges) {
        Set<Long> adGroupIds = listToSet(applicableAppliedChanges, c -> c.getModel().getPid());
        List<AdGroup> adGroups = adGroupRepository.getAdGroups(shard, adGroupIds);
        Map<Long, AdGroup> adGroupById = listToMap(adGroups, AdGroup::getId);

        revertStrategyWarnProperties(applicableAppliedChanges);

        boolean isConditionsUpdateAllowed =
                featureService.isEnabledForClientId(clientId, FeatureName.UPDATE_FILTER_CONDITIONS_ALLOWED);
        PerformanceFilterPreUpdateLogic preUpdateLogic =
                getPreUpdateLogic(applicableAppliedChanges, adGroupById, isConditionsUpdateAllowed);
        //Изменяем статусы и время у групп объявлений и банеров
        preUpdateChanges(shard, clientId, adGroupById, preUpdateLogic);
        //Применяем изменения непосредственно к фильтрам
        List<Long> result = filtersUpdate(shard, applicableAppliedChanges, preUpdateLogic);
        //Логируем изменения цен
        logPriceChanges(applicableAppliedChanges, adGroupById);

        return result;
    }

    /**
     * Отменяет изменение полей которые не прошли валидацию на соответствие стратегии
     */
    private void revertStrategyWarnProperties(List<AppliedChanges<PerformanceFilter>> applicableAppliedChanges) {
        for (AppliedChanges<PerformanceFilter> appliedChanges : applicableAppliedChanges) {
            Long filterId = appliedChanges.getModel().getId();
            List<ModelProperty> modelProperties = strategyWarnPropsByFilterId.get(filterId);
            if (isEmpty(modelProperties)) {
                continue;
            }
            for (ModelProperty notChangeProperty : modelProperties) {
                if (Objects.equals(PerformanceFilter.PRICE_CPA, notChangeProperty)) {
                    revertOldValue(appliedChanges, PerformanceFilter.PRICE_CPA);
                } else if (Objects.equals(PerformanceFilter.PRICE_CPC, notChangeProperty)) {
                    revertOldValue(appliedChanges, PerformanceFilter.PRICE_CPC);
                } else if (Objects.equals(PerformanceFilter.AUTOBUDGET_PRIORITY, notChangeProperty)) {
                    revertOldValue(appliedChanges, PerformanceFilter.AUTOBUDGET_PRIORITY);
                }
            }
        }
    }

    private <T> void revertOldValue(AppliedChanges<PerformanceFilter> appliedChanges,
                                    ModelProperty<? super PerformanceFilter, T> modelProperty) {
        T oldValue = appliedChanges.getOldValue(modelProperty);
        appliedChanges.modify(modelProperty, oldValue);
    }

    private PerformanceFilterPreUpdateLogic getPreUpdateLogic(
            List<AppliedChanges<PerformanceFilter>> applicableAppliedChanges,
            Map<Long, AdGroup> adGroupById, boolean isConditionsUpdateAllowed) {
        PerformanceFilterPreUpdateLogic preUpdateLogic = new PerformanceFilterPreUpdateLogic();
        for (AppliedChanges<PerformanceFilter> appliedChanges : applicableAppliedChanges) {
            fillPreUpdateBusinessLogicSets(appliedChanges, preUpdateLogic, adGroupById, isConditionsUpdateAllowed);
        }
        return preUpdateLogic;
    }

    /**
     * В соответствии с бизнес-логикой изменения фильтров заполняет списки соотвествующих изменений идентификаторами
     * объектов
     * к которым они должны быть применены.
     */
    private void fillPreUpdateBusinessLogicSets(AppliedChanges<PerformanceFilter> appliedChanges,
                                                PerformanceFilterPreUpdateLogic preUpdateLogic,
                                                Map<Long, AdGroup> adGroupById,
                                                boolean isConditionsUpdateAllowed) {
        PerformanceFilter newFilter = appliedChanges.getModel();
        Long filterId = newFilter.getId();
        Long adGroupId = newFilter.getPid();

        boolean isConditionsChanged = appliedChanges.changed(PerformanceFilter.CONDITIONS);
        boolean isUpdateConditions = isConditionsChanged && isConditionsUpdateAllowed;
        boolean isRenewFilter = isConditionsChanged && !isConditionsUpdateAllowed;
        boolean isDeletedChanged = isConditionsChanged || appliedChanges.changed(PerformanceFilter.IS_DELETED);
        boolean isNameChanged = appliedChanges.changed(PerformanceFilter.NAME);
        boolean isPriceCpcChanged = appliedChanges.changed(PerformanceFilter.PRICE_CPC);
        boolean isPriceCpaChanged = appliedChanges.changed(PerformanceFilter.PRICE_CPA);
        boolean isAutobudgetPriorityChanged = appliedChanges.changed(PerformanceFilter.AUTOBUDGET_PRIORITY);
        boolean isTargetFunnelChanged = appliedChanges.changed(PerformanceFilter.TARGET_FUNNEL);
        boolean isIsSuspendedChanged = appliedChanges.changed(PerformanceFilter.IS_SUSPENDED);
        boolean adGroupNotDraft = isNotDraft(adGroupById.get(adGroupId));
        // DIRECT-92377 #1. Если у фильтра поменялся Json (поле  condition_json ), то создаётся новый фильтр,
        // а старый помечается как удалённый: полю  is_deleted присваивается значение  1 .
        // UPD:
        // DIRECT-116567 - новая логика, при изменении условий фильтр больше не пересоздаём, но пока только по фиче
        // #2 Так же, если у фильтра поменялся Json, то у всех связанных перформанс-групп с
        // phrases.statusModerate!=New , поле  adgroups_performance.statusBlGenerated выставляется в статус Processing .
        if (isRenewFilter) {
            preUpdateLogic.getFilterIdsForRenew().add(filterId);
        }
        if (isConditionsChanged) {
            preUpdateLogic.getAdGroupIdsForStatusBlGeneratedReset().add(adGroupId);
        }
        // DIRECT-92377 #3. Если у фильтра поменялось одно из полей:  name ,  price_cpc ,  price_cpa ,
        // autobudgetPriority ,  target_funnel ,  is_suspended или  is_deleted,
        // то ему обновляется время последнего изменения, поле  LastChange .
        if (isNameChanged || isPriceCpcChanged || isPriceCpaChanged || isAutobudgetPriorityChanged
                || isTargetFunnelChanged || isIsSuspendedChanged || isDeletedChanged || isUpdateConditions) {
            preUpdateLogic.getFilterIdsForLastChangeUpdate().add(filterId);
        }
        // DIRECT-92377 #4. Если у фильтра поменялось одно из полей:  target_funnel,  is_suspended или is_deleted,
        // то у связанных групп, с  phrases.statusModerate!=New, а так же у связанных с ними банеров сбрасывается
        // статус синхронизации: поле  banners.statusBsSynced выставляется в  No и поле  phrases.statusBsSynced
        // выставляется в  No.
        if (adGroupNotDraft
                && (isTargetFunnelChanged || isIsSuspendedChanged || isDeletedChanged || isUpdateConditions)) {
            preUpdateLogic.getAdGroupIdsForStatusBsSyncedReset().add(adGroupId);
        }
        // DIRECT-92377 #5. Если у фильтра поменялось одно из полей  filter_name ,  target_funnel ,  is_suspended
        // или  is_deleted , то у связанной с ним группы обновляется время последнего изменения: полю
        // phrases.LastChange присваивается текущее значение.
        if (isNameChanged || isTargetFunnelChanged || isIsSuspendedChanged || isDeletedChanged || isUpdateConditions) {
            preUpdateLogic.getAdGroupIdsForLastChangeUpdate().add(adGroupId);
        }
        // DIRECT-92377 #6. Если у фильтра поменялось одно и из полей:  price_cpc,  price_cpa или
        // autobudget_priority, то выставляем ему поле  statusBsSynced в статус No.
        if (isPriceCpcChanged || isPriceCpaChanged || isAutobudgetPriorityChanged) {
            preUpdateLogic.getFilterIdsForStatusBsSyncedReset().add(filterId);
        }
        //TODO : после закапывания фильтров в перле здесь должна появиться логика удаления табов удаляемых фильтров в
        // опциях кампаний.
        // Пока этого не делаем, потому что в переле фильтры гипотетически могут воскресать.
    }

    //Изменяет заданные поля групп объявлений и банеров при изменении фильтров
    private void preUpdateChanges(int shard,
                                  ClientId clientId,
                                  Map<Long, AdGroup> adGroupById,
                                  PerformanceFilterPreUpdateLogic preUpdateLogic) {
        Set<Long> allChangingAdGroupIds = StreamEx.of(preUpdateLogic.getAdGroupIdsForStatusBlGeneratedReset(),
                preUpdateLogic.getAdGroupIdsForStatusBsSyncedReset(), preUpdateLogic.getAdGroupIdsForLastChangeUpdate())
                .flatMap(StreamEx::of)
                .toSet();
        List<AppliedChanges<AdGroup>> adGroupAppliedChanges = StreamEx.of(allChangingAdGroupIds)
                .map(adGroupId -> {
                    PerformanceAdGroup adGroup = (PerformanceAdGroup) adGroupById.get(adGroupId);
                    ModelChanges<PerformanceAdGroup> changes =
                            new ModelChanges<>(adGroupId, PerformanceAdGroup.class);
                    if (preUpdateLogic.getAdGroupIdsForLastChangeUpdate().contains(adGroupId)) {
                        changes.process(preUpdateLogic.getLocalDateTime(), PerformanceAdGroup.LAST_CHANGE);
                    }
                    if (preUpdateLogic.getAdGroupIdsForStatusBlGeneratedReset().contains(adGroupId)) {
                        changes.process(StatusBLGenerated.PROCESSING, PerformanceAdGroup.STATUS_B_L_GENERATED);
                    }
                    if (preUpdateLogic.getAdGroupIdsForStatusBsSyncedReset().contains(adGroupId)) {
                        changes.process(StatusBsSynced.NO, PerformanceAdGroup.STATUS_BS_SYNCED);
                    }
                    return changes.applyTo(adGroup)
                            .castModelUp(AdGroup.class);
                })
                .toList();
        adGroupRepository.updateAdGroups(shard, clientId, adGroupAppliedChanges);
        bannerCommonRepository.updateStatusBsSyncedByAdgroupId(shard,
                preUpdateLogic.getAdGroupIdsForStatusBsSyncedReset(),
                StatusBsSynced.NO);
    }

    private List<Long> filtersUpdate(int shard, List<AppliedChanges<PerformanceFilter>> applicableAppliedChanges,
                                     PerformanceFilterPreUpdateLogic preUpdateLogic) {
        List<PerformanceFilter> srcRenewFilters =
                performanceFilterRepository.getFiltersById(shard, preUpdateLogic.getFilterIdsForRenew());
        Map<Long, PerformanceFilter> srcRenewFilterById = listToMap(srcRenewFilters, PerformanceFilter::getId);
        int size = applicableAppliedChanges.size();
        ArrayList<AppliedChanges<PerformanceFilter>> updateChanges = new ArrayList<>(size);
        ArrayList<Supplier<Long>> idSuppliers = new ArrayList<>(size);
        List<PerformanceFilter> renewFilters = new ArrayList<>();
        ArrayList<Long> result = new ArrayList<>(size);
        for (AppliedChanges<PerformanceFilter> changes : applicableAppliedChanges) {
            Long filterId = changes.getModel().getId();
            // Если фильтр в списке пересоздаваемых, то ему меняем только дату последнего изменения и флаг IS_DELETED
            // все остальные изменения полей применятся уже к новой копии фильтра.
            if (preUpdateLogic.getFilterIdsForRenew().contains(filterId)) {
                ModelChanges<PerformanceFilter> deleteFilterChanges =
                        new ModelChanges<>(filterId, PerformanceFilter.class)
                                .process(preUpdateLogic.getLocalDateTime(), PerformanceFilter.LAST_CHANGE)
                                .process(true, PerformanceFilter.IS_DELETED);
                PerformanceFilter srcRenewFilter = srcRenewFilterById.get(filterId);
                AppliedChanges<PerformanceFilter> deleteChanges = deleteFilterChanges.applyTo(srcRenewFilter);
                updateChanges.add(deleteChanges);
                PerformanceFilter renewFilter = changes.getModel()
                        .withLastChange(preUpdateLogic.getLocalDateTime())
                        .withStatusBsSynced(StatusBsSynced.NO);
                renewFilters.add(renewFilter);
                idSuppliers.add(renewFilter::getId);
                continue;
            }
            if (preUpdateLogic.getFilterIdsForLastChangeUpdate().contains(filterId)) {
                changes.modify(PerformanceFilter.LAST_CHANGE, preUpdateLogic.getLocalDateTime());
            }
            if (preUpdateLogic.getFilterIdsForStatusBsSyncedReset().contains(filterId)) {
                changes.modify(PerformanceFilter.STATUS_BS_SYNCED, StatusBsSynced.NO);
            }
            updateChanges.add(changes);
            idSuppliers.add(() -> filterId);
        }
        performanceFilterRepository.addPerformanceFilters(shard, renewFilters);
        performanceFilterRepository.update(shard, updateChanges);
        for (int i = 0; i < size; i++) {
            AppliedChanges<PerformanceFilter> changes = applicableAppliedChanges.get(i);
            Supplier<Long> idSupplier = idSuppliers.get(i);
            Long id = idSupplier.get();
            changes.modify(PerformanceFilter.ID, id);
            result.add(id);
        }
        return result;
    }

    private void logPriceChanges(List<AppliedChanges<PerformanceFilter>> applicableAppliedChanges,
                                 Map<Long, AdGroup> adGroupById) {
        List<PerformanceFilter> performanceFilters = StreamEx.of(applicableAppliedChanges)
                .filter(changes ->
                        changes.changed(PerformanceFilter.PRICE_CPC) || changes.changed(PerformanceFilter.PRICE_CPA))
                .map(AppliedChanges::getModel)
                .toList();

        Currency clientCurrency = clientService.getWorkCurrency(clientId);

        Function<PerformanceFilter, LogPriceData> performanceFilterToLogFn = filter -> new LogPriceData(
                adGroupById.get(filter.getPid()).getCampaignId(),
                filter.getPid(),
                filter.getId(),
                nvl(filter.getPriceCpa(), BigDecimal.ZERO).doubleValue(),
                nvl(filter.getPriceCpc(), BigDecimal.ZERO).doubleValue(),
                clientCurrency.getCode(),
                LogPriceData.OperationType.PERF_FILTER_UPDATE);

        List<LogPriceData> priceDataList = mapList(performanceFilters, performanceFilterToLogFn);
        logPriceService.logPrice(priceDataList, operatorUid);
    }
}
