package ru.yandex.direct.core.entity.performancefilter.utils;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.google.common.collect.ImmutableList;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.performancefilter.model.Operator;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilter;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatSerializer;

import static org.apache.commons.collections4.CollectionUtils.isEmpty;

public class PerformanceFilterUtils {

    public static final Comparator<PerformanceFilterCondition> PERFORMANCE_FILTER_CONDITION_COMPARATOR =
            Comparator.<PerformanceFilterCondition, String>comparing(PerformanceFilterCondition::getFieldName)
                    .thenComparing(PerformanceFilterCondition::getOperator)
                    .thenComparing(PerformanceFilterCondition::getStringValue);

    private PerformanceFilterUtils() {
    }

    private static String conditionsToString(PerformanceFilter filter) {
        List<PerformanceFilterCondition> conditions = filter.getConditions();
        if (isEmpty(conditions)) {
            return "";
        }
        return serializeSorted(conditions);
    }

    public static boolean validAndEqual(Object obj1, Object obj2) {
        //noinspection unchecked
        return validAndEqual((List<PerformanceFilterCondition>) obj1, (List<PerformanceFilterCondition>) obj2);
    }

    /**
     * Возвращает {@code true}, если переданные списки выглядят валидными и равны в сериализованном виде.
     * Если списки содержат дубликаты по паре (поле, оператор) и не могут быть сериализованы, вернётся {@code false}.
     */
    public static boolean validAndEqual(List<PerformanceFilterCondition> conditions1,
                                        List<PerformanceFilterCondition> conditions2) {
        if (conditions1 == null || conditions2 == null) {
            return false;
        }
        if (containsDuplicateFieldOperator(conditions1) || containsDuplicateFieldOperator(conditions2)) {
            // дубликаты -- невалидные значения. Мы их должны отбросить на валидации
            return false;
        }
        return isEqual(conditions1, conditions2);
    }

    /**
     * @return {@code true}, если переданные списки равны в сериализованнном виде.
     */
    public static boolean isEqual(
            List<? extends PerformanceFilterCondition> conditions1,
            List<? extends PerformanceFilterCondition> conditions2
    ) {
        String serialize1 = serializeSorted(conditions1);
        String serialize2 = serializeSorted(conditions2);
        return Objects.equals(serialize1, serialize2);
    }

    /**
     * @return {@code true}, если у фильтров совпадают {@code targetFunnel} и {@code conditions}.
     */
    public static boolean isEqual(PerformanceFilter pf1, PerformanceFilter pf2) {
        return pf1.getTargetFunnel() == pf2.getTargetFunnel() && isEqual(pf1.getConditions(), pf2.getConditions());
    }

    public static boolean haveEqualConditions(DynamicFeedAdTarget pf1, DynamicFeedAdTarget pf2) {
        return isEqual(pf1.getCondition(), pf2.getCondition());
    }

    /**
     * Определяем, есть ли дубликат по паре поле-оператор
     */
    public static boolean containsDuplicateFieldOperator(List<PerformanceFilterCondition> conditions) {
        Set<Pair<String, Operator>> uniqFields = new HashSet<>();
        for (PerformanceFilterCondition condition : conditions) {
            Pair<String, Operator> field = Pair.of(condition.getFieldName(), condition.getOperator());
            boolean added = uniqFields.add(field);
            if (!added) {
                return true;
            }
        }
        return false;
    }

    private static String serializeSorted(List<? extends PerformanceFilterCondition> conditions) {
        ArrayList<PerformanceFilterCondition> copy1 = new ArrayList<>(conditions);
        copy1.sort(PERFORMANCE_FILTER_CONDITION_COMPARATOR);
        return PerformanceFilterConditionDBFormatSerializer.INSTANCE.serialize(copy1);
    }

    public static List<PerformanceFilterCondition> addDefaultConditionIfSite(
            List<PerformanceFilterCondition> conditions, Source source) {
        if (source != Source.SITE || conditions.stream().anyMatch(c -> c.getFieldName().equals("available"))) {
            return conditions;
        }

        return new ImmutableList.Builder<PerformanceFilterCondition>()
                .addAll(conditions)
                .add(new PerformanceFilterCondition<Boolean>("available", Operator.EQUALS, "true")
                        .withParsedValue(true))
                .build();
    }
}
