package ru.yandex.direct.core.entity.performancefilter.validation;

import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Optional;

import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTargetTab;
import ru.yandex.direct.core.entity.feedfilter.model.FeedFilterTab;
import ru.yandex.direct.core.entity.performancefilter.model.Operator;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterCondition;
import ru.yandex.direct.core.entity.performancefilter.model.PerformanceFilterTab;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.schema.parser.AbstractPerformanceConditionValueParser;
import ru.yandex.direct.core.entity.performancefilter.schema.parser.NopeParser;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatSerializer;
import ru.yandex.direct.core.entity.performancefilter.utils.PerformanceFilterUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.banner.service.validation.BannerTextConstraints.charsAreAllowed;
import static ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterDefects.filterConditionsIsTooLong;
import static ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterDefects.invalidOperator;
import static ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterDefects.mustContainAnyMoreConditions;
import static ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterDefects.unknownField;
import static ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterDefects.unknownOperator;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.unconditional;

public class FilterConditionsValidator implements Validator<List<PerformanceFilterCondition>, Defect> {
    private static final int MAX_CONDITION_JSON_SIZE = 65535;

    private final FilterSchema filterSchema;
    private final boolean requireConditions;

    public FilterConditionsValidator(FilterSchema filterSchema,
                                     PerformanceFilterTab filterTab) {
        this.filterSchema = filterSchema;
        this.requireConditions = filterTab != PerformanceFilterTab.ALL_PRODUCTS;
    }

    public FilterConditionsValidator(FilterSchema filterSchema,
                                     DynamicAdTargetTab filterTab) {
        this.filterSchema = filterSchema;
        this.requireConditions = filterTab != DynamicAdTargetTab.ALL_PRODUCTS;
    }

    public FilterConditionsValidator(FilterSchema filterSchema,
                                     FeedFilterTab filterTab) {
        this.filterSchema = filterSchema;
        this.requireConditions = filterTab != FeedFilterTab.ALL_PRODUCTS;
    }

    @Override
    public ValidationResult<List<PerformanceFilterCondition>, Defect> apply(
            List<PerformanceFilterCondition> conditions) {
        ListValidationBuilder<PerformanceFilterCondition, Defect> lvb = ListValidationBuilder.of(conditions);
        lvb.check(notNull())
                // 5000, "Необходимо задать список правил для фильтра"
                .check(notEmptyCollection(), When.isTrue(requireConditions))
                .check(conditionListConstraint(), When.isTrue(requireConditions))
                .check(uniqFieldOperatorPairs())
                .checkBy(getConditionsJsonSizeValidator(), When.isValid())
                .check(filterSchema.validFiltersCount(), When.isValid())
                .checkEachBy(conditionValidator(filterSchema), When.isValid());
        return lvb.getResult();
    }

    private Constraint<List<PerformanceFilterCondition>, Defect> conditionListConstraint() {
        // Не разрешаем сохранение фильтра, если в списке условий только "available"
        return conditions -> conditions.size() == 1 && conditions.get(0).getFieldName().equals(FilterSchema.AVAILABLE)
                ? mustContainAnyMoreConditions()
                : null;
    }

    private Constraint<List<PerformanceFilterCondition>, Defect> uniqFieldOperatorPairs() {
        return Constraint
                .fromPredicate(conditions -> !PerformanceFilterUtils.containsDuplicateFieldOperator(conditions),
                        CollectionDefects.duplicatedElement());
    }

    private Validator<List<PerformanceFilterCondition>, Defect> getConditionsJsonSizeValidator() {
        return conditions ->
                Optional.ofNullable(conditions)
                        .map(PerformanceFilterConditionDBFormatSerializer.INSTANCE::serialize)
                        .map(strCond -> strCond.getBytes(StandardCharsets.UTF_8))
                        .map(bytes -> bytes.length)
                        .filter(size -> size > MAX_CONDITION_JSON_SIZE)
                        .map(size -> ValidationResult.failed(conditions, filterConditionsIsTooLong()))
                        .orElse(ValidationResult.success(conditions));
    }

    private Validator<PerformanceFilterCondition, Defect> conditionValidator(FilterSchema filterSchema) {
        return condition -> {
            ModelItemValidationBuilder<PerformanceFilterCondition> vb = ModelItemValidationBuilder.of(condition);

            // DIRECT-166949: Условие по умолчанию для фильтров ТК
            if (condition.getFieldName().equals("available") && condition.getOperator() == Operator.NOT_EQUALS &&
                    condition.getStringValue().equals("false")) {
                return ValidationResult.success(condition);
            }

            // DIRECT-161247: Условие по умолчанию для фильтров в смартах по сайту
            if (filterSchema.isSiteSource() && condition.getFieldName().equals("available") &&
                    condition.getOperator() == Operator.EQUALS && condition.getStringValue().equals("true")) {
                return ValidationResult.success(condition);
            }

            vb.item(PerformanceFilterCondition.FIELD_NAME)
                    .check(inSet(filterSchema.allFields()), unknownField());
            vb.item(PerformanceFilterCondition.STRING_VALUE)
                    .check(charsAreAllowed());
            if (vb.getResult().hasAnyErrors()) {
                return vb.getResult();
            }

            AbstractPerformanceConditionValueParser<?> parser =
                    filterSchema.getParser(condition.getFieldName(), condition.getOperator());

            vb.item(PerformanceFilterCondition.OPERATOR)
                    .check(Constraint.fromPredicate(o -> o != Operator.UNKNOWN, unknownOperator()))
                    .check(unconditional(invalidOperator()),
                            When.isValidAnd(When.isTrue(parser instanceof NopeParser)));
            if (vb.getResult().hasAnyErrors()) {
                return vb.getResult();
            }

            vb.checkBy(parser::validate);

            return vb.getResult();
        };
    }
}
