package ru.yandex.direct.core.entity.placements.repository;

import java.util.function.Function;

import javax.annotation.Nullable;

import ru.yandex.direct.core.entity.placements.model1.DbIndoorBlockData;
import ru.yandex.direct.core.entity.placements.model1.DbOutdoorBlockData;
import ru.yandex.direct.core.entity.placements.model1.DbPlacementBlock;
import ru.yandex.direct.core.entity.placements.model1.DbPlacementBlockData;
import ru.yandex.direct.core.entity.placements.model1.IndoorBlock;
import ru.yandex.direct.core.entity.placements.model1.OutdoorBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementType;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkArgument;

public class PlacementBlockConverter {

    private static final Function<DbPlacementBlock, PlacementBlock> baseFromInternalConverter =
            dbBlock -> {
                DbPlacementBlockData jsonData = JsonUtils.fromJson(dbBlock.getData(), DbPlacementBlockData.class);
                return new PlacementBlock(dbBlock.getPageId(), dbBlock.getBlockId(), dbBlock.getBlockCaption(),
                        dbBlock.getLastChange(), dbBlock.getIsDeleted(), jsonData.getSizes());
            };

    private static final Function<PlacementBlock, DbPlacementBlock> baseToInternalConverter =
            block -> {
                DbPlacementBlockData jsonData = new DbPlacementBlockData()
                        .withSizes(block.getSizes());
                return new DbPlacementBlock()
                        .withPageId(block.getPageId())
                        .withBlockId(block.getBlockId())
                        .withBlockCaption(block.getBlockCaption())
                        .withLastChange(block.getLastChange())
                        .withIsDeleted(block.isDeleted())
                        .withData(JsonUtils.toJson(jsonData));
            };

    private static final Function<DbPlacementBlock, OutdoorBlock> outdoorFromInternalModelConverter =
            dbBlock -> {
                DbOutdoorBlockData jsonData = JsonUtils.fromJson(dbBlock.getData(), DbOutdoorBlockData.class);
                return new OutdoorBlock(dbBlock.getPageId(), dbBlock.getBlockId(), dbBlock.getBlockCaption(),
                        dbBlock.getLastChange(), dbBlock.getIsDeleted(), jsonData.getSizes(), dbBlock.getGeoId(),
                        jsonData.getAddress(), jsonData.getAddressTranslations(), jsonData.getCoordinates(),
                        jsonData.getResolution(), dbBlock.getFacilityType(), jsonData.getDirection(),
                        jsonData.getWidth(), jsonData.getHeight(), jsonData.getDuration(), jsonData.getPhotos(),
                        jsonData.getHidden());
            };

    private static final Function<OutdoorBlock, DbPlacementBlock> outdoorToInternalConverter =
            block -> {
                DbOutdoorBlockData jsonData = new DbOutdoorBlockData()
                        .withSizes(block.getSizes())
                        .withAddress(block.getAddress())
                        .withAddressTranslations(block.getAddressTranslations())
                        .withCoordinates(block.getCoordinates())
                        .withResolution(block.getResolution())
                        .withDirection(block.getDirection())
                        .withWidth(block.getWidth())
                        .withHeight(block.getHeight())
                        .withDuration(block.getDuration())
                        .withPhotos(block.getPhotos())
                        .withHidden(block.getHidden());
                return new DbPlacementBlock()
                        .withPageId(block.getPageId())
                        .withBlockId(block.getBlockId())
                        .withBlockCaption(block.getBlockCaption())
                        .withLastChange(block.getLastChange())
                        .withIsDeleted(block.isDeleted())
                        .withGeoId(block.getGeoId())
                        .withFacilityType(block.getFacilityType())
                        .withData(JsonUtils.toJson(jsonData));
            };

    private static final Function<DbPlacementBlock, IndoorBlock> indoorFromInternalModelConverter =
            dbBlock -> {
                DbIndoorBlockData jsonData = JsonUtils.fromJson(dbBlock.getData(), DbIndoorBlockData.class);
                return new IndoorBlock(dbBlock.getPageId(), dbBlock.getBlockId(), dbBlock.getLastChange(),
                        dbBlock.getIsDeleted(), jsonData.getSizes(), dbBlock.getGeoId(),
                        jsonData.getAddress(), jsonData.getAddressTranslations(), jsonData.getCoordinates(),
                        jsonData.getResolution(), dbBlock.getFacilityType(), dbBlock.getZoneCategory(),
                        jsonData.getAspectRatio(), jsonData.getPhotos(), jsonData.getHidden());
            };

    private static final Function<IndoorBlock, DbPlacementBlock> indoorToInternalConverter =
            block -> {
                DbIndoorBlockData jsonData = new DbIndoorBlockData()
                        .withSizes(block.getSizes())
                        .withAddress(block.getAddress())
                        .withAddressTranslations(block.getAddressTranslations())
                        .withCoordinates(block.getCoordinates())
                        .withResolution(block.getResolution())
                        .withAspectRatio(block.getAspectRatio())
                        .withPhotos(block.getPhotos())
                        .withHidden(block.getHidden());
                return new DbPlacementBlock()
                        .withPageId(block.getPageId())
                        .withBlockId(block.getBlockId())
                        .withBlockCaption(block.getBlockCaption())
                        .withLastChange(block.getLastChange())
                        .withIsDeleted(block.isDeleted())
                        .withGeoId(block.getGeoId())
                        .withFacilityType(block.getFacilityType())
                        .withZoneCategory(block.getZoneCategory())
                        .withData(JsonUtils.toJson(jsonData));
            };

    static PlacementBlock fromInternalModel(@Nullable PlacementType placementType, DbPlacementBlock dbBlock) {
        if (placementType == null) {
            return baseFromInternalConverter.apply(dbBlock);
        }
        switch (placementType) {
            case OUTDOOR:
                return outdoorFromInternalModelConverter.apply(dbBlock);
            case INDOOR:
                return indoorFromInternalModelConverter.apply(dbBlock);
            default:
                throw new IllegalStateException("unsupported placement type: " + placementType);
        }
    }

    static DbPlacementBlock toInternalModel(@Nullable PlacementType placementType, PlacementBlock placementBlock) {
        if (placementType == null) {
            return baseToInternalConverter.apply(placementBlock);
        }
        switch (placementType) {
            case OUTDOOR: {
                checkArgument(placementBlock instanceof OutdoorBlock,
                        "block class doesn't correspond to placement type");
                return outdoorToInternalConverter.apply((OutdoorBlock) placementBlock);
            }
            case INDOOR: {
                checkArgument(placementBlock instanceof IndoorBlock,
                        "block class doesn't correspond to placement type");
                return indoorToInternalConverter.apply((IndoorBlock) placementBlock);
            }
            default:
                throw new IllegalStateException("unsupported placement type: " + placementType);
        }
    }
}
