package ru.yandex.direct.core.entity.placements.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.DeleteConditionStep;
import org.jooq.Record;
import org.jooq.SelectQuery;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.placements.model1.DbPlacementBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlockKey;
import ru.yandex.direct.core.entity.placements.model1.PlacementType;
import ru.yandex.direct.core.entity.placements.model1.PlacementsFilter;
import ru.yandex.direct.dbschema.ppcdict.tables.records.PlacementBlocksRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.i18n.Language;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.singletonList;
import static org.jooq.impl.DSL.field;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.core.entity.placements.model1.DbPlacementBlock.BLOCK_CAPTION;
import static ru.yandex.direct.core.entity.placements.model1.DbPlacementBlock.BLOCK_ID;
import static ru.yandex.direct.core.entity.placements.model1.DbPlacementBlock.DATA;
import static ru.yandex.direct.core.entity.placements.model1.DbPlacementBlock.FACILITY_TYPE;
import static ru.yandex.direct.core.entity.placements.model1.DbPlacementBlock.GEO_ID;
import static ru.yandex.direct.core.entity.placements.model1.DbPlacementBlock.IS_DELETED;
import static ru.yandex.direct.core.entity.placements.model1.DbPlacementBlock.LAST_CHANGE;
import static ru.yandex.direct.core.entity.placements.model1.DbPlacementBlock.PAGE_ID;
import static ru.yandex.direct.core.entity.placements.model1.DbPlacementBlock.ZONE_CATEGORY;
import static ru.yandex.direct.core.entity.placements.model1.GeoBlock.ADDRESS_TRANSLATIONS_PROPERTY;
import static ru.yandex.direct.dbschema.ppcdict.tables.PlacementBlocks.PLACEMENT_BLOCKS;
import static ru.yandex.direct.dbschema.ppcdict.tables.Placements.PLACEMENTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class PlacementBlockRepository {

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<DbPlacementBlock> mapper;

    @Autowired
    public PlacementBlockRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.mapper = createMapper();
    }

    public List<? extends PlacementBlock> getBlocksByFilter(PlacementsFilter placementsFilter) {
        List<DbPlacementBlock> blocks = getBlocksByFilterInner(placementsFilter);
        return blocks.stream()
                .map(dbBlock -> PlacementBlockConverter.fromInternalModel(placementsFilter.getPlacementType(), dbBlock))
                .collect(Collectors.toList());
    }

    private List<DbPlacementBlock> getBlocksByFilterInner(PlacementsFilter placementsFilter) {
        Condition condition = PLACEMENTS.PAGE_TYPE.eq(PlacementType.toSource(placementsFilter.getPlacementType()));
        if (placementsFilter.getPageIds() != null) {
            condition = condition.and(PLACEMENT_BLOCKS.PAGE_ID.in(placementsFilter.getPageIds()));
        }
        if (placementsFilter.getFacilityType() != null) {
            condition =
                    condition.and(PLACEMENT_BLOCKS.FACILITY_TYPE.eq(placementsFilter.getFacilityType().longValue()));
        }
        if (placementsFilter.getGeoId() != null) {
            condition = condition.and(PLACEMENT_BLOCKS.GEO_ID.eq(placementsFilter.getGeoId()));
        }
        if (placementsFilter.getZoneCategory() != null) {
            condition =
                    condition.and(PLACEMENT_BLOCKS.ZONE_CATEGORY.eq(placementsFilter.getZoneCategory().longValue()));
        }

        return dslContextProvider.ppcdict()
                .select(mapper.getFieldsToRead(DbPlacementBlock.allModelProperties()))
                .from(PLACEMENT_BLOCKS)
                .join(PLACEMENTS).on(PLACEMENTS.PAGE_ID.eq(PLACEMENT_BLOCKS.PAGE_ID))
                .where(condition)
                .fetch(mapper::fromDb);
    }

    public List<PlacementBlockKey> getPlacementBlockKeysUpdatedSince(LocalDateTime lastChangeFrom) {
        return dslContextProvider.ppcdict()
                .select(PLACEMENT_BLOCKS.PAGE_ID, PLACEMENT_BLOCKS.BLOCK_ID)
                .from(PLACEMENT_BLOCKS)
                .where(PLACEMENT_BLOCKS.LAST_CHANGE.greaterOrEqual(lastChangeFrom))
                .orderBy(PLACEMENT_BLOCKS.LAST_CHANGE)
                .fetch(rec -> PlacementBlockKey.of(rec.get(PLACEMENT_BLOCKS.PAGE_ID),
                        rec.get(PLACEMENT_BLOCKS.BLOCK_ID)));
    }

    public List<PlacementBlock> getPlacementBlocks(Collection<PlacementBlockKey> placementBlockKeys) {
        return getPlacementBlocks(dslContextProvider.ppcdict(), placementBlockKeys, false);
    }

    public List<PlacementBlock> getPlacementBlocks(DSLContext dslContext,
                                                   Collection<PlacementBlockKey> placementBlockKeys,
                                                   boolean forUpdate) {
        SelectQuery<Record> selectQuery = dslContext.selectQuery();
        selectQuery.addFrom(PLACEMENT_BLOCKS);
        selectQuery.addJoin(PLACEMENTS, PLACEMENTS.PAGE_ID.eq(PLACEMENT_BLOCKS.PAGE_ID));

        selectQuery.addSelect(PLACEMENTS.PAGE_TYPE);
        selectQuery.addSelect(mapper.getFieldsToRead());

        Condition condition = DSL.falseCondition();
        for (PlacementBlockKey placementBlockKey : placementBlockKeys) {
            Long pageId = placementBlockKey.getPageId();
            Long blockId = placementBlockKey.getBlockId();
            condition = condition.or(PLACEMENT_BLOCKS.PAGE_ID.eq(pageId)
                    .and(PLACEMENT_BLOCKS.BLOCK_ID.eq(blockId)));
        }

        selectQuery.addConditions(condition);

        selectQuery.setForUpdate(forUpdate);

        return selectQuery.fetch(rec -> {
            PlacementType pageType = PlacementType.fromSource(rec.get(PLACEMENTS.PAGE_TYPE));
            DbPlacementBlock dbBlock = mapper.fromDb(rec);
            return PlacementBlockConverter.fromInternalModel(pageType, dbBlock);
        });
    }

    Map<Long, List<PlacementBlock>> getBlocksByPageIds(Set<Long> pageIds,
                                                       Map<Long, PlacementType> pageIdToPageTypeMap) {
        Map<Long, List<DbPlacementBlock>> pageIdToDbBlocks =
                getBlocksByPageIdsInternal(pageIds, DbPlacementBlock.allModelProperties());
        return EntryStream.of(pageIdToDbBlocks)
                .mapToValue((pageId, dbBlocks) -> {
                    PlacementType pageType = pageIdToPageTypeMap.get(pageId);
                    return mapList(dbBlocks, dbBlock -> PlacementBlockConverter.fromInternalModel(pageType, dbBlock));
                })
                .toMap();
    }

    private Map<Long, List<DbPlacementBlock>> getBlocksByPageIdsInternal(Collection<Long> pageIds,
                                                                         Collection<ModelProperty<?, ?>> modelProperties) {
        return dslContextProvider.ppcdict()
                .select(PLACEMENT_BLOCKS.PAGE_ID)
                .select(mapper.getFieldsToRead(modelProperties))
                .from(PLACEMENT_BLOCKS)
                .where(PLACEMENT_BLOCKS.PAGE_ID.in(pageIds))
                .fetchGroups(PLACEMENT_BLOCKS.PAGE_ID, mapper::fromDb);
    }

    /**
     * Обновляет список блоков каждой переданной площадки целиком. Это значит,
     * что новые добавляются, существующие обновляются, а отсутствующие в запросе,
     * но существующие в базе - удаляются.
     */
    void createOrUpdatePlacementBlocks(Map<Long, PlacementType> pageIdToPageTypeMap,
                                       Map<Long, List<? extends PlacementBlock>> pageIdToBlocksMap) {
        Map<Long, List<DbPlacementBlock>> pageIdToDbBlocks = EntryStream.of(pageIdToBlocksMap)
                .mapToValue((pageId, blocks) -> {
                    PlacementType pageType = pageIdToPageTypeMap.get(pageId);
                    return mapList(blocks, block -> PlacementBlockConverter.toInternalModel(pageType, block));
                })
                .toMap();
        createOrUpdatePlacementBlocksInternal(pageIdToDbBlocks);
        deleteBlocksAbsentInRequest(pageIdToDbBlocks);
    }


    /**
     * Добавляет/обновляет данные блоков.
     */
    private void createOrUpdatePlacementBlocksInternal(Map<Long, List<DbPlacementBlock>> pageIdToBlocksMap) {
        InsertHelper<PlacementBlocksRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppcdict(), PLACEMENT_BLOCKS);
        pageIdToBlocksMap.forEach((pageId, dbBlocks) -> insertHelper.addAll(mapper, dbBlocks));

        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(PLACEMENT_BLOCKS.BLOCK_CAPTION, MySQLDSL.values(PLACEMENT_BLOCKS.BLOCK_CAPTION))
                    .set(PLACEMENT_BLOCKS.LAST_CHANGE, MySQLDSL.values(PLACEMENT_BLOCKS.LAST_CHANGE))
                    .set(PLACEMENT_BLOCKS.IS_DELETED, MySQLDSL.values(PLACEMENT_BLOCKS.IS_DELETED))
                    .set(PLACEMENT_BLOCKS.FACILITY_TYPE, MySQLDSL.values(PLACEMENT_BLOCKS.FACILITY_TYPE))
                    .set(PLACEMENT_BLOCKS.ZONE_CATEGORY, MySQLDSL.values(PLACEMENT_BLOCKS.ZONE_CATEGORY))
                    .set(PLACEMENT_BLOCKS.DATA, MySQLDSL.values(PLACEMENT_BLOCKS.DATA));
        }

        insertHelper.executeIfRecordsAdded();
    }

    /**
     * Удаляет все существующие блоки у пейджей, которые не пришли на обновление.
     */
    private void deleteBlocksAbsentInRequest(Map<Long, List<DbPlacementBlock>> pageIdToUpdatedBlocksMap) {
        Map<Long, List<DbPlacementBlock>> pageIdToAllBlocksMap =
                getBlocksByPageIdsInternal(pageIdToUpdatedBlocksMap.keySet(), singletonList(BLOCK_ID));

        Map<Long, ? extends Set<Long>> pageIdToDeletedBlockIdsMap = EntryStream.of(pageIdToAllBlocksMap)
                .mapToValue((pageId, allBlocks) -> {
                    Set<Long> allBlockIds = listToSet(allBlocks, DbPlacementBlock::getBlockId);
                    Set<Long> updatedBlockIds = listToSet(pageIdToUpdatedBlocksMap.get(pageId),
                            DbPlacementBlock::getBlockId);
                    return Sets.difference(allBlockIds, updatedBlockIds);
                })
                .toMap();

        DeleteConditionStep<PlacementBlocksRecord> step = dslContextProvider.ppcdict()
                .deleteFrom(PLACEMENT_BLOCKS)
                .where(DSL.falseCondition());
        pageIdToDeletedBlockIdsMap.forEach((pageId, deletedBlockIds) ->
                step.or(PLACEMENT_BLOCKS.PAGE_ID.eq(pageId).and(PLACEMENT_BLOCKS.BLOCK_ID.in(deletedBlockIds))));
        step.execute();
    }

    /**
     * Метод для джобы выставления региона новым indoor/outdoor-блокам.
     * Из-за того что id составной, невозможно использовать "case-when" /;
     * поэтому делаем много апдейтов. Кажется, для джобы это не критично.
     */
    public void updateGeoIds(DSLContext dslContext, Map<PlacementBlockKey, Long> newGeoIds) {
        newGeoIds.forEach((key, newGeo) ->
                dslContext.update(PLACEMENT_BLOCKS)
                        .set(PLACEMENT_BLOCKS.GEO_ID, newGeo)
                        .where(PLACEMENT_BLOCKS.PAGE_ID.eq(key.getPageId()))
                        .and(PLACEMENT_BLOCKS.BLOCK_ID.eq(key.getBlockId()))
                        .execute());
    }

    /**
     * Метод для джобы выставления локализованных адресов новым indoor/outdoor-блокам.
     * Из-за того что id составной, невозможно использовать "case-when" /;
     * поэтому делаем много апдейтов. Кажется, для джобы это не критично.
     */
    public void updateAddressTranslations(DSLContext dslContext,
                                          Map<PlacementBlockKey, Map<Language, String>> newAddressTranslations) {
        newAddressTranslations.forEach((key, addressTranslations) ->
                dslContext.update(PLACEMENT_BLOCKS)
                        .set(PLACEMENT_BLOCKS.DATA, field("json_set({0}, {1}, CAST({2} as JSON))", String.class,
                                PLACEMENT_BLOCKS.DATA, "$." + ADDRESS_TRANSLATIONS_PROPERTY,
                                JsonUtils.toJson(addressTranslations)))
                        .where(PLACEMENT_BLOCKS.PAGE_ID.eq(key.getPageId()))
                        .and(PLACEMENT_BLOCKS.BLOCK_ID.eq(key.getBlockId()))
                        .execute());
    }

    private JooqMapperWithSupplier<DbPlacementBlock> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(DbPlacementBlock::new)
                .map(property(PAGE_ID, PLACEMENT_BLOCKS.PAGE_ID))
                .map(property(BLOCK_ID, PLACEMENT_BLOCKS.BLOCK_ID))
                .map(property(BLOCK_CAPTION, PLACEMENT_BLOCKS.BLOCK_CAPTION))
                .map(property(LAST_CHANGE, PLACEMENT_BLOCKS.LAST_CHANGE))
                .map(booleanProperty(IS_DELETED, PLACEMENT_BLOCKS.IS_DELETED))
                .map(property(GEO_ID, PLACEMENT_BLOCKS.GEO_ID))
                .map(integerProperty(FACILITY_TYPE, PLACEMENT_BLOCKS.FACILITY_TYPE))
                .map(integerProperty(ZONE_CATEGORY, PLACEMENT_BLOCKS.ZONE_CATEGORY))
                .map(property(DATA, PLACEMENT_BLOCKS.DATA))
                .build();
    }
}
