package ru.yandex.direct.core.entity.placements.repository;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.placements.model1.BlockSize;
import ru.yandex.direct.core.entity.placements.model1.DbPlacement;
import ru.yandex.direct.core.entity.placements.model1.DbPlacementWithMirrors;
import ru.yandex.direct.core.entity.placements.model1.IndoorBlock;
import ru.yandex.direct.core.entity.placements.model1.IndoorPlacement;
import ru.yandex.direct.core.entity.placements.model1.OutdoorBlock;
import ru.yandex.direct.core.entity.placements.model1.OutdoorPlacement;
import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementType;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class PlacementConverter {

    static Placement fromInternalModel(DbPlacementWithMirrors dbPlacement, List<PlacementBlock> placementBlocks) {
        if (placementBlocks == null) {
            placementBlocks = new ArrayList<>();
        }
        if (dbPlacement.getType() == null) {
            return baseFromInternalModel(dbPlacement, placementBlocks);
        }
        switch (dbPlacement.getType()) {
            case OUTDOOR:
                return outdoorFromInternalModel(dbPlacement, placementBlocks);
            case INDOOR:
                return indoorFromInternalModel(dbPlacement, placementBlocks);
            default:
                throw new IllegalStateException("unsupported placement type: " + dbPlacement.getType());
        }
    }

    private static OutdoorPlacement outdoorFromInternalModel(DbPlacement dbPlacement,
                                                             List<PlacementBlock> placementBlocks) {
        checkArgument(dbPlacement.getType() == PlacementType.OUTDOOR,
                "can't create OutdoorPlacement with type %", dbPlacement.getType());
        Set<String> otherBlockTypes = getBlockTypesExcept(placementBlocks, OutdoorBlock.class);
        checkArgument(otherBlockTypes.isEmpty(),
                "can't create OutdoorPlacement with blocks of types: %s", otherBlockTypes);
        List<OutdoorBlock> outdoorBlocks = mapList(placementBlocks, block -> (OutdoorBlock) block);
        return new OutdoorPlacement(dbPlacement.getId(), dbPlacement.getDomain(), dbPlacement.getCaption(),
                dbPlacement.getLogin(), dbPlacement.getOperatorName(),
                dbPlacement.getIsYandexPage(), dbPlacement.getIsDeleted(), dbPlacement.getIsTesting(), outdoorBlocks,
                List.of());
    }

    private static IndoorPlacement indoorFromInternalModel(DbPlacement dbPlacement,
                                                           List<PlacementBlock> placementBlocks) {
        checkArgument(dbPlacement.getType() == PlacementType.INDOOR,
                "can't create IndoorPlacement with type %", dbPlacement.getType());
        Set<String> otherBlockTypes = getBlockTypesExcept(placementBlocks, IndoorBlock.class);
        checkArgument(otherBlockTypes.isEmpty(),
                "can't create IndoorPlacement with blocks of types: %s", otherBlockTypes);
        List<IndoorBlock> indoorBlocks = mapList(placementBlocks, block -> (IndoorBlock) block);
        return new IndoorPlacement(dbPlacement.getId(), dbPlacement.getDomain(), dbPlacement.getCaption(),
                dbPlacement.getLogin(), dbPlacement.getOperatorName(),
                dbPlacement.getIsYandexPage(), dbPlacement.getIsDeleted(), dbPlacement.getIsTesting(), indoorBlocks,
                List.of());
    }

    private static Placement baseFromInternalModel(DbPlacementWithMirrors dbPlacement, List<PlacementBlock> placementBlocks) {
        return new Placement<>(dbPlacement.getId(), dbPlacement.getType(), dbPlacement.getDomain(),
                dbPlacement.getCaption(), dbPlacement.getLogin(), dbPlacement.getOperatorName(),
                dbPlacement.getIsYandexPage(),
                dbPlacement.getIsDeleted(),
                dbPlacement.getIsTesting(),
                placementBlocks,
                dbPlacement.getMirrorDomains());
    }

    private static Set<String> getBlockTypesExcept(List<PlacementBlock> placementBlocks,
                                                   Class<? extends PlacementBlock> exceptClass) {
        return StreamEx.of(placementBlocks)
                .map(PlacementBlock::getClass)
                .remove(exceptClass::isAssignableFrom)
                .map(Class::getSimpleName)
                .toSet();
    }

    static DbPlacementWithMirrors toInternalModel(Placement<?> placement) {
        // пишем размеры блоков в ppcdict.placements.blocks для временной обратной совместимости
        List<? extends PlacementBlock> blocks = placement.getBlocks();
        Map<Long, List<String>> blockSizesMap = new HashMap<>();
        blocks.forEach(block -> {
            List<BlockSize> blockSizeList = block.getSizes();
            List<String> blockSizesAsString = mapList(blockSizeList,
                    blockSize -> blockSize.getWidth() + "x" + blockSize.getHeight());
            blockSizesMap.put(block.getBlockId(), blockSizesAsString);
        });
        return new DbPlacementWithMirrors()
                .withId(placement.getId())
                .withType(placement.getType())
                .withDomain(placement.getDomain())
                .withCaption(placement.getCaption())
                .withLogin(placement.getLogin())
                .withIsYandexPage(placement.isYandexPage())
                .withIsDeleted(placement.isDeleted())
                .withIsTesting(placement.isTesting())
                .withMirrorDomains(placement.getMirrors())
                .withBlocks(JsonUtils.toJson(blockSizesMap));
    }
}
