package ru.yandex.direct.core.entity.placements.repository;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Configuration;
import org.jooq.Record;
import org.jooq.SelectJoinStep;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.outdoor.model.PlacementsOutdoorData;
import ru.yandex.direct.core.entity.outdoor.repository.PlacementsOutdoorDataRepository;
import ru.yandex.direct.core.entity.placements.model1.DbPlacement;
import ru.yandex.direct.core.entity.placements.model1.DbPlacementWithMirrors;
import ru.yandex.direct.core.entity.placements.model1.DbPlacementsMirrors;
import ru.yandex.direct.core.entity.placements.model1.OutdoorPlacement;
import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.placements.model1.PlacementType;
import ru.yandex.direct.dbschema.ppcdict.enums.PlacementsPageType;
import ru.yandex.direct.dbschema.ppcdict.tables.records.PlacementsMirrorsRecord;
import ru.yandex.direct.dbschema.ppcdict.tables.records.PlacementsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapper.read.ReaderBuilders;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.model.ModelProperty;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.dbschema.ppcdict.tables.OutdoorOperators.OUTDOOR_OPERATORS;
import static ru.yandex.direct.dbschema.ppcdict.tables.Placements.PLACEMENTS;
import static ru.yandex.direct.dbschema.ppcdict.tables.PlacementsMirrors.PLACEMENTS_MIRRORS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Repository
public class PlacementRepository {
    private static final Logger logger = LoggerFactory.getLogger(PlacementRepository.class);

    private final DslContextProvider dslContextProvider;
    private final PlacementBlockRepository blockRepository;
    private final PlacementsOutdoorDataRepository outdoorDataRepository;
    private final JooqMapperWithSupplier<DbPlacement> mapper;
    private final JooqMapperWithSupplier<DbPlacementsMirrors> placementsMirrorsMapper;
    private final JooqMapperWithSupplier<DbPlacementWithMirrors> withMirrorsMapper;

    @Autowired
    public PlacementRepository(DslContextProvider dslContextProvider,
                               PlacementBlockRepository blockRepository,
                               PlacementsOutdoorDataRepository outdoorDataRepository) {
        this.dslContextProvider = dslContextProvider;
        this.blockRepository = blockRepository;
        this.outdoorDataRepository = outdoorDataRepository;
        this.mapper = createMapper();
        this.placementsMirrorsMapper = createPlacementsMirrorsMapper();
        this.withMirrorsMapper = createPlacementWithMirrorsMapper();
    }

    public boolean isOutdoorPlacementExists(Long pageId) {
        return dslContextProvider.ppcdict().fetchExists(
                dslContextProvider.ppcdict()
                        .selectOne()
                        .from(PLACEMENTS)
                        .where(PLACEMENTS.PAGE_TYPE.eq(PlacementsPageType.outdoor).and(PLACEMENTS.PAGE_ID.eq(pageId))));
    }

    public Map<Long, Placement> getPlacements(Collection<Long> pageIds) {
        Map<Long, DbPlacementWithMirrors> pageIdToDbPlacementMap =
                getPlacementsInternal(pageIds, DbPlacementWithMirrors.allModelProperties());

        return enrichPlacementsWithBlocks(pageIdToDbPlacementMap);
    }

    public Map<Long, Set<String>> getPlacementMirrors(Collection<Long> pageIds) {
        return dslContextProvider.ppcdict()
                .select(PLACEMENTS_MIRRORS.PAGE_ID, PLACEMENTS_MIRRORS.MIRROR_DOMAIN)
                .from(PLACEMENTS_MIRRORS)
                .where(PLACEMENTS_MIRRORS.PAGE_ID.in(pageIds))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(e -> e.get(PLACEMENTS_MIRRORS.PAGE_ID),
                        mapping(e -> e.get(PLACEMENTS_MIRRORS.MIRROR_DOMAIN), toSet())));
    }

    private Map<Long, DbPlacementWithMirrors> getPlacementsInternal(Collection<Long> pageIds,
                                                                    Collection<ModelProperty<?, ?>> props) {
        SelectJoinStep<Record> selectStep = dslContextProvider.ppcdict()
                .select(PLACEMENTS.PAGE_ID)
                .select(withMirrorsMapper.getFieldsToRead(props))
                .from(PLACEMENTS)
                .leftJoin(OUTDOOR_OPERATORS)
                .on(OUTDOOR_OPERATORS.LOGIN.eq(PLACEMENTS.OWNER_LOGIN));

        if (props.contains(DbPlacementWithMirrors.MIRROR_DOMAINS)) {
            selectStep = selectStep.leftJoin(PLACEMENTS_MIRRORS).using(PLACEMENTS.PAGE_ID);
        }
        return selectStep.where(PLACEMENTS.PAGE_ID.in(pageIds))
                .groupBy(PLACEMENTS.PAGE_ID)
                .fetchMap(PLACEMENTS.PAGE_ID, withMirrorsMapper::fromDb);

    }

    private Map<Long, Placement> enrichPlacementsWithBlocks(Map<Long, DbPlacementWithMirrors> pageIdToDbPlacementMap) {
        Map<Long, PlacementType> pageIdToPageTypeMap = EntryStream.of(pageIdToDbPlacementMap)
                .mapValues(DbPlacement::getType)
                .filterValues(Objects::nonNull)
                .toMap();

        Map<Long, List<PlacementBlock>> pageIdToBlocksMap =
                blockRepository.getBlocksByPageIds(pageIdToDbPlacementMap.keySet(), pageIdToPageTypeMap);

        return EntryStream.of(pageIdToDbPlacementMap)
                .mapToValue((pageId, dbPlacement) ->
                        PlacementConverter.fromInternalModel(dbPlacement, pageIdToBlocksMap.get(pageId)))
                .toMap();
    }

    public Map<Long, Placement> getPlacementsByType(PlacementType type) {
        checkArgument(type != null, "type must be defined");
        Map<Long, DbPlacementWithMirrors> pageIdToDbPlacementMap = getPlacementsByTypeInternal(type);
        return enrichPlacementsWithBlocks(pageIdToDbPlacementMap);
    }

    public List<Placement> getPlacementsByIds(List<Long> ids) {
        return StreamEx.of(getPlacementsInternal(ids, DbPlacementWithMirrors.allModelProperties()).values())
                .sorted(Comparator.comparing(DbPlacement::getDomain))
                .map(it -> PlacementConverter.fromInternalModel(it, null))
                .toList();
    }

    public List<Placement> findPlacementsByDomain(String filterString, int limit) {
        List<Long> pageIds = dslContextProvider.ppcdict()
                .selectDistinct(PLACEMENTS.PAGE_ID)
                .from(PLACEMENTS)
                .where(PLACEMENTS.IS_DELETED.eq(0L))
                .and(PLACEMENTS.DOMAIN.startsWith(filterString))
                .unionAll(
                        DSL.selectDistinct(PLACEMENTS.PAGE_ID)
                                .from(PLACEMENTS)
                                .join(PLACEMENTS_MIRRORS)
                                .using(PLACEMENTS.PAGE_ID)
                                .where(PLACEMENTS.IS_DELETED.eq(0L))
                                .and(PLACEMENTS_MIRRORS.MIRROR_DOMAIN.startsWith(filterString))
                )
                .limit(limit)
                .fetch(PLACEMENTS.PAGE_ID);

        return StreamEx.of(getPlacementsInternal(pageIds, DbPlacementWithMirrors.allModelProperties()).values())
                .sorted(Comparator.comparing(DbPlacement::getDomain))
                .map(it -> PlacementConverter.fromInternalModel(it, null))
                .toList();
    }

    private Map<Long, DbPlacementWithMirrors> getPlacementsByTypeInternal(PlacementType type) {
        return dslContextProvider.ppcdict()
                .select(withMirrorsMapper.getFieldsToRead())
                .from(PLACEMENTS)
                .leftJoin(OUTDOOR_OPERATORS)
                .on(OUTDOOR_OPERATORS.LOGIN.eq(PLACEMENTS.OWNER_LOGIN))
                .leftJoin(PLACEMENTS_MIRRORS).using(PLACEMENTS.PAGE_ID)
                .where(PLACEMENTS.PAGE_TYPE.eq(PlacementType.toSource(type)))
                .groupBy(PLACEMENTS.PAGE_ID)
                .fetchMap(PLACEMENTS.PAGE_ID, withMirrorsMapper::fromDb);
    }

    /**
     * (!) Не предназначен для конкурентного выполнения.
     * Не должен вызываться параллельно ни в рамках одного приложения, ни в разных,
     * так как синхронизация отсутствует.
     */
    public void createOrUpdatePlacements(Collection<Placement> placements) {
        checkPlacementTypesWereNotChanged(placements);

        List<DbPlacementWithMirrors> dbPlacements = new ArrayList<>(placements.size());
        Map<Long, PlacementType> pageIdToPageTypeMap = new HashMap<>(placements.size());
        Map<Long, List<? extends PlacementBlock>> pageIdToBlocksMap = new HashMap<>(placements.size());

        List<OutdoorPlacement> outdoorPlacements = StreamEx.of(placements)
                .select(OutdoorPlacement.class)
                .toList();
        placements.forEach(placement -> {
            dbPlacements.add(PlacementConverter.toInternalModel(placement));
            pageIdToPageTypeMap.put(placement.getId(), placement.getType());
            //noinspection unchecked
            pageIdToBlocksMap.put(placement.getId(), new ArrayList<>(placement.getBlocks()));
        });

        updateOutdoorPlacementsData(outdoorPlacements);
        createOrUpdatePlacementsInternal(dbPlacements);
        blockRepository.createOrUpdatePlacementBlocks(pageIdToPageTypeMap, pageIdToBlocksMap);
    }


    /**
     * Заполнение internal_name для новых пейджей
     *
     * @param placements список outdoor пейджей
     */
    private void updateOutdoorPlacementsData(Collection<OutdoorPlacement> placements) {
        if (placements.isEmpty()) {
            return;
        }

        var placementIdToInternalName = outdoorDataRepository.getPlacementIdToInternalName(listToSet(placements,
                Placement::getId));
        var loginToInternalName = outdoorDataRepository.getLoginToInternalNames(listToSet(
                filterList(placements, p -> placementIdToInternalName.get(p.getId()) == null),
                Placement::getLogin));
        List<PlacementsOutdoorData> outdoorDataList = new ArrayList<>();

        placements.forEach(placement -> {
            //если internal_name присутствует в базе, то ничего не делаем
            if (placementIdToInternalName.containsKey(placement.getId())) {
                return;
            }
            //получаем все internal_name, которые привязаны к логину, если привязано больше одного или ноль, то пишем
            // ошибку
            var internalNames = loginToInternalName.get(placement.getLogin());
            if (internalNames == null || internalNames.size() != 1) {
                logger.warn("unable to guess internal_name for page {}, add it manually", placement.getId());
                return;
            }

            outdoorDataList.add(new PlacementsOutdoorData()
                    .withPageId(placement.getId())
                    .withInternalName(internalNames.get(0))
            );
        });
        outdoorDataRepository.addOrUpdate(outdoorDataList);
    }

    /**
     * Проверяет, что у пришедших на обновление пейджей не случилось непредвиденного
     * изменения типа по сравнению с сохраненным в базу ранее.
     * Допустимо изменение неизвестного типа (null) на известный.
     *
     * @param placements список пейджей на обновление.
     */
    private void checkPlacementTypesWereNotChanged(Collection<Placement> placements) {
        Set<Long> requestPageIds = listToSet(placements, Placement::getId);
        Map<Long, DbPlacementWithMirrors> existingPlacements =
                getPlacementsInternal(requestPageIds, ImmutableSet.of(DbPlacement.TYPE));

        Map<Long, PlacementType> requestNonNullPlacementTypes = StreamEx.of(placements)
                .mapToEntry(Placement::getId, Placement::getType)
                .filterValues(Objects::nonNull)
                .toMap();
        Map<Long, PlacementType> existingNonNullPlacementTypes = EntryStream.of(existingPlacements)
                .mapValues(DbPlacement::getType)
                .filterValues(Objects::nonNull)
                .toMap();

        Map<Long, PlacementType> invalidTypesInRequest = EntryStream.of(existingNonNullPlacementTypes)
                .filterKeyValue((pageId, placementType) -> placementType != requestNonNullPlacementTypes.get(pageId))
                .toMap();

        if (!invalidTypesInRequest.isEmpty()) {
            StringBuilder msgBuilder = new StringBuilder().append("Next partner pages changed their types: \n");
            invalidTypesInRequest.forEach((pageId, placementType) ->
                    msgBuilder.append(pageId).append(" -> ").append(placementType).append(" \n"));
            throw new IllegalArgumentException(msgBuilder.toString());
        }
    }

    private void createOrUpdateMirrorsInTransaction(Collection<DbPlacementWithMirrors> placementWithMirrors) {
        Map<Long, List<String>> pageIdToMirrors =
                placementWithMirrors.stream().collect(Collectors.toMap(DbPlacementWithMirrors::getId,
                        DbPlacementWithMirrors::getMirrorDomains));

        dslContextProvider.ppcdict().transaction(
                configuration -> createOrUpdateMirrors(configuration, pageIdToMirrors)
        );
    }

    private void createOrUpdateMirrors(Configuration configuration, Map<Long, List<String>> pageIdToMirrors) {
        if (pageIdToMirrors.size() == 0) {
            return;
        }

        configuration.dsl().deleteFrom(PLACEMENTS_MIRRORS).where(
                PLACEMENTS_MIRRORS.PAGE_ID.in(pageIdToMirrors.keySet())
        ).execute();

        InsertHelper<PlacementsMirrorsRecord> insertHelper =
                new InsertHelper<>(configuration.dsl(), PLACEMENTS_MIRRORS);

        List<DbPlacementsMirrors> dbPlacementsMirrors = new ArrayList<>();

        for (Long pageId : pageIdToMirrors.keySet()) {
            for (String domain : pageIdToMirrors.get(pageId)) {

                if (domain == null || domain.isBlank()) {
                    continue;
                }

                dbPlacementsMirrors.add(new DbPlacementsMirrors().withPageId(pageId).withMirrorDomain(domain));
            }
        }

        if (!dbPlacementsMirrors.isEmpty()) {
            insertHelper.addAll(placementsMirrorsMapper, dbPlacementsMirrors);
            insertHelper.execute();
        }
    }

    private void createOrUpdatePlacementsInternal(Collection<DbPlacementWithMirrors> dbPlacements) {
        if (isEmpty(dbPlacements)) {
            return;
        }

        InsertHelper<PlacementsRecord> insertHelper = new InsertHelper<>(dslContextProvider.ppcdict(), PLACEMENTS);
        insertHelper.addAll(mapper, dbPlacements);

        insertHelper.onDuplicateKeyUpdate()
                .set(PLACEMENTS.PAGE_TYPE, MySQLDSL.values(PLACEMENTS.PAGE_TYPE))
                .set(PLACEMENTS.DOMAIN, MySQLDSL.values(PLACEMENTS.DOMAIN))
                .set(PLACEMENTS.CAPTION, MySQLDSL.values(PLACEMENTS.CAPTION))
                .set(PLACEMENTS.OWNER_LOGIN, MySQLDSL.values(PLACEMENTS.OWNER_LOGIN))
                .set(PLACEMENTS.BLOCKS, MySQLDSL.values(PLACEMENTS.BLOCKS))
                .set(PLACEMENTS.IS_YANDEX_PAGE, MySQLDSL.values(PLACEMENTS.IS_YANDEX_PAGE))
                .set(PLACEMENTS.IS_DELETED, MySQLDSL.values(PLACEMENTS.IS_DELETED))
                .set(PLACEMENTS.IS_TESTING, MySQLDSL.values(PLACEMENTS.IS_TESTING));

        insertHelper.execute();

        createOrUpdateMirrorsInTransaction(dbPlacements);
    }

    private <T extends DbPlacement> JooqMapperWithSupplierBuilder<T> dbPlacementMapperBuilder(Supplier<T> modelSupplier) {
        return JooqMapperWithSupplierBuilder.builder(modelSupplier)
                .map(property(DbPlacement.ID, PLACEMENTS.PAGE_ID))
                .map(convertibleProperty(DbPlacement.TYPE, PLACEMENTS.PAGE_TYPE,
                        PlacementType::fromSource, PlacementType::toSource))
                .map(property(DbPlacement.DOMAIN, PLACEMENTS.DOMAIN))
                .map(property(DbPlacement.CAPTION, PLACEMENTS.CAPTION))
                .map(property(DbPlacement.LOGIN, PLACEMENTS.OWNER_LOGIN))
                .map(property(DbPlacement.OPERATOR_NAME, OUTDOOR_OPERATORS.NAME))
                .map(property(DbPlacement.BLOCKS, PLACEMENTS.BLOCKS))
                .map(booleanProperty(DbPlacement.IS_YANDEX_PAGE, PLACEMENTS.IS_YANDEX_PAGE))
                .map(booleanProperty(DbPlacement.IS_DELETED, PLACEMENTS.IS_DELETED))
                .map(booleanProperty(DbPlacement.IS_TESTING, PLACEMENTS.IS_TESTING));
    }

    private JooqMapperWithSupplier<DbPlacement> createMapper() {
        return dbPlacementMapperBuilder(DbPlacement::new).build();
    }

    private JooqMapperWithSupplier<DbPlacementWithMirrors> createPlacementWithMirrorsMapper() {
        return dbPlacementMapperBuilder(DbPlacementWithMirrors::new)
                .readProperty(DbPlacementWithMirrors.MIRROR_DOMAINS,
                        ReaderBuilders.fromField(DSL.groupConcat(PLACEMENTS_MIRRORS.MIRROR_DOMAIN).separator(",").as(
                                "mirror_domains"))
                                .by(e -> {
                                    String nonNullMirrorsString = nvl(e, "");

                                    if (nonNullMirrorsString.isBlank()) {
                                        return Collections.emptyList();
                                    }

                                    return Arrays.asList(nonNullMirrorsString.split("\\s*,\\s*"));
                                })
                )
                .build();
    }

    private JooqMapperWithSupplier<DbPlacementsMirrors> createPlacementsMirrorsMapper() {
        return JooqMapperWithSupplierBuilder.builder(DbPlacementsMirrors::new)
                .map(property(DbPlacementsMirrors.ID, PLACEMENTS_MIRRORS.ID))
                .map(property(DbPlacementsMirrors.PAGE_ID, PLACEMENTS_MIRRORS.PAGE_ID))
                .map(property(DbPlacementsMirrors.MIRROR_DOMAIN, PLACEMENTS_MIRRORS.MIRROR_DOMAIN))
                .build();
    }

}
