package ru.yandex.direct.core.entity.placements.repository;

import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.placements.model.Placement;
import ru.yandex.direct.dbschema.ppcdict.tables.records.PlacementsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.dbschema.ppcdict.tables.Placements.PLACEMENTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;


/**
 * Репозиторий для работы с поступающей из AdFox информацией о площадках
 */
@Repository
@ParametersAreNonnullByDefault
public class PlacementsRepository {
    public static final int CHUNK_SIZE = 1000;
    private static final Logger logger = LoggerFactory.getLogger(PlacementsRepository.class);
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<Placement> mapper;

    @Autowired
    public PlacementsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.mapper = createMapper();
    }

    /**
     * Добавляет записи в таблицу ppcdict.placements порциями не более {@link PlacementsRepository#CHUNK_SIZE}.
     * Для каждой порции вызывает {@link PlacementsRepository#insertChunk(List)}.
     * При наличии записи с таким же PageID обновляет данные.
     *
     * @param placements Список данных о размещениях
     */
    public void insertPlacements(List<Placement> placements) {
        if (placements.isEmpty()) {
            logger.info("Nothing to update");
            return;
        }
        Lists.partition(placements, CHUNK_SIZE).forEach(this::insertChunk);
    }

    /**
     * Удалить записи из таблицы ppcdict.placements по pageId
     */
    public void deletePlacementsBy(Collection<Long> pageIds) {
        dslContextProvider.ppcdict()
                .deleteFrom(PLACEMENTS)
                .where(PLACEMENTS.PAGE_ID.in(pageIds))
                .execute();
    }

    /**
     * Добавляет записи в таблицу placements. При наличии записи с таким же PageID обновляет данные.
     *
     * @param placements Список размещений
     */
    private void insertChunk(List<Placement> placements) {
        InsertHelper<PlacementsRecord> insertHelper = new InsertHelper<>(dslContextProvider.ppcdict(), PLACEMENTS)
                .addAll(mapper, placements);
        if (insertHelper.hasAddedRecords()) {
            insertHelper
                    .onDuplicateKeyUpdate()
                    .set(PLACEMENTS.BLOCKS, MySQLDSL.values(PLACEMENTS.BLOCKS))
                    .set(PLACEMENTS.DOMAIN, MySQLDSL.values(PLACEMENTS.DOMAIN))
                    .set(PLACEMENTS.IS_YANDEX_PAGE, MySQLDSL.values(PLACEMENTS.IS_YANDEX_PAGE));
        }
        insertHelper.executeIfRecordsAdded();
    }

    /**
     * Получение информации о площадках по списку PageID
     * Читает данные из {@code ppcdict.placements}.
     */
    public List<Placement> getPlacements(Collection<Long> pageIds) {
        if (pageIds.isEmpty()) {
            return emptyList();
        }

        return dslContextProvider.ppcdict()
                .select(mapper.getFieldsToRead())
                .from(PLACEMENTS)
                .where(PLACEMENTS.PAGE_ID.in(pageIds))
                .fetch(mapper::fromDb);
    }

    /**
     * Возвращает n доменов неудалённых площадок
     * NB: запрос делает full scan и не предназначен для частого выполнения
     */
    public List<String> getRandomPlacementsDomains(int n) {
        return dslContextProvider.ppcdict()
                .select(PLACEMENTS.DOMAIN)
                .from(PLACEMENTS)
                .where(PLACEMENTS.IS_DELETED.eq(0L))
                .orderBy(DSL.rand())
                .limit(n)
                .fetch(PLACEMENTS.DOMAIN);
    }

    /**
     * По переданным PageId возвращает все PageId, которые существуют в таблице
     */
    public List<Long> getExistingPageIds(List<Long> pageIds) {
        return dslContextProvider.ppcdict()
                .select(PLACEMENTS.PAGE_ID)
                .from(PLACEMENTS)
                .where(PLACEMENTS.PAGE_ID.in(pageIds))
                .fetch(PLACEMENTS.PAGE_ID);
    }

    private JooqMapperWithSupplier<Placement> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(Placement::new)
                .map(property(Placement.PAGE_ID, PLACEMENTS.PAGE_ID))
                .map(property(Placement.BLOCKS, PLACEMENTS.BLOCKS))
                .map(property(Placement.DOMAIN, PLACEMENTS.DOMAIN))
                .map(property(Placement.IS_YANDEX_PAGE, PLACEMENTS.IS_YANDEX_PAGE))
                .build();
    }

}
