package ru.yandex.direct.core.entity.placements.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Lists;
import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.placements.model1.Placement;
import ru.yandex.direct.core.entity.placements.model1.PlacementBlock;
import ru.yandex.direct.core.entity.placements.repository.PlacementRepository;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptySet;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Service
public class UpdatePlacementService {

    private static final int CHUNK_SIZE = 1000;

    private final PlacementRepository placementRepository;

    @Autowired
    public UpdatePlacementService(PlacementRepository placementRepository) {
        this.placementRepository = placementRepository;
    }

    /**
     * (!) Не предназначен для конкурентного выполнения.
     * Не должен вызываться параллельно ни в рамках одного приложения, ни в разных,
     * так как синхронизация отсутствует.
     * <p>
     * Получает на вход площадки, в которых отсутствуют удаленные в ПИ блоки.
     * Для типизированных площадок находит в базе существующие блоки
     * и формирует новые модели площадок, в которых присутствуют все блоки,
     * а отсутствующие в исходном запросе блоки помечены удаленными,
     * после чего отправляет модели в репозиторий на обновление.
     * <p>
     * Нетипизированные блоки удаляются из базы. Сделано так потому что если у нетипизированной
     * площадки оставлять удаленные блоки в базе, то при присвоении ей типа и сохранении
     * расширенных данных по существующим блокам данные для удаленных блоков будет уже негде взять
     * и они останутся без дополнительных данных, что может вызвать проблемы.
     * <p>
     * Если блок отсутствует в запросе, но указан в dontDeleteBlocks,
     * то такой блок не удалится и не будет помечен удаленным.
     */
    public void addOrUpdatePlacementsAndMarkDeletedBlocks(List<? extends Placement> placements, Map<Long, Set<Long>> dontDeleteBlocks) {
        Lists.partition(placements, CHUNK_SIZE).forEach(placementsChunk ->
                updatePlacementsChunkAndMarkDeletedBlocks(placementsChunk, dontDeleteBlocks));
    }

    private void updatePlacementsChunkAndMarkDeletedBlocks(List<? extends Placement> placements, Map<Long, Set<Long>> dontDeleteBlocks) {
        Map<Long, Placement> requestPlacements = listToMap(placements, Placement::getId);
        Map<Long, Placement> existingPlacements = placementRepository.getPlacements(requestPlacements.keySet());
        Map<Long, Placement> readyPlacements =
                markUnexistingBlocksOfTypedPlacementsAsDeleted(requestPlacements, existingPlacements, dontDeleteBlocks);
        placementRepository.createOrUpdatePlacements(readyPlacements.values());
    }

    private Map<Long, Placement> markUnexistingBlocksOfTypedPlacementsAsDeleted(
            Map<Long, Placement> requestPlacements, Map<Long, Placement> existingPlacements, Map<Long, Set<Long>> dontDeleteBlocks) {
        return EntryStream.of(requestPlacements)
                .mapToValue((pageId, reqPlacement) -> {
                    // если у существующей площадки нет типа, то отсутствующие
                    // в запросе существующие блоки, т.е. удаленные, необходимо удалить
                    Placement existingPlacement = existingPlacements.get(pageId);
                    Set<Long> dontDeletePlacementBlocks = nvl(dontDeleteBlocks.get(pageId), emptySet());
                    if (existingPlacement == null) {
                        return reqPlacement;
                    }
                    return existingPlacement.getType() == null ?
                            mergeDontDeleteBlocks(reqPlacement, existingPlacement, dontDeletePlacementBlocks) :
                            markUnexistingBlocksAsDeleted(reqPlacement, existingPlacement, dontDeletePlacementBlocks);
                })
                .toMap();
    }

    /**
     * Для типизированных площадок помечает удаленные блоки
     * (отсутствующие в запросе, но присутствующие в базе) удаленными.
     * Блоки указанные в dontDeleteBlocks не должны быть помечены удаленными.
     */
    @SuppressWarnings("unchecked")
    private static <B extends Placement> B markUnexistingBlocksAsDeleted(B reqPlacement, B existingPlacement, Set<Long> dontDeleteBlocks) {
        checkArgument(reqPlacement.getId().equals(existingPlacement.getId()),
                "request and existing placement ids must be equal, actual request placement id: %d, existing placement id: %d",
                reqPlacement.getId(), existingPlacement.getId());
        checkArgument(reqPlacement.getType() != null, "type of request placement must be defined for placement %d", reqPlacement.getId());
        checkArgument(existingPlacement.getType() != null, "type of existing placement must be defined for placement %d", existingPlacement.getId());
        checkArgument(reqPlacement.getType().equals(existingPlacement.getType()),
                "types of request placement and existing placement must be equal for placement %d", reqPlacement.getId());

        Map<Long, PlacementBlock> blockIdToRequestBlocks =
                listToMap(reqPlacement.getBlocks(), PlacementBlock::getBlockId);
        Map<Long, PlacementBlock> blockIdToExistingBlocks =
                listToMap(existingPlacement.getBlocks(), PlacementBlock::getBlockId);

        List<PlacementBlock> newPlacementBlocks = new ArrayList<>(blockIdToRequestBlocks.values());
        EntryStream.of(blockIdToExistingBlocks)
                .removeKeys(blockIdToRequestBlocks::containsKey)
                .forKeyValue((blockId, existingBlock) -> {
                    PlacementBlock blockToBeStored = dontDeleteBlocks.contains(blockId)
                            ? existingBlock
                            : existingBlock.markDeleted();
                    newPlacementBlocks.add(blockToBeStored);
                });

        return (B) reqPlacement.replaceBlocks(newPlacementBlocks);
    }

    /**
     * Для нетипизированных площадок удаляет блоки отсутствующие в запросе, но присутствующие в базе.
     * Блоки указанные в dontDeleteBlocks не должны быть удалены из базы.
     */
    @SuppressWarnings("unchecked")
    private static Placement mergeDontDeleteBlocks(Placement reqPlacement, Placement existingPlacement, Set<Long> dontDeleteBlocks) {
        checkArgument(reqPlacement.getId().equals(existingPlacement.getId()),
                "request and existing placement ids must be equal");
        checkArgument(existingPlacement.getType() == null, "type of existing placement must be undefined");

        Map<Long, PlacementBlock> blockIdToRequestBlocks =
                listToMap(reqPlacement.getBlocks(), PlacementBlock::getBlockId);
        Map<Long, PlacementBlock> blockIdToExistingBlocks =
                listToMap(existingPlacement.getBlocks(), PlacementBlock::getBlockId);

        List<PlacementBlock> newPlacementBlocks = new ArrayList<>(blockIdToRequestBlocks.values());
        EntryStream.of(blockIdToExistingBlocks)
                .removeKeys(blockIdToRequestBlocks::containsKey)
                .forKeyValue((blockId, existingBlock) -> {
                    if (dontDeleteBlocks.contains(blockId)) {
                        newPlacementBlocks.add(existingBlock);
                    }
                });

        return reqPlacement.replaceBlocks(newPlacementBlocks);
    }
}
