package ru.yandex.direct.core.entity.postviewofflinereport.repository

import org.slf4j.Logger
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Repository
import ru.yandex.direct.core.entity.postviewofflinereport.model.PostViewOfflineReportDeviceType
import ru.yandex.direct.core.entity.postviewofflinereport.model.PostViewOfflineReportRow
import ru.yandex.direct.core.entity.postviewofflinereport.model.PostViewOfflineReportYtRow
import ru.yandex.direct.core.entity.postviewofflinereport.model.PostviewOfflineReportJobParams
import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.model.YtTable
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest
import ru.yandex.yt.ytclient.tables.ColumnValueType
import ru.yandex.yt.ytclient.tables.TableSchema
import java.time.Duration
import java.time.LocalDate
import java.util.concurrent.TimeUnit

@Lazy
@Repository
class PostViewOfflineReportYtRepository(
    var ytProvider: YtProvider,
    ytConfig: DirectYtDynamicConfig,
) {

    private val ytCluster = ytConfig.postViewOfflineReportYtClusters.iterator().next()
    private val tasksTablePath = ytConfig.tables().direct().postViewOfflineReportTasksTablePath()
    private val reportsPath = ytConfig.tables().direct().postViewOfflineReportReportsDirectoryPath()

    companion object {
        @JvmField val LOGGER: Logger = LoggerFactory.getLogger(PostViewOfflineReportYtRepository::class.java)

        const val WRITE_TIMEOUT_IN_SECONDS = 10L
        const val READ_TIMEOUT_IN_SECONDS = 5L
        const val TRACE_TAG = "postview_offline_report:yql"

        const val REPORT_ID_COLUMN = "report_id"
        const val CAMPAIGN_IDS_COLUMN = "cids"
        const val DATE_FROM_COLUMN = "date_from"
        const val DATE_TO_COLUMN = "date_to"

        private const val READ_CHUNK_SIZE = 1000

        val CAMPAIGN_REGEX = "[, ]+".toRegex()

        @JvmField val TASKS_TABLE_SCHEMA: TableSchema = TableSchema.Builder()
            .addKey(REPORT_ID_COLUMN, ColumnValueType.INT64)
            .addValue(CAMPAIGN_IDS_COLUMN, ColumnValueType.STRING)
            .addValue(DATE_FROM_COLUMN, ColumnValueType.STRING)
            .addValue(DATE_TO_COLUMN, ColumnValueType.STRING)
            .build()
    }

    /**
     * Поставить задание на построение отчёта — сделать запись в таблице для заказов
     * @param taskId — id задания, которое нужно поставить, совпадает со связанным jobId в dbqueue
     * @param params — параметры задания, те же, что у задания в dbqueue
     * @return удалось ли поставить задание
     */
    fun createTask(
        taskId: Long,
        params: PostviewOfflineReportJobParams,
    ) : Boolean {
        val trace = ru.yandex.direct.tracing.Trace.current().profile(TRACE_TAG, "createTask")
        trace.use {
            try {
                val operator = ytProvider.getDynamicOperator(ytCluster)
                val request = ModifyRowsRequest(tasksTablePath, TASKS_TABLE_SCHEMA)
                    .addInsert(listOf(taskId,
                        params.campaignIds.joinToString(separator = ", "),
                        params.dateFrom.toString(),
                        params.dateTo.toString()
                    ))
                operator.runInTransaction { it.modifyRows(request).get(WRITE_TIMEOUT_IN_SECONDS, TimeUnit.SECONDS) }
                return true
            } catch (e: RuntimeException) {
                LOGGER.error("Couldn't create a task in YT, id: $taskId, error: $e")
                return false
            }
        }
    }

    /**
     * Удалить задание на построение отчёта — удалить запись в таблице для заказов
     * @param taskId — id задания, которое нужно удалить, совпадает со связанным jobId в dbqueue
     * @return удалось ли удалить
     */
    fun deleteTask(
        taskId: Long,
    ) : Boolean {
        val trace = ru.yandex.direct.tracing.Trace.current().profile(TRACE_TAG, "deleteTask")
        trace.use {
            try {
                val operator = ytProvider.getDynamicOperator(ytCluster)
                val request = ModifyRowsRequest(tasksTablePath, TASKS_TABLE_SCHEMA)
                    .addDelete(listOf(taskId))
                operator.runInTransaction { it.modifyRows(request).get(WRITE_TIMEOUT_IN_SECONDS, TimeUnit.SECONDS) }
            } catch (e: RuntimeException) {
                LOGGER.error("Couldn't delete a task in YT, id: $taskId, error: $e")
                return false
            }
        }
        return true
    }

    /**
     * Получить задание на построение отчёта, для ТС
     * @return пара из id отчёта и его параметров или null, если нет заданий
     */
    fun getAnyTask() : Pair<Long, PostviewOfflineReportJobParams>? {
        val trace = ru.yandex.direct.tracing.Trace.current().profile(TRACE_TAG, "getAnyTask")
        trace.use {
            try {
                val operator = ytProvider.getDynamicOperator(ytCluster)
                val query = "* from [$tasksTablePath] limit 1"
                val resultRows = operator.selectRows(query, Duration.ofSeconds(READ_TIMEOUT_IN_SECONDS)).yTreeRows
                if (resultRows.isEmpty()) {
                    return null
                }

                val result = resultRows[0]

                val taskId = result.getOrThrow(REPORT_ID_COLUMN).longValue()
                val campaignIds = result.getOrThrow(CAMPAIGN_IDS_COLUMN).stringValue()
                    .split(CAMPAIGN_REGEX)
                    .map { it.toLong() }
                    .toSet()
                val dateFrom = LocalDate.parse(result.getOrThrow(DATE_FROM_COLUMN).stringValue())
                val dateTo = LocalDate.parse(result.getOrThrow(DATE_TO_COLUMN).stringValue())
                return Pair(taskId, PostviewOfflineReportJobParams(campaignIds, dateFrom, dateTo))
            } catch (e: RuntimeException) {
                LOGGER.error("Couldn't read or parse a task in YT, error: $e")
                throw e
            }
        }
    }

    /**
     * Проверить наличие и получить отчёт из YT
     * @param reportId — название таблицы с отчётом, совпадает со связанным jobId в dbqueue
     * @return список из строк отчёта или null, если таблицы с отчётом нет
     */
    fun getReport(reportId: Long) : List<PostViewOfflineReportRow>? {
        val tablePath = "$reportsPath/$reportId"
        val table = YtTable(tablePath)
        val trace = ru.yandex.direct.tracing.Trace.current().profile(TRACE_TAG, "getReport")
        trace.use {
            try {
                val operator = ytProvider.getOperator(ytCluster)
                if (!operator.exists(table)) {
                    return null
                }
                if (operator.readTableRowCount(table) == 0L) {
                    return listOf() // отчёт готов, данных нет
                }

                val tableRow = PostViewOfflineReportYtRow()
                val rowCollector = RowCollector()
                operator.readTableSnapshot(table,
                    tableRow, this::convertYtRow, rowCollector::collectRows, READ_CHUNK_SIZE)
                return rowCollector.rows
            } catch (e: RuntimeException) {
                LOGGER.error("Couldn't read or parse a report $tablePath in YT, error: $e")
                throw e
            }
        }
    }

    private fun convertYtRow(from: PostViewOfflineReportYtRow) : PostViewOfflineReportRow {
        return PostViewOfflineReportRow(
            from.clicks,
            from.shows,
            from.cost,
            from.date,
            from.visits,
            from.bounce,
            from.convertedSession,
            PostViewOfflineReportDeviceType.fromValue(from.deviceType),
            campaigns = from.campaign.split(CAMPAIGN_REGEX).map { it.toLong() }.toList(),
            start = from.start,
            end = from.end,
            goals = from.goals,
            postponedPeriod = from.postponedPeriod,
        )
    }

    private class RowCollector {
        val rows = mutableListOf<PostViewOfflineReportRow>()

        fun collectRows(rows: List<PostViewOfflineReportRow>) {
            this.rows.addAll(rows)
        }
    }
}
