package ru.yandex.direct.core.entity.postviewofflinereport.service

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.dbqueue.DbQueueJobTypes.POSTVIEW_OFFLINE_REPORT
import ru.yandex.direct.core.entity.postviewofflinereport.PostviewOfflineReportConverter.Companion.convertJobToReport
import ru.yandex.direct.core.entity.postviewofflinereport.model.PostviewOfflineReport
import ru.yandex.direct.core.entity.postviewofflinereport.model.PostviewOfflineReportJobParams
import ru.yandex.direct.core.entity.postviewofflinereport.model.PostviewOfflineReportJobResult
import ru.yandex.direct.core.entity.postviewofflinereport.repository.PostViewOfflineReportYtRepository
import ru.yandex.direct.dbqueue.LimitOffset
import ru.yandex.direct.dbqueue.repository.DbQueueRepository
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import java.time.Duration
import java.time.LocalDate

@Service
class PostviewOfflineReportService(
    private val dbQueueRepository: DbQueueRepository,
    private val shardHelper: ShardHelper,
    private val postViewOfflineReportYtRepository: PostViewOfflineReportYtRepository
) {
    fun getReport(clientId: ClientId, reportId: Long): PostviewOfflineReport? {
        val shard = shardHelper.getShardByClientId(clientId)
        val job = dbQueueRepository.findJobById(
            shard, POSTVIEW_OFFLINE_REPORT, reportId
        )
        if (job == null || job.clientId != clientId) {
            return null
        }

        return convertJobToReport(job)
    }

    fun getReportList(clientId: ClientId): Set<PostviewOfflineReport> {
        val shard = shardHelper.getShardByClientId(clientId)
        val jobs = dbQueueRepository.getJobsByJobTypeAndClientIds(
            shard,
            POSTVIEW_OFFLINE_REPORT, setOf(clientId.asLong()), LimitOffset.maxLimited()
        ) + dbQueueRepository.getJobsFromArchiveByJobTypeAndClientIds(
            shard,
            POSTVIEW_OFFLINE_REPORT, setOf(clientId.asLong()), LimitOffset.maxLimited()
        )

        return jobs.map { convertJobToReport(it) }.toSet()
    }

    fun createReport(
        operatorUid: Long, clientId: ClientId, campaignIds: Set<Long>, dateFrom: LocalDate, dateTo: LocalDate
    ): PostviewOfflineReport {
        val shard = shardHelper.getShardByClientId(clientId)
        val job = dbQueueRepository.insertDelayedJob(
            shard,
            POSTVIEW_OFFLINE_REPORT,
            clientId,
            operatorUid,
            PostviewOfflineReportJobParams(campaignIds, dateFrom, dateTo),
            Duration.ofHours(1)
        )

        var isYtTaskCreated = false
        for (index in 1 .. 3) {
            isYtTaskCreated = postViewOfflineReportYtRepository.createTask(job.id, job.args)
            if (isYtTaskCreated) {
                break
            }
        }
        if (!isYtTaskCreated) {
            dbQueueRepository.markJobRevoked(
                shard,
                job,
                PostviewOfflineReportJobResult(null, "save report data to YT failed")
            )
        }

        return convertJobToReport(job)
    }

    fun deleteReport(clentId: ClientId, reportId: Long): Boolean {
        if (!postViewOfflineReportYtRepository.deleteTask(reportId)) {
            return false
        }
        val shard = shardHelper.getShardByClientId(clentId)
        return dbQueueRepository.deleteArchivedJobs(shard, setOf(reportId)) == 1
    }
}
