package ru.yandex.direct.core.entity.pricepackage.model;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Set;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktop;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDesktopAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobileAdjustment;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.currency.CurrencyCode;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.DESKTOP_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.INVENTORY_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.BidModifierType.MOBILE_MULTIPLIER;
import static ru.yandex.direct.core.entity.bidmodifier.InventoryType.INAPP;
import static ru.yandex.direct.utils.CommonUtils.nvl;

public class PricePackage extends PricePackageBase {

    /**
     * Возвращает признак, что этот пакет на главной.
     */
    public boolean isFrontpagePackage() {
        return this.getAvailableAdGroupTypes().contains(AdGroupType.CPM_YNDX_FRONTPAGE);
    }

    public boolean isFrontpageVideoPackage() {
        return this.getAvailableAdGroupTypes().contains(AdGroupType.CPM_VIDEO)
                && this.getIsFrontpage();
    }

    public boolean needDefaultAdGroup() {
        return isFrontpagePackage() || isFrontpageVideoPackage();
    }

    public PricePackagePlatform getPricePackagePlatform() {
        boolean isDesktop = false;
        boolean isMobile = false;

        if (isFrontpagePackage()) {
            if (getTargetingsFixed() != null && getTargetingsFixed().getViewTypes() != null) {
                List<ViewType> viewTypes = getTargetingsFixed().getViewTypes();
                isDesktop = viewTypes.contains(ViewType.DESKTOP) || viewTypes.contains(ViewType.NEW_TAB);
                isMobile = viewTypes.contains(ViewType.MOBILE);
            }
        } else {
            if (getBidModifiers() != null) {
                isDesktop = getBidModifiers().stream()
                        .anyMatch(bidModifier -> bidModifier.getType() == DESKTOP_MULTIPLIER);
                isMobile = getBidModifiers().stream()
                        .anyMatch(bidModifier -> {
                            if (bidModifier.getType() == MOBILE_MULTIPLIER) {
                                return true;
                            } else if (bidModifier.getType() == INVENTORY_MULTIPLIER) {
                                var bidModifierInventory = (BidModifierInventory) bidModifier;
                                List<BidModifierInventoryAdjustment> adjList =
                                        bidModifierInventory.getInventoryAdjustments();
                                return adjList != null && adjList.size() == 1 &&
                                        adjList.get(0).getInventoryType() == INAPP;
                            }
                            return false;
                        });
            }
        }

        if (isDesktop && !isMobile) {
            return PricePackagePlatform.DESKTOP;
        } else if (isMobile && !isDesktop) {
            return PricePackagePlatform.MOBILE;
        }

        return PricePackagePlatform.ALL;
    }

    @Override
    public PricePackage withId(Long id) {
        setId(id);
        return this;
    }

    @Override
    public PricePackage withProductId(Long productId) {
        setProductId(productId);
        return this;
    }

    @Override
    public PricePackage withAuctionPriority(Long auctionPriority) {
        setAuctionPriority(auctionPriority);
        return this;
    }

    @Override
    public PricePackage withTitle(String title) {
        setTitle(title);
        return this;
    }

    @Override
    public PricePackage withTrackerUrl(String trackerUrl) {
        setTrackerUrl(trackerUrl);
        return this;
    }

    @Override
    public PricePackage withPrice(BigDecimal price) {
        setPrice(price);
        return this;
    }

    @Override
    public PricePackage withCurrency(CurrencyCode currency) {
        setCurrency(currency);
        return this;
    }

    @Override
    public PricePackage withOrderVolumeMin(Long orderVolumeMin) {
        setOrderVolumeMin(orderVolumeMin);
        return this;
    }

    @Override
    public PricePackage withOrderVolumeMax(Long orderVolumeMax) {
        setOrderVolumeMax(orderVolumeMax);
        return this;
    }

    @Override
    public PricePackage withPriceMarkups(List<PriceMarkup> priceMarkups) {
        setPriceMarkups(priceMarkups);
        return this;
    }

    @Override
    public PricePackage withTargetingMarkups(List<TargetingMarkup> targetingMarkups) {
        setTargetingMarkups(targetingMarkups);
        return this;
    }

    @Override
    public PricePackage withTargetingsFixed(TargetingsFixed targetingsFixed) {
        setTargetingsFixed(targetingsFixed);
        return this;
    }

    @Override
    public PricePackage withTargetingsCustom(TargetingsCustom targetingsCustom) {
        setTargetingsCustom(targetingsCustom);
        return this;
    }

    @Override
    public PricePackage withStatusApprove(StatusApprove statusApprove) {
        setStatusApprove(statusApprove);
        return this;
    }

    @Override
    public PricePackage withLastUpdateTime(LocalDateTime lastUpdateTime) {
        setLastUpdateTime(lastUpdateTime);
        return this;
    }

    @Override
    public PricePackage withDateStart(LocalDate dateStart) {
        setDateStart(dateStart);
        return this;
    }

    @Override
    public PricePackage withDateEnd(LocalDate dateEnd) {
        setDateEnd(dateEnd);
        return this;
    }

    @Override
    public PricePackage withIsPublic(Boolean isPublic) {
        setIsPublic(isPublic);
        return this;
    }

    @Override
    public PricePackage withIsSpecial(Boolean isSpecial) {
        setIsSpecial(isSpecial);
        return this;
    }

    @Override
    public PricePackage withIsCpd(Boolean isCpd) {
        setIsCpd(isCpd);
        return this;
    }

    @Override
    public PricePackage withIsFrontpage(Boolean val) {
        setIsFrontpage(val);
        return this;
    }

    @Override
    public PricePackage withIsArchived(Boolean isArchived) {
        setIsArchived(isArchived);
        return this;
    }

    @Override
    public PricePackage withClients(List<PricePackageClient> clients) {
        setClients(clients);
        return this;
    }

    @Override
    public PricePackage withAvailableAdGroupTypes(Set<AdGroupType> availableAdGroupTypes) {
        setAvailableAdGroupTypes(availableAdGroupTypes);
        return this;
    }

    @Override
    public PricePackage withAllowedPageIds(List<Long> allowedPageIds) {
        setAllowedPageIds(allowedPageIds);
        return this;
    }

    @Override
    public PricePackage withAllowedProjectParamConditions(List<Long> allowedProjectParamConditions) {
        setAllowedProjectParamConditions(allowedProjectParamConditions);
        return this;
    }

    @Override
    public PricePackage withAllowedDomains(List<String> allowedDomains) {
        setAllowedDomains(allowedDomains);
        return this;
    }

    @Override
    public PricePackage withAllowedSsp(List<String> allowedSsp) {
        setAllowedSsp(allowedSsp);
        return this;
    }

    @Override
    public PricePackage withAllowedTargetTags(List<String> allowedTargetTags) {
        setAllowedTargetTags(allowedTargetTags);
        return this;
    }

    @Override
    public PricePackage withAllowedOrderTags(List<String> allowedOrderTags) {
        setAllowedOrderTags(allowedOrderTags);
        return this;
    }

    @Override
    public PricePackage withCampaignAutoApprove(Boolean campaignAutoApprove) {
        setCampaignAutoApprove(campaignAutoApprove);
        return this;
    }

    @Override
    public PricePackage withBidModifiers(List<BidModifier> bidModifiers) {
        setBidModifiers(bidModifiers);
        return this;
    }

    @Override
    public PricePackage withCampaignOptions(PricePackageCampaignOptions campaignOptions) {
        setCampaignOptions(campaignOptions);
        return this;
    }

    @Override
    public PricePackage withAllowedCreativeTemplates(
            Map<CreativeType, List<Long>> allowedCreativeTemplates) {
        setAllowedCreativeTemplates(allowedCreativeTemplates);
        return this;
    }

    @Override
    public PricePackage withIsDraftApproveAllowed(Boolean isDraftApproveAllowed) {
        setIsDraftApproveAllowed(isDraftApproveAllowed);
        return this;
    }

    @Override
    public PricePackage withCategoryId(Long categoryId) {
        setCategoryId(categoryId);
        return this;
    }

    public void normalizeBidModifiers() {
        List<BidModifier> bidModifiers = nvl(getBidModifiers(), emptyList());
        bidModifiers.forEach(bidModifier -> {
            switch(bidModifier.getType()) {
                case INVENTORY_MULTIPLIER:
                    BidModifierInventory bidModifierInventory = (BidModifierInventory) bidModifier;
                    List<BidModifierInventoryAdjustment> adjList = nvl(bidModifierInventory.getInventoryAdjustments(),
                            emptyList());
                    adjList.forEach(adj -> adj.setIsRequiredInPricePackage(
                            nvl(adj.getIsRequiredInPricePackage(), true)));
                    bidModifierInventory.setInventoryAdjustments(adjList);
                    break;
                case MOBILE_MULTIPLIER:
                    BidModifierMobile bidModifierMobile = (BidModifierMobile) bidModifier;
                    BidModifierMobileAdjustment mAdj = nvl(bidModifierMobile.getMobileAdjustment(),
                            new BidModifierMobileAdjustment());
                    mAdj.setIsRequiredInPricePackage(nvl(mAdj.getIsRequiredInPricePackage(), true));
                    bidModifierMobile.setMobileAdjustment(mAdj);
                    break;
                case DESKTOP_MULTIPLIER:
                    BidModifierDesktop bidModifierDesktop = (BidModifierDesktop) bidModifier;
                    BidModifierDesktopAdjustment dAdj = nvl(bidModifierDesktop.getDesktopAdjustment(),
                            new BidModifierDesktopAdjustment());
                    dAdj.setIsRequiredInPricePackage(nvl(dAdj.getIsRequiredInPricePackage(), true));
                    bidModifierDesktop.setDesktopAdjustment(dAdj);
                    break;
            }
        });
        setBidModifiers(bidModifiers);
    }

    public void normalizeCampaignOptions() {
        PricePackageCampaignOptions campaignOptions =
                nvl(getCampaignOptions(), new PricePackageCampaignOptions());
        campaignOptions.setAllowBrandLift(nvl(campaignOptions.getAllowBrandLift(), false));
        //Если флаг явно не проставлен, то для главной по умолчанию брендсейфити не включено, для остальных включено
        campaignOptions.setAllowBrandSafety(nvl(campaignOptions.getAllowBrandSafety(), !isFrontpagePackage()));
        campaignOptions.setAllowDisabledPlaces(nvl(campaignOptions.getAllowDisabledPlaces(), false));
        campaignOptions.setAllowDisabledVideoPlaces(nvl(campaignOptions.getAllowDisabledVideoPlaces(), false));
        campaignOptions.setAllowImage(nvl(campaignOptions.getAllowImage(), false));
        campaignOptions.setAllowGallery(nvl(campaignOptions.getAllowGallery(), false));
        setCampaignOptions(campaignOptions);
    }

    public void normalizeFixedCustomTargetings() {
        TargetingsCustom targetingsCustom = getTargetingsCustom();
        PriceRetargetingCondition priceRetCond = targetingsCustom.getRetargetingCondition();
        if (priceRetCond == null) {
            targetingsCustom.setRetargetingCondition(new PriceRetargetingCondition()
                    .withAllowAudienceSegments(false)
                    .withAllowMetrikaSegments(false)
                    .withCryptaSegments(emptyList())
                    .withLowerCryptaTypesCount(0)
                    .withUpperCryptaTypesCount(0));
        } else {
            priceRetCond.setLowerCryptaTypesCount(nvl(priceRetCond.getLowerCryptaTypesCount(), 0));
            priceRetCond.setUpperCryptaTypesCount(nvl(priceRetCond.getUpperCryptaTypesCount(), 0));
            priceRetCond.setAllowAudienceSegments(nvl(priceRetCond.getAllowAudienceSegments(), false));
            priceRetCond.setAllowMetrikaSegments(nvl(priceRetCond.getAllowMetrikaSegments(), false));
            priceRetCond.setCryptaSegments(nvl(priceRetCond.getCryptaSegments(), emptyList()));
        }
        TargetingsFixed targetingsFixed = getTargetingsFixed();
        targetingsFixed.setCryptaSegments(nvl(targetingsFixed.getCryptaSegments(), emptyList()));
        targetingsFixed.setAllowExpandedDesktopCreative(nvl(targetingsFixed.getAllowExpandedDesktopCreative(), false));
        targetingsFixed.setAllowPremiumDesktopCreative(nvl(targetingsFixed.getAllowPremiumDesktopCreative(), false));
        targetingsFixed.setHideIncomeSegment(nvl(targetingsFixed.getHideIncomeSegment(), false));
    }
}
