package ru.yandex.direct.core.entity.pricepackage.repository;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.JoinType;
import org.jooq.Record;
import org.jooq.SelectConditionStep;
import org.jooq.SelectQuery;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.container.LocalDateRange;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.mapping.CommonMappings;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageCategory;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageClient;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageForClient;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageForLock;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageOrderBy;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageWithoutClients;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackagesFilter;
import ru.yandex.direct.core.entity.pricepackage.model.StatusApprove;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsCustom;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsFixed;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackageGeoTree;
import ru.yandex.direct.core.util.GeoTreeUtils;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbschema.ppcdict.enums.CpmPricePackagesCurrency;
import ru.yandex.direct.dbschema.ppcdict.tables.records.CpmPricePackageClientsRecord;
import ru.yandex.direct.dbschema.ppcdict.tables.records.CpmPricePackagesRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.regions.GeoTree;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.core.entity.pricepackage.repository.PricePackageMapping.currencyToDbFormat;
import static ru.yandex.direct.core.entity.pricepackage.repository.PricePackageMapping.statusApproveToDbFormat;
import static ru.yandex.direct.dbschema.ppcdict.tables.CpmPricePackageCategories.CPM_PRICE_PACKAGE_CATEGORIES;
import static ru.yandex.direct.dbschema.ppcdict.tables.CpmPricePackageClients.CPM_PRICE_PACKAGE_CLIENTS;
import static ru.yandex.direct.dbschema.ppcdict.tables.CpmPricePackages.CPM_PRICE_PACKAGES;
import static ru.yandex.direct.dbschema.ppcdict.tables.Products.PRODUCTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
@ParametersAreNonnullByDefault
public class PricePackageRepository {

    private static final JooqMapperWithSupplier<PricePackage> PRICE_PACKAGE_MAPPER = createPricePackageMapper();

    private static final JooqMapperWithSupplier<PricePackageClient> PRICE_PACKAGE_CLIENT_MAPPER =
            createPricePackageClientMapper();

    private static final JooqMapperWithSupplier<PricePackageCategory> PRICE_PACKAGE_CATEGORY_MAPPER =
            createPricePackageCategoryMapper();

    private static final int BATCH_SIZE = 1000;

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final FeatureService featureService;
    private final PricePackageGeoTree pricePackageGeoTree;
    private final ClientService clientService;

    @Autowired
    public PricePackageRepository(DslContextProvider dslContextProvider,
                                  ShardHelper shardHelper,
                                  FeatureService featureService,
                                  PricePackageGeoTree pricePackageGeoTree,
                                  ClientService clientService) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.featureService = featureService;
        this.pricePackageGeoTree = pricePackageGeoTree;
        this.clientService = clientService;
    }

    public Map<Long, PricePackage> getPricePackages(Collection<Long> packageIds) {
        if (packageIds.isEmpty()) {
            return emptyMap();
        }
        PricePackagesWithTotalCount pricePackagesWithTotalCount = getPricePackages(
                new PricePackagesFilter().withPackageIdIn(new HashSet<>(packageIds)),
                null,
                LimitOffset.maxLimited());
        return listToMap(pricePackagesWithTotalCount.pricePackages, PricePackage::getId);
    }

    public PricePackagesWithTotalCount getPricePackages(PricePackagesFilter filter,
                                                        @Nullable PricePackageOrderBy orderBy,
                                                        LimitOffset limitOffset) {
        Map<ClientId, Currency> clientsCurrencies;
        if (filter.getClientIn() != null) {
            Map<String, Long> loginToClientId = shardHelper.getClientIdsByLogins(filter.getClientIn());
            List<ClientId> clientIds = mapList(loginToClientId.values(), ClientId::fromLong);
            clientsCurrencies = clientService.massGetWorkCurrency(clientIds);
        } else {
            clientsCurrencies = null;
        }
        List<PricePackage> dbPricePackages = getPricePackagesFromDb(filter, clientsCurrencies);
        List<PricePackage> javaPricePackages = applyJavaFilters(dbPricePackages, filter, clientsCurrencies, orderBy);
        List<PricePackage> resultPricePackages = javaPricePackages.stream()
                .skip(limitOffset.offset())
                .limit(limitOffset.limit())
                .collect(toList());
        return new PricePackagesWithTotalCount(resultPricePackages, javaPricePackages.size());
    }

    /**
     * Возвращает пакеты удовлетворяющие filter (только та часть, что делаем на уровне базы), отсортированные по id.
     */
    private List<PricePackage> getPricePackagesFromDb(PricePackagesFilter filter,
                                                      @Nullable Map<ClientId, Currency> clientsCurrencies) {
        Long lastPricePackageId = 0L;
        List<PricePackage> result = new ArrayList<>();
        // бьём на batches чтобы черезчур не грузить ppcdict
        while (true) {
            var batch = getPricePackagesFromDbOneBatch(filter, clientsCurrencies, lastPricePackageId);
            result.addAll(batch);
            if (batch.size() < BATCH_SIZE) {
                return result;
            }
            lastPricePackageId = batch.get(batch.size() - 1).getId();
        }
    }

    private List<PricePackage> getPricePackagesFromDbOneBatch(
            PricePackagesFilter filter,
            @Nullable Map<ClientId, Currency> clientsCurrencies,
            Long lastPricePackageId) {
        return dslContextProvider.ppcdictTransactionResult(configuration -> {
            SelectQuery<Record> select = DSL.using(configuration).selectQuery();

            select.addSelect(PRICE_PACKAGE_MAPPER.getFieldsToRead());
            select.addFrom(CPM_PRICE_PACKAGES);

            if (filter.getAdGroupTypes() != null) {
                var adGroupTypes = filter.getAdGroupTypes().stream().map(AdGroupType::name).collect(toList());
                select.addConditions(CPM_PRICE_PACKAGES.AVAILABLE_AD_GROUP_TYPES.in(adGroupTypes));
            }
            if (filter.getPackageIdIn() != null) {
                select.addConditions(CPM_PRICE_PACKAGES.PACKAGE_ID.in(filter.getPackageIdIn()));
            }
            if (filter.getTitleContains() != null) {
                select.addConditions(CPM_PRICE_PACKAGES.TITLE.containsIgnoreCase(filter.getTitleContains()));
            }
            if (filter.getBusinessUnitIdIn() != null || nvl(filter.getIsNotBusinessUnit(), false)) {
                select.addJoin(PRODUCTS, JoinType.LEFT_OUTER_JOIN,
                        CPM_PRICE_PACKAGES.PRODUCT_ID.eq(PRODUCTS.PRODUCT_ID));
                Condition businessUnitCondition = DSL.noCondition();
                if (filter.getBusinessUnitIdIn() != null) {
                    businessUnitCondition = businessUnitCondition.or(
                            PRODUCTS.BUSINESS_UNIT.in(filter.getBusinessUnitIdIn()));
                }
                if (nvl(filter.getIsNotBusinessUnit(), false)) {
                    businessUnitCondition = businessUnitCondition.or(PRODUCTS.BUSINESS_UNIT.isNull());
                }
                select.addConditions(businessUnitCondition);
            }
            if (filter.getIsPublic() != null) {
                select.addConditions(filter.getIsPublic() ?
                        CPM_PRICE_PACKAGES.IS_PUBLIC.isTrue()
                        : CPM_PRICE_PACKAGES.IS_PUBLIC.isFalse());
            }
            if (filter.getIsSpecial() != null) {
                select.addConditions(filter.getIsSpecial() ?
                        CPM_PRICE_PACKAGES.IS_SPECIAL.isTrue()
                        : CPM_PRICE_PACKAGES.IS_SPECIAL.isFalse());
            }
            if (filter.getIsArchived() != null) {
                select.addConditions(filter.getIsArchived() ?
                        CPM_PRICE_PACKAGES.IS_ARCHIVED.isTrue()
                        : CPM_PRICE_PACKAGES.IS_ARCHIVED.isFalse());
            }
            if (filter.getClientIn() != null) {
                Set<CpmPricePackagesCurrency> allowedCurrencies = clientsCurrencies.values().stream()
                        .map(x -> currencyToDbFormat(x.getCode()))
                        .collect(Collectors.toSet());
                select.addConditions(CPM_PRICE_PACKAGES.CURRENCY.in(allowedCurrencies));
            }

            if (filter.getActivityIntervals() != null) {
                Condition dateCondition = DSL.noCondition();
                for (LocalDateRange interval : filter.getActivityIntervals()) {
                    Condition oneDateCondition = DSL.noCondition();
                    LocalDate from = interval.getFromInclusive();
                    if (from != null) {
                        oneDateCondition = oneDateCondition.and(CPM_PRICE_PACKAGES.DATE_END.greaterOrEqual(from));
                    }
                    LocalDate to = interval.getToInclusive();
                    if (to != null) {
                        oneDateCondition = oneDateCondition.and(CPM_PRICE_PACKAGES.DATE_START.lessOrEqual(to));
                    }
                    dateCondition = dateCondition.or(oneDateCondition);
                }
                select.addConditions(dateCondition);
            }

            if (filter.getMinPrice() != null) {
                select.addConditions(CPM_PRICE_PACKAGES.PRICE.greaterOrEqual(filter.getMinPrice()));
            }

            if (filter.getIsCpd() != null) {
                select.addConditions(filter.getIsCpd() ?
                        CPM_PRICE_PACKAGES.IS_CPD.eq(1L)
                        : CPM_PRICE_PACKAGES.IS_CPD.eq(0L));
            }

            if (filter.getMaxPrice() != null) {
                select.addConditions(CPM_PRICE_PACKAGES.PRICE.lessOrEqual(filter.getMaxPrice()));
            }

            if (filter.getCurrencyIn() != null) {
                var dbCurrency = filter.getCurrencyIn().stream()
                        .map(PricePackageMapping::currencyToDbFormat)
                        .collect(toList());
                select.addConditions(CPM_PRICE_PACKAGES.CURRENCY.in(dbCurrency));
            }

            if (filter.getStatusApproveIn() != null) {
                var dbStatusApprove = filter.getStatusApproveIn().stream()
                        .map(PricePackageMapping::statusApproveToDbFormat)
                        .collect(toList());
                select.addConditions(CPM_PRICE_PACKAGES.STATUS_APPROVE.in(dbStatusApprove));
            }

            if (filter.getDateEndAfter() != null) {
                select.addConditions(CPM_PRICE_PACKAGES.DATE_END.gt(filter.getDateEndAfter()));
            }

            select.addConditions(CPM_PRICE_PACKAGES.PACKAGE_ID.greaterThan(lastPricePackageId));
            select.addOrderBy(CPM_PRICE_PACKAGES.PACKAGE_ID);
            select.addLimit(BATCH_SIZE);

            List<PricePackage> pricePackages = select.fetch().map(PRICE_PACKAGE_MAPPER::fromDb);
            enrichWithClients(configuration.dsl(), pricePackages);
            normalizePricePackagesData(pricePackages);

            return pricePackages;
        });
    }

    private List<PricePackage> applyJavaFilters(List<PricePackage> dbPricePackages,
                                                PricePackagesFilter filter,
                                                @Nullable Map<ClientId, Currency> clientsCurrencies,
                                                @Nullable PricePackageOrderBy orderBy) {
        Set<ClientId> clientsWithPublicPackagesEnabled;
        if (clientsCurrencies == null) {
            clientsWithPublicPackagesEnabled = null;
        } else {
            clientsWithPublicPackagesEnabled = getClientsWithPublicPackagesEnabled(
                    new HashSet<>(clientsCurrencies.keySet()));
        }

        return dbPricePackages.stream()
                .filter(pricePackage -> pricePackageIsSelectedByJavaFilter(pricePackage, filter, clientsCurrencies,
                        clientsWithPublicPackagesEnabled))
                .sorted(getOrderByComparator(orderBy))
                .collect(toList());
    }

    private boolean pricePackageIsSelectedByJavaFilter(PricePackage pricePackage,
                                                       PricePackagesFilter filter,
                                                       @Nullable Map<ClientId, Currency> clientsCurrencies,
                                                       @Nullable Set<ClientId> clientsWithPublicPackagesEnabled) {
        boolean regionIdFilterIsSet;
        boolean regionIdFilterPassed = false;
        if (filter.getRegionIds() == null) {
            regionIdFilterIsSet = false;
        } else {
            regionIdFilterIsSet = true;
            regionIdFilterPassed = filterByGeoRegions(pricePackage, filter);
        }

        boolean clientIdFilterIsSet;
        boolean clientIdFilterPassed = false;
        if (filter.getClientIn() == null) {
            clientIdFilterIsSet = false;
        } else {
            clientIdFilterIsSet = true;
            clientIdFilterPassed = EntryStream.of(clientsCurrencies)
                    .anyMatch((clientId, clientCurrency) ->
                            isPricePackageAvailableForClient(clientId, clientCurrency, pricePackage,
                                    clientsWithPublicPackagesEnabled));
        }

        return (!regionIdFilterIsSet || regionIdFilterPassed)
                && (!clientIdFilterIsSet || clientIdFilterPassed);
    }

    private static boolean isPricePackageAvailableForClient(ClientId clientId,
                                                            Currency clientCurrency,
                                                            PricePackage pricePackage,
                                                            Set<ClientId> clientsWithPublicPackagesEnabled) {
        if (pricePackage.getCurrency() != clientCurrency.getCode()) {
            return false;
        }
        if (pricePackage.getIsPublic()) {
            return clientsWithPublicPackagesEnabled.contains(clientId)
                    && clientIsNotBanned(clientId, pricePackage.getClients());
        } else {
            return clientIsAssigned(clientId, pricePackage.getClients());
        }
    }

    private static boolean clientIsNotBanned(ClientId clientId,
                                             List<PricePackageClient> pricePackageClients) {
        return pricePackageClients.stream().noneMatch(
                pricePackageClient -> pricePackageClient.getClientId().equals(clientId.asLong())
                        && !pricePackageClient.getIsAllowed());
    }


    private static boolean clientIsAssigned(ClientId clientId,
                                            List<PricePackageClient> pricePackageClients) {
        return pricePackageClients.stream().anyMatch(
                pricePackageClient -> pricePackageClient.getClientId().equals(clientId.asLong())
                        && pricePackageClient.getIsAllowed());
    }

    private Comparator<PricePackage> getOrderByComparator(@Nullable PricePackageOrderBy orderBy) {
        if (orderBy == null) {
            return (a, b) -> 0;
        }
        Comparator<PricePackage> result;
        switch (orderBy.getField()) {
            case PACKAGE_ID:
                result = comparing(PricePackage::getId);
                break;
            case TITLE:
                result = comparing(PricePackage::getTitle, String.CASE_INSENSITIVE_ORDER);
                break;
            case STATUS_APPROVE:
                // так совпало, что statusApprove требуется сортировать в том же порядке, как они перечислены в enum
                result = comparing(PricePackage::getStatusApprove);
                break;
            case PRICE:
                result = comparing(PricePackage::getPrice);
                break;
            default:
                throw new IllegalStateException("not implemented");
        }
        switch (orderBy.getOrder()) {
            case ASC:
                return result;
            case DESC:
                return result.reversed();
            default:
                throw new IllegalStateException("not implemented");
        }
    }

    private void enrichWithClients(DSLContext dslContext, Collection<PricePackage> pricePackages) {
        Set<Long> pricePackageIds = listToSet(pricePackages, PricePackage::getId);

        Map<Long, List<PricePackageClient>> pricePackageIdToClients = getPricePackageIdToClients(dslContext,
                pricePackageIds);

        pricePackages.forEach(pricePackage ->
                pricePackage.setClients(pricePackageIdToClients.getOrDefault(pricePackage.getId(), emptyList())));
    }

    private static void normalizePricePackagesData(Collection<PricePackage> pricePackages) {
        pricePackages.forEach(PricePackageRepository::normalizePricePackageData);

    }

    private static void normalizePricePackageData(PricePackage pricePackage) {
        pricePackage.setProductId(nvl(pricePackage.getProductId(), 0L));
        pricePackage.setAuctionPriority(pricePackage.getAuctionPriority());
        pricePackage.setAllowedPageIds(nvl(pricePackage.getAllowedPageIds(), emptyList()));
        pricePackage.setPriceMarkups(nvl(pricePackage.getPriceMarkups(), emptyList()));
        pricePackage.setTargetingMarkups(nvl(pricePackage.getTargetingMarkups(), emptyList()));

        pricePackage.normalizeBidModifiers();
        pricePackage.normalizeCampaignOptions();
        pricePackage.normalizeFixedCustomTargetings();
    }

    /**
     * Возвращает все пакеты доступные для клиента (публичные и приватные).
     */
    public List<PricePackageWithoutClients> getActivePricePackagesWithoutClients(ClientId clientId,
                                                                                 CurrencyCode clientCurrency,
                                                                                 Collection<Long> pricePackageIds) {
        List<PricePackage> privatePackages = getActivePrivatePackages(clientId,
                PricePackageWithoutClients.allModelProperties(),
                pricePackageIds);
        List<PricePackage> publicPackages = getActivePublicPackages(clientId,
                clientCurrency,
                PricePackageWithoutClients.allModelProperties(),
                pricePackageIds);

        List<PricePackageWithoutClients> result = new ArrayList<>(privatePackages);
        result.addAll(publicPackages);

        return result;
    }

    /**
     * Возвращает все пакеты доступные для клиента (публичные и приватные).
     */
    public List<PricePackageForClient> getActivePricePackagesForClient(ClientId clientId,
                                                                       CurrencyCode clientCurrency,
                                                                       PricePackagesFilter filter) {

        List<PricePackage> privatePackages = getActivePrivatePackages(clientId,
                PricePackageForClient.allModelProperties());

        List<PricePackage> publicPackages = getActivePublicPackages(clientId,
                clientCurrency,
                PricePackageForClient.allModelProperties());

        return Stream.concat(privatePackages.stream(), publicPackages.stream())
                .filter(pricePackage -> filterByIsSpecial(pricePackage, filter))
                .filter(pricePackage -> filterByFormats(pricePackage, filter))
                .filter(pricePackage -> filterByPlatforms(pricePackage, filter))
                .filter(pricePackage -> filterByGeoRegions(pricePackage, filter))
                .collect(toList());
    }

    private static boolean filterByIsSpecial(PricePackage pricePackage, PricePackagesFilter filter) {
        return filter.getIsSpecial() == null || filter.getIsSpecial().equals(pricePackage.getIsSpecial());
    }

    private static boolean filterByFormats(PricePackage pricePackage, PricePackagesFilter filter) {
        return filter.getFormats() == null ||
                pricePackage.getAvailableAdGroupTypes().stream().anyMatch(filter.getFormats()::contains);
    }

    private static boolean filterByPlatforms(PricePackage pricePackage, PricePackagesFilter filter) {
        return filter.getPlatforms() == null ||
                filter.getPlatforms().contains(pricePackage.getPricePackagePlatform());
    }

    private boolean filterByGeoRegions(PricePackage pricePackage, PricePackagesFilter filter) {
        if (filter.getRegionIds() == null) {
            return true;
        }

        boolean fixedGeoOverlap = false;
        boolean customGeoOverlap = false;

        TargetingsFixed targetingsFixed = pricePackage.getTargetingsFixed();
        TargetingsCustom targetingsCustom = pricePackage.getTargetingsCustom();
        if (targetingsFixed != null && targetingsFixed.getGeo() != null) {
            fixedGeoOverlap = GeoTreeUtils.areTreesOverlap(getGeoTree(), targetingsFixed.getGeo(),
                    filter.getRegionIds());
        }
        if (targetingsCustom != null && targetingsCustom.getGeo() != null) {
            customGeoOverlap = GeoTreeUtils.areTreesOverlap(getGeoTree(), targetingsCustom.getGeo(),
                    filter.getRegionIds());
        }

        return fixedGeoOverlap || customGeoOverlap;
    }

    private List<PricePackage> getActivePrivatePackages(ClientId clientId,
                                                        Collection<ModelProperty<?, ?>> fieldsToRead) {
        return getActivePrivatePackagesStep(clientId, fieldsToRead)
                .fetch(PRICE_PACKAGE_MAPPER::fromDb);
    }

    private List<PricePackage> getActivePrivatePackages(ClientId clientId,
                                                        Collection<ModelProperty<?, ?>> fieldsToRead,
                                                        Collection<Long> pricePackageIds) {
        return getActivePrivatePackagesStep(clientId, fieldsToRead)
                .and(CPM_PRICE_PACKAGES.PACKAGE_ID.in(pricePackageIds))
                .fetch(PRICE_PACKAGE_MAPPER::fromDb);
    }

    private SelectConditionStep<Record> getActivePrivatePackagesStep(ClientId clientId,
                                                                     Collection<ModelProperty<?, ?>> fieldsToRead) {
        return dslContextProvider.ppcdict()
                .select(PRICE_PACKAGE_MAPPER.getFieldsToRead(fieldsToRead))
                .from(CPM_PRICE_PACKAGES)
                .join(CPM_PRICE_PACKAGE_CLIENTS)
                .on(CPM_PRICE_PACKAGE_CLIENTS.PACKAGE_ID.eq(CPM_PRICE_PACKAGES.PACKAGE_ID))
                .where(CPM_PRICE_PACKAGE_CLIENTS.CLIENT_ID.eq(clientId.asLong()))
                .and(CPM_PRICE_PACKAGE_CLIENTS.IS_ALLOWED.isTrue())
                .and(CPM_PRICE_PACKAGES.IS_PUBLIC.isFalse())
                .and(CPM_PRICE_PACKAGES.IS_ARCHIVED.isFalse())
                .and(CPM_PRICE_PACKAGES.DATE_END.greaterOrEqual(DSL.currentLocalDate()))
                .and(CPM_PRICE_PACKAGES.STATUS_APPROVE.eq(statusApproveToDbFormat(StatusApprove.YES)));
    }

    private List<PricePackage> getActivePublicPackages(ClientId clientId,
                                                       CurrencyCode clientCurrency,
                                                       Collection<ModelProperty<?, ?>> fieldsToRead,
                                                       Collection<Long> pricePackageIds) {
        List<PricePackage> publicPackages = getActivePublicPackagesStep(clientCurrency, fieldsToRead)
                .and(CPM_PRICE_PACKAGES.PACKAGE_ID.in(pricePackageIds))
                .fetch(PRICE_PACKAGE_MAPPER::fromDb);

        return filterPublicPackagesForClient(clientId, publicPackages);
    }

    private List<PricePackage> getActivePublicPackages(ClientId clientId,
                                                       CurrencyCode clientCurrency,
                                                       Collection<ModelProperty<?, ?>> fieldsToRead) {


        List<PricePackage> publicPackages = getActivePublicPackagesStep(clientCurrency, fieldsToRead)
                .fetch(PRICE_PACKAGE_MAPPER::fromDb);

        return filterPublicPackagesForClient(clientId, publicPackages);
    }

    private SelectConditionStep<Record> getActivePublicPackagesStep(CurrencyCode clientCurrency,
                                                                    Collection<ModelProperty<?, ?>> fieldsToRead) {
        return dslContextProvider.ppcdict()
                .select(PRICE_PACKAGE_MAPPER.getFieldsToRead(fieldsToRead))
                .from(CPM_PRICE_PACKAGES)
                .where(CPM_PRICE_PACKAGES.IS_PUBLIC.isTrue())
                .and(CPM_PRICE_PACKAGES.IS_ARCHIVED.isFalse())
                .and(CPM_PRICE_PACKAGES.CURRENCY.eq(currencyToDbFormat(clientCurrency)))
                .and(CPM_PRICE_PACKAGES.STATUS_APPROVE.eq(statusApproveToDbFormat(StatusApprove.YES)))
                .and(CPM_PRICE_PACKAGES.DATE_END.greaterOrEqual(DSL.currentLocalDate()));
    }

    private List<PricePackage> filterPublicPackagesForClient(ClientId clientId, List<PricePackage> publicPackages) {

        var clientsWithPublicPackagesEnabled = getClientsWithPublicPackagesEnabled(Set.of(clientId));
        if (clientsWithPublicPackagesEnabled.size() == 0) {
            return emptyList();
        }

        Set<Long> publicPackagesIds = listToSet(publicPackages, PricePackageForClient::getId);
        Map<Long, List<PricePackageClient>> publicPackageIdToClients =
                getPricePackageIdToClients(dslContextProvider.ppcdict(), publicPackagesIds);

        return StreamEx.of(publicPackages)
                .filter(pricePackage -> clientIsNotBanned(clientId,
                        publicPackageIdToClients.getOrDefault(pricePackage.getId(), emptyList())))
                .toList();
    }

    private Set<ClientId> getClientsWithPublicPackagesEnabled(Set<ClientId> clientIds) {
        // DIRECT-136675: [soc_adv] Фильтровать публичные пакеты для социальных киентов
        var socialAdvertisingEnabledMap = featureService.isEnabledForClientIds(clientIds,
                FeatureName.SOCIAL_ADVERTISING.getName());
        return EntryStream.of(socialAdvertisingEnabledMap)
                .filterValues(socialAdvertisingEnabled -> !socialAdvertisingEnabled)
                .keys()
                .toSet();
    }

    private Map<Long, List<PricePackageClient>> getPricePackageIdToClients(DSLContext dslContext,
                                                                           Set<Long> pricePackageIds) {
        return dslContext
                .select(PRICE_PACKAGE_CLIENT_MAPPER.getFieldsToRead())
                .select(CPM_PRICE_PACKAGE_CLIENTS.PACKAGE_ID)
                .from(CPM_PRICE_PACKAGE_CLIENTS)
                .where(CPM_PRICE_PACKAGE_CLIENTS.PACKAGE_ID.in(pricePackageIds))
                .fetchGroups(CPM_PRICE_PACKAGE_CLIENTS.PACKAGE_ID, PRICE_PACKAGE_CLIENT_MAPPER::fromDb);
    }

    public Map<Long, PricePackageForLock> lockPricePackagesForUpdate(DSLContext dslContext,
                                                                     Collection<Long> packageIds) {
        return dslContext.select(PRICE_PACKAGE_MAPPER.getFieldsToRead(PricePackageForLock.allModelProperties()))
                .from(CPM_PRICE_PACKAGES)
                .where(CPM_PRICE_PACKAGES.PACKAGE_ID.in(packageIds))
                .forUpdate()
                .fetchMap(CPM_PRICE_PACKAGES.PACKAGE_ID, PRICE_PACKAGE_MAPPER::fromDb);
    }

    public List<Long> addPricePackages(List<PricePackage> pricePackages) {
        DSLContext dslContext = dslContextProvider.ppcdict();

        List<Long> pricePackageIds = InsertHelper.addModelsAndReturnIds(dslContext, CPM_PRICE_PACKAGES,
                PRICE_PACKAGE_MAPPER, pricePackages, CPM_PRICE_PACKAGES.PACKAGE_ID);

        Map<Long, List<PricePackageClient>> pricePackageIdToClientIds = EntryStream.zip(pricePackageIds, pricePackages)
                .mapValues(PricePackage::getClients)
                .toMap();
        syncPricePackagesClients(dslContext, pricePackageIdToClientIds);

        return pricePackageIds;
    }

    public void updatePricePackages(DSLContext dslContext,
                                    Collection<AppliedChanges<PricePackage>> applicableAppliedChanges) {
        JooqUpdateBuilder<CpmPricePackagesRecord, PricePackage> ub =
                new JooqUpdateBuilder<>(CPM_PRICE_PACKAGES.PACKAGE_ID, applicableAppliedChanges);

        ub.processProperty(PricePackage.PRODUCT_ID, CPM_PRICE_PACKAGES.PRODUCT_ID);
        ub.processProperty(PricePackage.AUCTION_PRIORITY, CPM_PRICE_PACKAGES.AUCTION_PRIORITY);
        ub.processProperty(PricePackage.TITLE, CPM_PRICE_PACKAGES.TITLE);
        ub.processProperty(PricePackage.TRACKER_URL, CPM_PRICE_PACKAGES.TRACKER_URL);
        ub.processProperty(PricePackage.PRICE, CPM_PRICE_PACKAGES.PRICE);
        ub.processProperty(PricePackage.CURRENCY, CPM_PRICE_PACKAGES.CURRENCY,
                PricePackageMapping::currencyToDbFormat);
        ub.processProperty(PricePackage.ORDER_VOLUME_MIN, CPM_PRICE_PACKAGES.ORDER_VOLUME_MIN);
        ub.processProperty(PricePackage.ORDER_VOLUME_MAX, CPM_PRICE_PACKAGES.ORDER_VOLUME_MAX);
        ub.processProperty(PricePackage.PRICE_MARKUPS, CPM_PRICE_PACKAGES.PRICE_MARKUPS,
                PricePackageMapping::priceMarkupsToDbFormat);
        ub.processProperty(PricePackage.TARGETING_MARKUPS, CPM_PRICE_PACKAGES.TARGETING_MARKUPS,
                PricePackageMapping::targetingMarkupsToDbFormat);
        ub.processProperty(PricePackage.TARGETINGS_FIXED, CPM_PRICE_PACKAGES.TARGETINGS_FIXED,
                PricePackageMapping::targetingsFixedToDbFormat);
        ub.processProperty(PricePackage.TARGETINGS_CUSTOM, CPM_PRICE_PACKAGES.TARGETINGS_CUSTOM,
                PricePackageMapping::targetingsCustomToDbFormat);
        ub.processProperty(PricePackage.STATUS_APPROVE, CPM_PRICE_PACKAGES.STATUS_APPROVE,
                PricePackageMapping::statusApproveToDbFormat);
        ub.processProperty(PricePackage.LAST_UPDATE_TIME, CPM_PRICE_PACKAGES.LAST_UPDATE_TIME);
        ub.processProperty(PricePackage.DATE_START, CPM_PRICE_PACKAGES.DATE_START);
        ub.processProperty(PricePackage.IS_PUBLIC, CPM_PRICE_PACKAGES.IS_PUBLIC,
                RepositoryUtils::booleanToLong);
        ub.processProperty(PricePackage.IS_SPECIAL, CPM_PRICE_PACKAGES.IS_SPECIAL,
                RepositoryUtils::booleanToLong);
        ub.processProperty(PricePackage.IS_CPD, CPM_PRICE_PACKAGES.IS_CPD,
                RepositoryUtils::booleanToLong);
        ub.processProperty(PricePackage.IS_FRONTPAGE, CPM_PRICE_PACKAGES.IS_FRONTPAGE, RepositoryUtils::booleanToLong);
        ub.processProperty(PricePackage.IS_ARCHIVED, CPM_PRICE_PACKAGES.IS_ARCHIVED,
                RepositoryUtils::booleanToLong);
        ub.processProperty(PricePackage.DATE_END, CPM_PRICE_PACKAGES.DATE_END);
        ub.processProperty(PricePackage.CAMPAIGN_AUTO_APPROVE, CPM_PRICE_PACKAGES.IS_CAMPAIGN_AUTO_APPROVE,
                RepositoryUtils::booleanToLong);
        ub.processProperty(PricePackage.IS_DRAFT_APPROVE_ALLOWED, CPM_PRICE_PACKAGES.IS_DRAFT_APPROVE_ALLOWED,
                RepositoryUtils::booleanToLong);
        ub.processProperty(PricePackage.ALLOWED_PAGE_IDS, CPM_PRICE_PACKAGES.ALLOWED_PAGE_IDS,
                PricePackageMapping::allowedPageIdsToDbFormat);
        ub.processProperty(PricePackage.ALLOWED_PROJECT_PARAM_CONDITIONS,
                CPM_PRICE_PACKAGES.ALLOWED_PROJECT_PARAM_CONDITIONS,
                CommonMappings::longListToDbJsonFormat);
        ub.processProperty(PricePackage.ALLOWED_DOMAINS, CPM_PRICE_PACKAGES.ALLOWED_DOMAINS,
                PricePackageMapping::stringListToDbJsonFormat);
        ub.processProperty(PricePackage.ALLOWED_SSP, CPM_PRICE_PACKAGES.ALLOWED_SSP,
                PricePackageMapping::stringListToDbJsonFormat);
        ub.processProperty(PricePackage.ALLOWED_TARGET_TAGS, CPM_PRICE_PACKAGES.ALLOWED_TARGET_TAGS,
                PricePackageMapping::stringListToDbJsonFormat);
        ub.processProperty(PricePackage.ALLOWED_ORDER_TAGS, CPM_PRICE_PACKAGES.ALLOWED_ORDER_TAGS,
                PricePackageMapping::stringListToDbJsonFormat);
        ub.processProperty(PricePackage.ALLOWED_CREATIVE_TEMPLATES, CPM_PRICE_PACKAGES.ALLOWED_CREATIVE_TEMPLATES,
                PricePackageMapping::allowedCreativeTemplatesToDbFormat);
        ub.processProperty(PricePackage.AVAILABLE_AD_GROUP_TYPES, CPM_PRICE_PACKAGES.AVAILABLE_AD_GROUP_TYPES,
                PricePackageMapping::availableAdGroupTypesToDbFormat);
        ub.processProperty(PricePackage.CAMPAIGN_OPTIONS, CPM_PRICE_PACKAGES.CAMPAIGN_OPTIONS,
                PricePackageMapping::campaignOptionsToDbFormat);
        ub.processProperty(PricePackage.BID_MODIFIERS, CPM_PRICE_PACKAGES.BID_MODIFIERS,
                PricePackageMapping::bidModifiersToDbFormat);
        ub.processProperty(PricePackage.CATEGORY_ID, CPM_PRICE_PACKAGES.CATEGORY_ID);

        dslContext.update(CPM_PRICE_PACKAGES)
                .set(ub.getValues())
                .where(CPM_PRICE_PACKAGES.PACKAGE_ID.in(ub.getChangedIds()))
                .execute();

        Map<Long, List<PricePackageClient>> pricePackageIdToClients = StreamEx.of(applicableAppliedChanges)
                .filter(changes -> changes.changed(PricePackage.CLIENTS))
                .mapToEntry(changes -> changes.getModel().getId(),
                        changes -> changes.getNewValue(PricePackage.CLIENTS))
                .toMap();
        syncPricePackagesClients(dslContext, pricePackageIdToClients);
    }

    private void syncPricePackagesClients(DSLContext dslContext,
                                          Map<Long, List<PricePackageClient>> pricePackageIdToClients) {
        Condition deleteRowsCondition = EntryStream.of(pricePackageIdToClients)
                .mapKeyValue((packageId, clients) -> {
                    Condition deleteCondition = CPM_PRICE_PACKAGE_CLIENTS.PACKAGE_ID.eq(packageId);
                    List<Long> clientIds = mapList(clients, PricePackageClient::getClientId);
                    if (!clientIds.isEmpty()) {
                        deleteCondition = deleteCondition.and(CPM_PRICE_PACKAGE_CLIENTS.CLIENT_ID.notIn(clientIds));
                    }
                    return deleteCondition;
                })
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());
        dslContext.delete(CPM_PRICE_PACKAGE_CLIENTS)
                .where(deleteRowsCondition)
                .execute();

        addPackagesToClients(dslContext, pricePackageIdToClients);
    }

    public void addPackagesToClients(DSLContext dslContext,
                                     Map<Long, List<PricePackageClient>> pricePackageIdToClients) {
        InsertHelper<CpmPricePackageClientsRecord> insertHelper =
                new InsertHelper<>(dslContext, CPM_PRICE_PACKAGE_CLIENTS);
        EntryStream.of(pricePackageIdToClients)
                .flatMapValues(Collection::stream)
                .forKeyValue((packageId, packageClient) -> {
                    insertHelper.set(CPM_PRICE_PACKAGE_CLIENTS.PACKAGE_ID, packageId);
                    insertHelper.add(PRICE_PACKAGE_CLIENT_MAPPER, packageClient);
                    insertHelper.newRecord();
                });
        if (insertHelper.hasAddedRecords()) {
            insertHelper.onDuplicateKeyUpdate()
                    .set(CPM_PRICE_PACKAGE_CLIENTS.IS_ALLOWED, MySQLDSL.values(CPM_PRICE_PACKAGE_CLIENTS.IS_ALLOWED));
        }
        insertHelper.executeIfRecordsAdded();
    }

    public void addPackagesToClients(Map<ClientId, Set<Long>> packageIdsByClientId) {
        var pricePackageIdToClients  = EntryStream.of(packageIdsByClientId)
                .flatMapValues(Collection::stream)
                .invert()
                .mapValues(clientId -> new PricePackageClient().withClientId(clientId.asLong()).withIsAllowed(true))
                .grouping();
        addPackagesToClients(dslContextProvider.ppcdict(), pricePackageIdToClients);
    }

    public void deletePackagesFromClients(Map<ClientId, Set<Long>> packageIdsByClientId) {
        Condition deleteCondition = EntryStream.of(packageIdsByClientId)
                .filterValues(packageIds -> !packageIds.isEmpty())
                .mapKeyValue((clientId, packageIds) -> CPM_PRICE_PACKAGE_CLIENTS.CLIENT_ID.eq(clientId.asLong())
                            .and(CPM_PRICE_PACKAGE_CLIENTS.PACKAGE_ID.in(packageIds)))
                .reduce(Condition::or)
                .orElse(DSL.falseCondition());
        dslContextProvider.ppcdict()
                .deleteFrom(CPM_PRICE_PACKAGE_CLIENTS)
                .where(deleteCondition)
                .execute();
    }

    public void deletePricePackages(DSLContext dslContext, Collection<Long> packageIds) {
        if (packageIds.isEmpty()) {
            return;
        }
        dslContext.deleteFrom(CPM_PRICE_PACKAGES)
                .where(CPM_PRICE_PACKAGES.PACKAGE_ID.in(packageIds))
                .execute();
    }

    private static JooqMapperWithSupplier<PricePackage> createPricePackageMapper() {
        return JooqMapperWithSupplierBuilder.builder(PricePackage::new)
                .readProperty(PricePackage.ID, fromField(CPM_PRICE_PACKAGES.PACKAGE_ID))
                .map(property(PricePackage.PRODUCT_ID, CPM_PRICE_PACKAGES.PRODUCT_ID))
                .map(property(PricePackage.AUCTION_PRIORITY, CPM_PRICE_PACKAGES.AUCTION_PRIORITY))
                .map(property(PricePackage.TITLE, CPM_PRICE_PACKAGES.TITLE))
                .map(property(PricePackage.TRACKER_URL, CPM_PRICE_PACKAGES.TRACKER_URL))
                .map(property(PricePackage.PRICE, CPM_PRICE_PACKAGES.PRICE))
                .map(convertibleProperty(PricePackage.CURRENCY, CPM_PRICE_PACKAGES.CURRENCY,
                        PricePackageMapping::currencyFromDbFormat,
                        PricePackageMapping::currencyToDbFormat))
                .map(property(PricePackage.ORDER_VOLUME_MIN, CPM_PRICE_PACKAGES.ORDER_VOLUME_MIN))
                .map(property(PricePackage.ORDER_VOLUME_MAX, CPM_PRICE_PACKAGES.ORDER_VOLUME_MAX))
                .map(convertibleProperty(PricePackage.PRICE_MARKUPS, CPM_PRICE_PACKAGES.PRICE_MARKUPS,
                        PricePackageMapping::priceMarkupsFromDbFormat,
                        PricePackageMapping::priceMarkupsToDbFormat))
                .map(convertibleProperty(PricePackage.TARGETINGS_FIXED, CPM_PRICE_PACKAGES.TARGETINGS_FIXED,
                        PricePackageMapping::targetingsFixedFromDbFormat,
                        PricePackageMapping::targetingsFixedToDbFormat))
                .map(convertibleProperty(PricePackage.TARGETINGS_CUSTOM, CPM_PRICE_PACKAGES.TARGETINGS_CUSTOM,
                        PricePackageMapping::targetingsCustomFromDbFormat,
                        PricePackageMapping::targetingsCustomToDbFormat))
                .map(convertibleProperty(PricePackage.TARGETING_MARKUPS, CPM_PRICE_PACKAGES.TARGETING_MARKUPS,
                        PricePackageMapping::targetingMarkupsFromDbFormat,
                        PricePackageMapping::targetingMarkupsToDbFormat))
                .map(convertibleProperty(PricePackage.STATUS_APPROVE, CPM_PRICE_PACKAGES.STATUS_APPROVE,
                        PricePackageMapping::statusApproveFromDbFormat,
                        PricePackageMapping::statusApproveToDbFormat))
                .map(property(PricePackage.LAST_UPDATE_TIME, CPM_PRICE_PACKAGES.LAST_UPDATE_TIME))
                .map(property(PricePackage.DATE_START, CPM_PRICE_PACKAGES.DATE_START))
                .map(property(PricePackage.DATE_END, CPM_PRICE_PACKAGES.DATE_END))
                .map(booleanProperty(PricePackage.IS_PUBLIC, CPM_PRICE_PACKAGES.IS_PUBLIC))
                .map(booleanProperty(PricePackage.IS_SPECIAL, CPM_PRICE_PACKAGES.IS_SPECIAL))
                .map(booleanProperty(PricePackage.IS_CPD, CPM_PRICE_PACKAGES.IS_CPD))
                .map(booleanProperty(PricePackage.IS_FRONTPAGE, CPM_PRICE_PACKAGES.IS_FRONTPAGE))
                .map(booleanProperty(PricePackage.IS_ARCHIVED, CPM_PRICE_PACKAGES.IS_ARCHIVED))
                .map(booleanProperty(PricePackage.CAMPAIGN_AUTO_APPROVE, CPM_PRICE_PACKAGES.IS_CAMPAIGN_AUTO_APPROVE))
                .map(booleanProperty(PricePackage.IS_DRAFT_APPROVE_ALLOWED,
                        CPM_PRICE_PACKAGES.IS_DRAFT_APPROVE_ALLOWED))
                .map(convertibleProperty(PricePackage.AVAILABLE_AD_GROUP_TYPES,
                        CPM_PRICE_PACKAGES.AVAILABLE_AD_GROUP_TYPES,
                        PricePackageMapping::availableAdGroupTypesFromDbFormat,
                        PricePackageMapping::availableAdGroupTypesToDbFormat))
                .map(convertibleProperty(PricePackage.CAMPAIGN_OPTIONS, CPM_PRICE_PACKAGES.CAMPAIGN_OPTIONS,
                        PricePackageMapping::campaignOptionsFromDbFormat,
                        PricePackageMapping::campaignOptionsToDbFormat))
                .map(convertibleProperty(PricePackage.BID_MODIFIERS, CPM_PRICE_PACKAGES.BID_MODIFIERS,
                        PricePackageMapping::bidModifiersFromDbFormat,
                        PricePackageMapping::bidModifiersToDbFormat))
                .map(convertibleProperty(PricePackage.ALLOWED_PAGE_IDS, CPM_PRICE_PACKAGES.ALLOWED_PAGE_IDS,
                        PricePackageMapping::allowedPageIdsFromDbFormat,
                        PricePackageMapping::allowedPageIdsToDbFormat))
                .map(convertibleProperty(PricePackage.ALLOWED_PROJECT_PARAM_CONDITIONS, CPM_PRICE_PACKAGES.ALLOWED_PROJECT_PARAM_CONDITIONS,
                        CommonMappings::longListFromDbJsonFormat,
                        CommonMappings::longListToDbJsonFormat))
                .map(convertibleProperty(PricePackage.ALLOWED_TARGET_TAGS, CPM_PRICE_PACKAGES.ALLOWED_TARGET_TAGS,
                        PricePackageMapping::stringListFromDbJsonFormat,
                        PricePackageMapping::stringListToDbJsonFormat))
                .map(convertibleProperty(PricePackage.ALLOWED_ORDER_TAGS, CPM_PRICE_PACKAGES.ALLOWED_ORDER_TAGS,
                        PricePackageMapping::stringListFromDbJsonFormat,
                        PricePackageMapping::stringListToDbJsonFormat))
                .map(convertibleProperty(PricePackage.ALLOWED_DOMAINS, CPM_PRICE_PACKAGES.ALLOWED_DOMAINS,
                        PricePackageMapping::stringListFromDbJsonFormat,
                        PricePackageMapping::stringListToDbJsonFormat))
                .map(convertibleProperty(PricePackage.ALLOWED_SSP, CPM_PRICE_PACKAGES.ALLOWED_SSP,
                        PricePackageMapping::stringListFromDbJsonFormat,
                        PricePackageMapping::stringListToDbJsonFormat))
                .map(convertibleProperty(PricePackage.ALLOWED_CREATIVE_TEMPLATES,
                        CPM_PRICE_PACKAGES.ALLOWED_CREATIVE_TEMPLATES,
                        PricePackageMapping::allowedCreativeTemplatesFromDbFormat,
                        PricePackageMapping::allowedCreativeTemplatesToDbFormat))
                .map(property(PricePackage.CATEGORY_ID, CPM_PRICE_PACKAGES.CATEGORY_ID))
                .build();
    }

    private static JooqMapperWithSupplier<PricePackageClient> createPricePackageClientMapper() {
        return JooqMapperWithSupplierBuilder.builder(PricePackageClient::new)
                .map(property(PricePackageClient.CLIENT_ID, CPM_PRICE_PACKAGE_CLIENTS.CLIENT_ID))
                .map(booleanProperty(PricePackageClient.IS_ALLOWED, CPM_PRICE_PACKAGE_CLIENTS.IS_ALLOWED))
                .build();
    }

    private static JooqMapperWithSupplier<PricePackageCategory> createPricePackageCategoryMapper() {
        return JooqMapperWithSupplierBuilder.builder(PricePackageCategory::new)
                .map(property(PricePackageCategory.ID, CPM_PRICE_PACKAGE_CATEGORIES.ID))
                .map(property(PricePackageCategory.PARENT_ID, CPM_PRICE_PACKAGE_CATEGORIES.PARENT_ID))
                .map(property(PricePackageCategory.ORDER, CPM_PRICE_PACKAGE_CATEGORIES.ORDER))
                .map(property(PricePackageCategory.TITLE, CPM_PRICE_PACKAGE_CATEGORIES.TITLE))
                .map(property(PricePackageCategory.DESCRIPTION, CPM_PRICE_PACKAGE_CATEGORIES.DESCRIPTION))
                .build();
    }

    private GeoTree getGeoTree() {
        return pricePackageGeoTree.getGeoTree();
    }

    public static class PricePackagesWithTotalCount {
        private final List<PricePackage> pricePackages;
        private final int totalCount;

        PricePackagesWithTotalCount(List<PricePackage> pricePackages, int totalCount) {
            this.pricePackages = pricePackages;
            this.totalCount = totalCount;
        }

        public List<PricePackage> getPricePackages() {
            return pricePackages;
        }

        public int getTotalCount() {
            return totalCount;
        }
    }

    public List<PricePackage> getAllPricePackages() {
        return dslContextProvider.ppcdict()
                .select(PRICE_PACKAGE_MAPPER.getFieldsToRead(PricePackageWithoutClients.allModelProperties()))
                .from(CPM_PRICE_PACKAGES)
                .orderBy(CPM_PRICE_PACKAGES.PACKAGE_ID.desc())
                .fetch(PRICE_PACKAGE_MAPPER::fromDb);
    }

    public List<PricePackageCategory> getAllPricePackageCategories() {
        return dslContextProvider.ppcdict()
                .select(PRICE_PACKAGE_CATEGORY_MAPPER.getFieldsToRead(PricePackageCategory.allModelProperties()))
                .from(CPM_PRICE_PACKAGE_CATEGORIES)
                .orderBy(CPM_PRICE_PACKAGE_CATEGORIES.ID.desc())
                .fetch(PRICE_PACKAGE_CATEGORY_MAPPER::fromDb);
    }
}
