package ru.yandex.direct.core.entity.pricepackage.service;

import java.time.ZoneOffset;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.StatusApprove;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.pricepackage.service.validation.PricePackageAddValidationService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.ModelsPreValidatedStep;
import ru.yandex.direct.operation.add.SimpleAbstractAddOperation;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.time.LocalDateTime.now;
import static ru.yandex.direct.core.entity.pricepackage.service.PricePackageOperationUtils.processVideoFrontpagePackage;

@ParametersAreNonnullByDefault
public class PricePackageAddOperation extends SimpleAbstractAddOperation<PricePackage, Long> {

    private final PricePackageRepository repository;
    private final PricePackageAddValidationService validationService;
    private final GeoTree geoTree;
    private final PricePackageGeoProcessor geoProcessor;
    private final User operator;

    PricePackageAddOperation(Applicability applicability,
                             List<PricePackage> models,
                             PricePackageRepository repository,
                             PricePackageAddValidationService validationService,
                             PricePackageService pricePackageService,
                             User operator) {
        super(applicability, models);
        this.repository = repository;
        this.validationService = validationService;
        this.geoTree = pricePackageService.getGeoTree();
        this.geoProcessor = pricePackageService.getGeoProcessor();
        this.operator = operator;
    }

    @Override
    protected void validate(ValidationResult<List<PricePackage>, Defect> preValidationResult) {
        new ItemValidationBuilder<>(preValidationResult)
                .checkBy(pricePackages -> validationService.validatePricePackages(pricePackages, geoTree, operator));
        // TODO: DIRECT-103777: добавить проверку, что lastUpdateTime, statusApprove и geoExpanded не установленны
    }

    @Override
    protected void onPreValidated(ModelsPreValidatedStep<PricePackage> modelsPreValidatedStep) {
        super.onPreValidated(modelsPreValidatedStep);
        for (PricePackage pricePackage : modelsPreValidatedStep.getPreValidModelsMap().values()) {
            geoProcessor.expandGeo(pricePackage.getTargetingsFixed());
            geoProcessor.expandGeo(pricePackage.getTargetingsCustom());
        }
    }

    @Override
    protected void beforeExecution(Map<Integer, PricePackage> validModelsMapToApply) {
        for (PricePackage pricePackage : validModelsMapToApply.values()) {
            pricePackage.setLastUpdateTime(now(ZoneOffset.UTC));
            pricePackage.setStatusApprove(StatusApprove.NEW);
            pricePackage.setIsArchived(false);
            pricePackage.setTitle(pricePackage.getTitle().trim());
            if (pricePackage.isFrontpageVideoPackage()) {
                processVideoFrontpagePackage(pricePackage);
            }
        }
    }

    @Override
    protected List<Long> execute(List<PricePackage> validModelsMapToApply) {
        return repository.addPricePackages(validModelsMapToApply);
    }
}
