package ru.yandex.direct.core.entity.pricepackage.service;

import java.time.Duration;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageForLock;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.pricepackage.service.validation.PricePackageDeleteValidationService;
import ru.yandex.direct.core.grut.replication.GrutApiService;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.operationwithid.AbstractOperationWithId;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.common.db.PpcPropertyNames.UPDATE_AUCTION_PRIORITY_IN_GRUT;
import static ru.yandex.direct.core.entity.pricepackage.service.PricePackageOperationUtils.checkPricePackagesLocked;
import static ru.yandex.direct.core.entity.pricepackage.service.PricePackageOperationUtils.checkPricePackagesNotApproved;

@ParametersAreNonnullByDefault
public class PricePackageDeleteOperation extends AbstractOperationWithId {

    private static final Logger logger = LoggerFactory.getLogger(PricePackageDeleteOperation.class);
    private final DslContextProvider dslContextProvider;
    private final PricePackageRepository pricePackageRepository;
    private final PricePackageDeleteValidationService pricePackageDeleteValidationService;
    private final GrutApiService grutApiService;
    private final PpcProperty<Boolean> updateAuctionPriorityInGrutProperty;

    public PricePackageDeleteOperation(Applicability applicability, List<Long> modelIds,
                                       DslContextProvider dslContextProvider,
                                       PricePackageRepository pricePackageRepository,
                                       PricePackageDeleteValidationService pricePackageDeleteValidationService,
                                       GrutApiService grutApiService,
                                       PpcPropertiesSupport ppcPropertiesSupport) {
        super(applicability, modelIds);
        this.dslContextProvider = dslContextProvider;
        this.pricePackageRepository = pricePackageRepository;
        this.pricePackageDeleteValidationService = pricePackageDeleteValidationService;
        this.grutApiService = grutApiService;
        this.updateAuctionPriorityInGrutProperty =
                ppcPropertiesSupport.get(UPDATE_AUCTION_PRIORITY_IN_GRUT, Duration.ofMinutes(5));
    }

    @Override
    protected ValidationResult<List<Long>, Defect> validate(List<Long> ids) {
        return pricePackageDeleteValidationService.validate(ids);
    }

    @Override
    protected void execute(List<Long> ids) {
        dslContextProvider.ppcdictTransaction(configuration -> {
            if (updateAuctionPriorityInGrutProperty.getOrDefault(false)) {
                logger.info("Delete price packages in GrUT " + ids);
                grutApiService.getPricePackageGrutApi().deleteObjects(ids, true);
            }
            Map<Long, PricePackageForLock> lockedPricePackages = pricePackageRepository.lockPricePackagesForUpdate(configuration.dsl(), ids);
            checkPricePackagesLocked(ids, lockedPricePackages);
            checkPricePackagesNotApproved(lockedPricePackages.values());
            pricePackageRepository.deletePricePackages(configuration.dsl(), ids);
        });
    }

}
