package ru.yandex.direct.core.entity.pricepackage.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageForLock;
import ru.yandex.direct.core.entity.pricepackage.model.StatusApprove;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.core.entity.creative.model.CreativeType.CPM_VIDEO_CREATIVE;

class PricePackageOperationUtils {
    public static final Map<CreativeType, List<Long>> VIDEO_FRONTPAGE_CREATIVE_TEMPLATES =
            Map.of(CPM_VIDEO_CREATIVE, List.of(406L));
    private static final List<String> VIDEO_FRONTPAGE_CREATIVE_TAGS =
            List.of("portal-trusted");

    private PricePackageOperationUtils() {
    }

    static void checkPricePackagesLocked(Collection<Long> pricePackageIds,
                                         Map<Long, PricePackageForLock> lockedPricePackages) {
        boolean allPricePackagesLocked = pricePackageIds.stream().allMatch(lockedPricePackages::containsKey);
        checkState(allPricePackagesLocked,
                "Failed to lock some price package for update");
    }

    static void checkPricePackagesNotChanged(Collection<PricePackageForLock> pricePackages,
                                             Map<Long, LocalDateTime> userTimestampsByPackageId) {
        boolean allPricePackagesNotChanged = pricePackages.stream().allMatch(pricePackage -> {
            var userTimestamp = userTimestampsByPackageId.get(pricePackage.getId());
            return pricePackage.getLastUpdateTime().equals(userTimestamp);
        });
        checkState(allPricePackagesNotChanged,
                "Some of the price packages last update time differs from last update time passed by the user");
    }

    static void checkPricePackagesNotApproved(Collection<PricePackageForLock> pricePackages) {
        boolean allPricePackagesNotApproved = pricePackages.stream()
                .allMatch(pricePackage -> pricePackage.getStatusApprove() != StatusApprove.YES);
        checkState(allPricePackagesNotApproved,
                "Some of the price packages are approved");
    }

    /**
     * Для прайсового видео на главной на создании-редактировании проставить значения по умолчанию.
     * @param pricePackage
     */
    static void processVideoFrontpagePackage(PricePackage pricePackage) {
        pricePackage.setAllowedCreativeTemplates(VIDEO_FRONTPAGE_CREATIVE_TEMPLATES);
        pricePackage.setAllowedOrderTags(processVideoFrontpagePackageTag(pricePackage.getAllowedOrderTags()));
        pricePackage.setAllowedTargetTags(processVideoFrontpagePackageTag(pricePackage.getAllowedTargetTags()));
    }

    static List<String> processVideoFrontpagePackageTag(List<String> tag) {
        if (tag == null || tag.isEmpty()) {
            return VIDEO_FRONTPAGE_CREATIVE_TAGS;
        }
        return StreamEx.of(tag)
                .append(StreamEx.of(VIDEO_FRONTPAGE_CREATIVE_TAGS).filter(it -> !tag.contains(it)))
                .toList();
    }
}
