package ru.yandex.direct.core.entity.pricepackage.service;

import java.time.LocalDate;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageCampaignOptions;
import ru.yandex.direct.core.entity.pricepackage.model.PriceRetargetingCondition;
import ru.yandex.direct.core.entity.pricepackage.model.StatusApprove;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingMarkup;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsCustom;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsFixed;
import ru.yandex.direct.core.entity.pricepackage.service.validation.PricePackageModelChangesValidationContext;
import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.model.ModelProperty;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.core.entity.user.utils.UserUtil.hasOneOfRoles;
import static ru.yandex.direct.rbac.RbacRole.SUPER;
import static ru.yandex.direct.rbac.RbacRole.SUPERREADER;
import static ru.yandex.direct.rbac.RbacRole.SUPPORT;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.DateTimeUtils.inFutureOrToday;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Одновременное изменение статуса с другими полями эквивалентно последовательности:
 * 1. изменения других полей
 * 2. изменение статуса
 * т.е. "другие поля" проверяются относительно старого statusApprove, а не нового.
 */
public class PricePackagePermissionUtils {

    private PricePackagePermissionUtils() {
    }

    private static final Map<Predicate<User>, Map<StatusApprove, Set<StatusApprove>>>
            STATUS_APPROVE_CHANGES_ALLOWED_BY_PERMISSION = ImmutableMap.of(
            PricePackagePermissionUtils::canManagePricePackages, ImmutableMap.of(
                    StatusApprove.NEW, ImmutableSet.of(StatusApprove.WAITING),
                    StatusApprove.NO, ImmutableSet.of(StatusApprove.WAITING)
            ),
            PricePackagePermissionUtils::canApprovePricePackages, ImmutableMap.of(
                    StatusApprove.WAITING, ImmutableSet.of(StatusApprove.NO, StatusApprove.YES),
                    StatusApprove.YES, ImmutableSet.of(StatusApprove.NO)
            )
    );

    public static boolean canViewPricePackages(User operator) {
        return hasOneOfRoles(operator, SUPERREADER)
                || canManagePricePackages(operator)
                || canApprovePricePackages(operator)
                || canManagePricePackageClients(operator);
    }

    public static boolean canManagePricePackageClients(User operator) {
        return hasOneOfRoles(operator, SUPER, SUPPORT)
                || Boolean.TRUE.equals(operator.getCanManagePricePackages());
    }

    public static boolean canManagePricePackages(User operator) {
        return hasOneOfRoles(operator, SUPER)
                || Boolean.TRUE.equals(operator.getCanManagePricePackages());
    }

    public static boolean canApprovePricePackages(User operator) {
        return hasOneOfRoles(operator, SUPER, SUPPORT)
                || Boolean.TRUE.equals(operator.getCanApprovePricePackages());
    }

    public static boolean canChangeStatusApprove(
            PricePackage pricePackage, PricePackageModelChangesValidationContext context,
            StatusApprove newStatusApprove) {

        return EntryStream.of(STATUS_APPROVE_CHANGES_ALLOWED_BY_PERMISSION)
                .filterKeys(permissionPredicate -> permissionPredicate.test(context.getOperator()))
                .values()
                .map(statusApproveAllowedChanges -> statusApproveAllowedChanges.get(pricePackage.getStatusApprove()))
                .filter(Objects::nonNull)
                .anyMatch(statusApproveAllowedChangesTo -> statusApproveAllowedChangesTo.contains(newStatusApprove));
    }

    public static boolean isDateStartOnlyExpand(
            PricePackage pricePackage, PricePackageModelChangesValidationContext context, LocalDate newDateStart) {
        return newDateStart.isBefore(pricePackage.getDateStart())
                // Если пакет не заапрувлен, то можно изменять дату в любом направлении (хоть расширять, хоть сужать)
                || pricePackage.getStatusApprove() != StatusApprove.YES;
    }

    public static boolean isDateEndOnlyExpand(
            PricePackage pricePackage, PricePackageModelChangesValidationContext context, LocalDate newDateEnd) {
        return newDateEnd.isAfter(pricePackage.getDateEnd())
                // Если пакет не заапрувлен, то можно изменять дату в любом направлении (хоть расширять, хоть сужать)
                || pricePackage.getStatusApprove() != StatusApprove.YES;
    }

    public static boolean pricePackageIsNotExpired(
            PricePackage pricePackage, PricePackageModelChangesValidationContext context, Object propertyNewValue) {
        return inFutureOrToday(pricePackage.getDateEnd());
    }

    public static boolean retargetingCategoriesAmountIsNotLessLowerLimit(PriceRetargetingCondition retargetingCondition) {
        List<Long> goalIds = nvl(retargetingCondition.getCryptaSegments(), emptyList());
        Set<GoalType> goalTypes = listToSet(mapList(goalIds, Goal::computeType));
        // объединяем жанры и категории
        if (goalTypes.containsAll(List.of(GoalType.CONTENT_GENRE, GoalType.CONTENT_CATEGORY))) {
            goalTypes.remove(GoalType.CONTENT_GENRE);
        }
        // считаем метрику
        if (nvl(retargetingCondition.getAllowMetrikaSegments(), false)) {
            goalTypes.add(GoalType.GOAL);
        }
        // считаем аудитории
        if (nvl(retargetingCondition.getAllowAudienceSegments(), false)) {
            goalTypes.add(GoalType.AUDIENCE);
        }
        return (goalTypes.size() >= nvl(retargetingCondition.getLowerCryptaTypesCount(), 0));
    }

    public static boolean retargetingCategoriesAmountIsNotLessLowerLimit(
            PricePackage pricePackage, PricePackageModelChangesValidationContext context,
            TargetingsCustom targetingsCustom) {
        return retargetingCategoriesAmountIsNotLessLowerLimit(targetingsCustom.getRetargetingCondition());
    }

    public static boolean retargetingCategoriesAmountIsNotGreaterUpperLimit(List<Long> cryptaSegments,
                                                                            PriceRetargetingCondition retargetingCondition) {
        int upperCryptaTypesCount = nvl(retargetingCondition.getUpperCryptaTypesCount(), 0);
        if (upperCryptaTypesCount == 0) {
            return true;
        }
        List<Long> goalIds = nvl(cryptaSegments, emptyList());
        Set<GoalType> goalTypes = listToSet(mapList(goalIds, Goal::computeType));
        // объединяем жанры и категории
        if (goalTypes.containsAll(List.of(GoalType.CONTENT_GENRE, GoalType.CONTENT_CATEGORY))) {
            goalTypes.remove(GoalType.CONTENT_GENRE);
        }
        return goalTypes.size() <= upperCryptaTypesCount;
    }

    public static boolean retargetingCategoriesAmountIsNotGreaterUpperLimit(PricePackage pricePackage,
                                                                            PricePackageModelChangesValidationContext context, TargetingsFixed targetingsFixed) {
        return retargetingCategoriesAmountIsNotGreaterUpperLimit(targetingsFixed.getCryptaSegments(),
                pricePackage.getTargetingsCustom().getRetargetingCondition());
    }

    public static boolean allowExpandedDesktopCreativeAllowedToRemove(
            PricePackage pricePackage, PricePackageModelChangesValidationContext context,
            TargetingsFixed targetingsFixed) {
        if (pricePackage.getStatusApprove() != StatusApprove.YES) {
            return true;
        } else {
            var newValue = targetingsFixed.getAllowExpandedDesktopCreative();
            return pricePackage.getTargetingsFixed().getAllowExpandedDesktopCreative() == newValue ||
                    !pricePackage.isFrontpagePackage() && newValue == null;
        }
    }

    public static boolean equalsIgnoringGeoExpanded(TargetingsFixed t1,
                                                    TargetingsFixed t2) {
        if (t1 == null || t2 == null) {
            return false;
        }
        Set<ModelProperty<?, ?>> ignoredProperties = Set.of(
                TargetingsFixed.GEO_EXPANDED,
                // наличие поля зависит от типа пакета, проверяется в allowExpandedDesktopCreativeAllowedToRemove()
                TargetingsFixed.ALLOW_EXPANDED_DESKTOP_CREATIVE
        );
        for (var modelProperty : TargetingsFixed.allModelProperties()) {

            if (ignoredProperties.contains(modelProperty)) {
                continue;
            }
            var v1 = modelProperty.getRaw(t1);
            var v2 = modelProperty.getRaw(t2);
            if (notEqualsConsideringUnorderedCollections(v1, v2)) {
                return false;
            }
        }
        return true;
    }

    public static boolean equalsIgnoringGeoExpanded(TargetingsCustom t1,
                                                    TargetingsCustom t2) {
        if (t1 == null || t2 == null) {
            return false;
        }
        for (var modelProperty : TargetingsCustom.allModelProperties()) {
            if (modelProperty == TargetingsCustom.GEO_EXPANDED) {
                continue;
            }
            var v1 = modelProperty.getRaw(t1);
            var v2 = modelProperty.getRaw(t2);
            if (notEqualsConsideringUnorderedCollections(v1, v2)) {
                return false;
            }
        }
        return true;
    }

    private static boolean notEqualsConsideringUnorderedCollections(Object v1, Object v2) {
        if (v1 instanceof Collection<?> && v2 instanceof Collection<?>) {
            return !new HashSet<Object>((Collection<?>) v1).equals(new HashSet<Object>((Collection<?>) v2));
        }
        return !Objects.equals(v1, v2);
    }

    public static boolean isNewCreativeTemplatesOnlyExpand(
            PricePackage pricePackage,
            PricePackageModelChangesValidationContext context,
            Map<CreativeType, List<Long>> newAllowedCreativeTemplates) {
        // Если пакет не заапрувлен, то можно изменять шаблоны в любом направлении (хоть расширять, хоть сужать)
        if (pricePackage.getStatusApprove() != StatusApprove.YES) {
            return true;
        }

        // Проверяем детальнее по типам креативов
        for (CreativeType creativeType : newAllowedCreativeTemplates.keySet()) {
            var newIds = newAllowedCreativeTemplates.getOrDefault(creativeType, emptyList());
            var oldIds = pricePackage.getAllowedCreativeTemplates().getOrDefault(creativeType, emptyList());

            // Если новый набор содержит старый и такого же или больше размера - то так изменять можно
            if (!newIds.containsAll(oldIds)) {
                return false;
            }
        }
        return true;
    }

    //удалять наценку за таргетинг нельзя, добавлять можно, давай пока так
    public static boolean targetingsMarkupsCanBeChanged(
            PricePackage pricePackage,
            PricePackageModelChangesValidationContext context,
            List<TargetingMarkup> newMarkups) {

        if (pricePackage.getStatusApprove() == StatusApprove.NEW) {
            return true;
        }

        Set<TargetingMarkup> newTargetings = new HashSet<>(newMarkups);

        return newTargetings.containsAll(nvl(pricePackage.getTargetingMarkups(), emptyList()));
    }

    // удалять параметры проектов нельзя, добавлять можно
    public static boolean allowedProjectParamConditionsCanBeChanged(
            PricePackage pricePackage,
            PricePackageModelChangesValidationContext context,
            List<Long> newProjectParamConditions) {

        if (pricePackage.getStatusApprove() == StatusApprove.NEW) {
            return true;
        }

        Set<Long> newConditions = new HashSet<>(newProjectParamConditions);

        return newConditions.containsAll(nvl(pricePackage.getAllowedProjectParamConditions(), emptyList()));
    }

    public static boolean campaignOptionsCanBeChanged(
            PricePackage pricePackage,
            PricePackageCampaignOptions newOptions) {

        // Если пакет не заапрувлен, то можно изменять
        if (pricePackage.getStatusApprove() != StatusApprove.YES) {
            return true;
        }

        //На запрувленом продукте разрешать только включать, без возможности отключения,
        // такие настройки продукта: Brand Safety, Brand Lift, черный список площадок.
        var old = pricePackage.getCampaignOptions();
        return (!old.getAllowBrandSafety() || newOptions.getAllowBrandSafety())
                && (!old.getAllowBrandLift() || newOptions.getAllowBrandLift())
                && (!old.getAllowDisabledVideoPlaces() || newOptions.getAllowDisabledVideoPlaces())
                && (!old.getAllowDisabledPlaces() || newOptions.getAllowDisabledPlaces())
                && Objects.equals(old.getShowsFrequencyLimit(), newOptions.getShowsFrequencyLimit());
    }
}
