package ru.yandex.direct.core.entity.pricepackage.service;

import java.time.LocalDate;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.beust.jcommander.internal.Nullable;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsSettings;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPricePackage;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageForClient;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageWithoutClients;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackagesFilter;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.util.GeoTreeConverter;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.regions.GeoTree;

import static ru.yandex.direct.core.entity.adgroup.service.bstags.AdGroupBsTagsUtils.entryStreamOfAdGroups;
import static ru.yandex.direct.core.entity.campaign.repository.filter.CampaignFilterFactory.campaignIdsFilter;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterToSet;
import static ru.yandex.direct.utils.FunctionalUtils.flatMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
public class PricePackageService {

    public static final AdGroupBsTagsSettings DEFAULT_BS_TAGS = new AdGroupBsTagsSettings.Builder().build();

    private final PricePackageGeoTree pricePackageGeoTree;
    private final PricePackageRepository pricePackageRepository;
    private final CampaignTypedRepository campaignTypedRepository;
    private final FeatureService featureService;
    private final ClientService clientService;
    private final ShardHelper shardHelper;
    private final CampaignRepository campaignRepository;

    @Autowired
    public PricePackageService(PricePackageGeoTree pricePackageGeoTree,
                               PricePackageRepository pricePackageRepository,
                               FeatureService featureService,
                               ClientService clientService,
                               CampaignTypedRepository campaignTypedRepository,
                               ShardHelper shardHelper,
                               CampaignRepository campaignRepository) {
        this.pricePackageGeoTree = pricePackageGeoTree;
        this.pricePackageRepository = pricePackageRepository;
        this.featureService = featureService;
        this.clientService = clientService;
        this.campaignTypedRepository = campaignTypedRepository;
        this.shardHelper = shardHelper;
        this.campaignRepository = campaignRepository;
    }

    /**
     * В прайсовых используем Российское гео-дерево, потому что транслокальность сложно поддержать в прайсовых.
     * С юридической точки зрения использовать Российское гео-дерево в прайсовых - это ок.
     */
    public GeoTree getGeoTree() {
        return pricePackageGeoTree.getGeoTree();
    }

    public GeoTreeConverter getGeoTreeConverter() {
        return pricePackageGeoTree.getGeoTreeConverter();
    }

    public PricePackageGeoProcessor getGeoProcessor() {
        return pricePackageGeoTree.getGeoProcessor();
    }

    public List<PricePackageForClient> getActivePricePackagesForClient(ClientId clientId, PricePackagesFilter filter) {
        Currency clientCurrency = clientService.getWorkCurrency(clientId);
        return featureService.isEnabledForClientId(clientId, FeatureName.CPM_PRICE_CAMPAIGN) ?
                pricePackageRepository.getActivePricePackagesForClient(clientId, clientCurrency.getCode(), filter) :
                Collections.emptyList();
    }

    public List<PricePackageWithoutClients> getActivePricePackagesWithoutClients(ClientId clientId,
                                                                                 Set<Long> pricePackageIds) {
        Currency clientCurrency = clientService.getWorkCurrency(clientId);
        return featureService.isEnabledForClientId(clientId, FeatureName.CPM_PRICE_CAMPAIGN) ?
                pricePackageRepository.getActivePricePackagesWithoutClients(clientId, clientCurrency.getCode(),
                        pricePackageIds) :
                Collections.emptyList();
    }
    public Map<Long, PricePackage> getPricePackageByCampaignIds(ClientId clientId, Collection<Long> campaignIds) {
        int shard = clientService.getShardByClientIdStrictly(clientId);
        return getPricePackageByCampaignIds(shard, campaignIds);
    }

    public Map<Long, PricePackage> getPricePackageByCampaigns(int shard, Collection<Campaign> campaigns) {
        List<Long> cpmPriceCampaignIds = filterAndMapList(campaigns,
                c -> c.getType() == CampaignType.CPM_PRICE, Campaign::getId);
        List<CampaignWithPricePackage> cpmPriceCampaigns =
                campaignTypedRepository.getSafely(shard, cpmPriceCampaignIds, CampaignWithPricePackage.class);

        var pricePackageIds = mapList(cpmPriceCampaigns, CampaignWithPricePackage::getPricePackageId);
        var pricePackages = pricePackageRepository.getPricePackages(pricePackageIds);
        var result = new HashMap<Long, PricePackage>();
        cpmPriceCampaigns.forEach(campaign ->
                result.put(campaign.getId(), pricePackages.get(campaign.getPricePackageId())));
        return result;
    }

    public PricePackage getPricePackage(Long pricePackageId) {
        return pricePackageRepository.getPricePackages(List.of(pricePackageId)).get(pricePackageId);
    }

    public Map<Long, PricePackage> getPricePackages(Collection<Long> packageIds) {
        return pricePackageRepository.getPricePackages(packageIds);
    }

    public Map<Long, PricePackage> getPricePackageByCampaignIds(int shard, Collection<Long> campaignIds) {
        List<CampaignWithPricePackage> priceCampaigns = campaignTypedRepository.getSafely(shard,
                campaignIdsFilter(campaignIds), CampaignWithPricePackage.class);
        Set<Long> pricePackageIds = priceCampaigns.stream()
                .map(CampaignWithPricePackage::getPricePackageId)
                .collect(Collectors.toSet());
        Map<Long, PricePackage> pricePackages = pricePackageRepository.getPricePackages(pricePackageIds);
        var result = new HashMap<Long, PricePackage>();
        priceCampaigns.forEach(campaign ->
                result.put(campaign.getId(), pricePackages.get(campaign.getPricePackageId())));
        return result;
    }

    public Map<Long, PricePackage> getPricePackageByCampaignIds(Collection<Long> campaignIds) {
        Map<Long, PricePackage> result = new HashMap<>();
        shardHelper.groupByShard(campaignIds, ShardKey.CID)
                .forEach((shard, cids) -> {
                    result.putAll(getPricePackageByCampaignIds(shard, cids));
                });
        return result;
    }

    public IdentityHashMap<AdGroup, AdGroupBsTagsSettings> getAdGroupBsTagsSettings(
            List<? extends AdGroup> adGroups, ClientId clientId)
    {
        List<Long> campaignIds = mapList(adGroups, AdGroup::getCampaignId);

        Map<Long, PricePackage> cidToPricePackage = getPricePackageByCampaignIds(clientId, campaignIds);
        Map<Long, AdGroupBsTagsSettings> cidToTagSettings =
                EntryStream.of(cidToPricePackage)
                        .mapValues(pricePackage -> new AdGroupBsTagsSettings.Builder()
                                .withAllowedTargetTags(pricePackage.getAllowedTargetTags())
                                .withDefaultTargetTags(pricePackage.getAllowedTargetTags())
                                .withAllowedPageGroupTags(pricePackage.getAllowedOrderTags())
                                .withDefaultPageGroupTags(pricePackage.getAllowedOrderTags())
                                .build())
                        .toMap();

        return entryStreamOfAdGroups(adGroups)
                .mapValues(AdGroup::getCampaignId)
                .mapValues(packageId -> cidToTagSettings.getOrDefault(packageId, DEFAULT_BS_TAGS))
                .toCustomMap(IdentityHashMap::new);
    }

    public List<PricePackage> getAllBLOffPricePackages() {
        return StreamEx.of(pricePackageRepository.getAllPricePackages())
                .filter(pricePackage -> {
                    pricePackage.normalizeCampaignOptions();
                    var bl = pricePackage.getCampaignOptions().getAllowBrandLift();
                    return bl == null || !bl;
                })
                .toList();
    }

    public Map<Long, PricePackage> getActivePricePackages(@Nullable Set<AdGroupType> adGroupTypes) {
        var  pricePackageFilter = new PricePackagesFilter()
                .withIsArchived(false)
                .withDateEndAfter(LocalDate.now())
                .withAdGroupTypes(adGroupTypes);
        PricePackageRepository.PricePackagesWithTotalCount pricePackagesWithTotalCount =
                pricePackageRepository.getPricePackages(pricePackageFilter,
                null, LimitOffset.maxLimited());
        return listToMap(pricePackagesWithTotalCount.getPricePackages(), PricePackage::getId);
    }

    /**
     * Добавляет связь клиента с прайсовым пакетом
     *
     * @param clientIdToPackageIds пакеты, которые нужно добавить клиентам
     */
    public void addPackagesToClients(Map<ClientId, Set<Long>> clientIdToPackageIds) {
        pricePackageRepository.addPackagesToClients(clientIdToPackageIds);
    }

    /**
     * Удаляет связь клиента с прайсовым пакетом, если клиент не использует пакет
     *
     * @param clientIdToPackageIds пакеты, которые нужно удалить клиентам
     * @return пакеты, которые были успешно удалены
     */
    public Map<ClientId, Set<Long>> deletePackagesFromClients(Map<ClientId, Set<Long>> clientIdToPackageIds) {
        var packageIdsUsedByClientId = getPricePackageIds(clientIdToPackageIds.keySet());
        var packageIdsToDeleteByClientId = EntryStream.of(clientIdToPackageIds)
                .mapToValue(((clientId, packageIds) -> {
                    var packageIdsUsed = packageIdsUsedByClientId.get(clientId);
                    return filterToSet(packageIds,
                            packageId -> packageIdsUsed == null || !packageIdsUsed.contains(packageId));
                }))
                .toMap();
        pricePackageRepository.deletePackagesFromClients(packageIdsToDeleteByClientId);
        return packageIdsToDeleteByClientId;
    }

    /**
     * Возвращает пакеты прайсовых кампаний клиентов
     *
     * @param clientIds id клиентов
     * @return id прайсовых пакетов
     */
    public Map<ClientId, Set<Long>> getPricePackageIds(Collection<ClientId> clientIds) {
        Map<ClientId, Set<Long>> clientIdToPackageIds = new HashMap<>();
        shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID)
                .forEach((shard, shardClientIds) -> {
                    var clientIdToCampaignIds =
                            EntryStream.of(campaignRepository.searchAllCampaigns(
                                    shard, shardClientIds, Set.of(CampaignType.CPM_PRICE)))
                            .mapKeys(ClientId::fromLong)
                            .mapValues(campaigns -> mapList(campaigns, CampaignSimple::getId))
                            .toMap();
                    var campaignIdToPricePackage = getPricePackageByCampaignIds(
                            shard, flatMap(clientIdToCampaignIds.values(), Function.identity()));
                    var shardClientIdToPackageIds = EntryStream.of(clientIdToCampaignIds)
                            .mapValues(campaignIds -> StreamEx.of(campaignIds)
                                    .filter(campaignIdToPricePackage::containsKey)
                                    .map(campaignIdToPricePackage::get)
                                    .map(PricePackage::getId)
                                    .toSet()
                            )
                            .toMap();
                    clientIdToPackageIds.putAll(shardClientIdToPackageIds);
                });
        return clientIdToPackageIds;
    }
}
