package ru.yandex.direct.core.entity.pricepackage.service;

import java.text.DecimalFormat;
import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcProperty;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageForLock;
import ru.yandex.direct.core.entity.pricepackage.model.StatusApprove;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.pricepackage.service.validation.PricePackageUpdateValidationService;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.grut.api.PricePackageGrut;
import ru.yandex.direct.core.grut.replication.GrutApiService;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.ChangesAppliedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.common.db.PpcPropertyNames.UPDATE_AUCTION_PRIORITY_IN_GRUT;
import static ru.yandex.direct.core.entity.pricepackage.service.PricePackageOperationUtils.VIDEO_FRONTPAGE_CREATIVE_TEMPLATES;
import static ru.yandex.direct.core.entity.pricepackage.service.PricePackageOperationUtils.checkPricePackagesLocked;
import static ru.yandex.direct.core.entity.pricepackage.service.PricePackageOperationUtils.checkPricePackagesNotChanged;
import static ru.yandex.direct.core.entity.pricepackage.service.PricePackageOperationUtils.processVideoFrontpagePackageTag;
import static ru.yandex.direct.model.AppliedChanges.isChanged;
import static ru.yandex.direct.model.AppliedChanges.mapper;
import static ru.yandex.direct.model.AppliedChanges.setter;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class PricePackageUpdateOperation extends SimpleAbstractUpdateOperation<PricePackage, Long> {

    private static final Logger logger = LoggerFactory.getLogger(PricePackageUpdateOperation.class);

    private final Map<Long, LocalDateTime> userTimestampsByPackageId;
    private final DslContextProvider dslContextProvider;
    private final PricePackageUpdateValidationService updateValidationService;
    private final PricePackageRepository repository;
    private final CampaignRepository campaignRepository;
    private final ShardHelper shardHelper;

    private final GeoTree geoTree;
    private final User operator;
    private final PricePackageGeoProcessor geoProcessor;
    private final GrutApiService grutApiService;
    private final PpcProperty<Boolean> updateAuctionPriorityInGrutProperty;

    public PricePackageUpdateOperation(Applicability applicability,
                                       List<ModelChanges<PricePackage>> modelChanges,
                                       List<LocalDateTime> userTimestamps,
                                       DslContextProvider dslContextProvider,
                                       PricePackageUpdateValidationService updateValidationService,
                                       PricePackageRepository repository,
                                       PricePackageService pricePackageService,
                                       CampaignRepository campaignRepository,
                                       GrutApiService grutApiService,
                                       ShardHelper shardHelper,
                                       PpcPropertiesSupport ppcPropertiesSupport,
                                       User operator) {
        super(applicability, modelChanges, id -> new PricePackage().withId(id));
        checkState(modelChanges.size() == userTimestamps.size(),
                "На каждый modelChanges должен быть передан пользовательский timestamp");
        this.userTimestampsByPackageId = new HashMap<>();
        for (int i = 0; i < userTimestamps.size(); i++) {
            var userTimestamp = userTimestamps.get(i);
            var packageId = modelChanges.get(i).getId();
            userTimestampsByPackageId.put(packageId, userTimestamp);
        }
        this.dslContextProvider = dslContextProvider;
        this.updateValidationService = updateValidationService;
        this.repository = repository;
        this.geoTree = pricePackageService.getGeoTree();
        this.geoProcessor = pricePackageService.getGeoProcessor();
        this.campaignRepository = campaignRepository;
        this.shardHelper = shardHelper;
        this.operator = operator;
        this.grutApiService = grutApiService;
        this.updateAuctionPriorityInGrutProperty =
                ppcPropertiesSupport.get(UPDATE_AUCTION_PRIORITY_IN_GRUT, Duration.ofMinutes(5));
    }

    @Override
    protected ValidationResult<List<ModelChanges<PricePackage>>, Defect> validateModelChanges(
            List<ModelChanges<PricePackage>> modelChanges) {
        return updateValidationService.validateModelChanges(modelChanges, userTimestampsByPackageId, operator);
    }

    @Override
    protected Collection<PricePackage> getModels(Collection<Long> ids) {
        return repository.getPricePackages(ids).values();
    }

    @Override
    protected void onChangesApplied(ChangesAppliedStep<PricePackage> changesAppliedStep) {
        super.onChangesApplied(changesAppliedStep);
        changesAppliedStep.getAppliedChangesForValidModelChanges().stream()
                .filter(isChanged(PricePackage.TARGETINGS_FIXED))
                .forEach(ac -> {
                    var targetingsFixed = ac.getNewValue(PricePackage.TARGETINGS_FIXED);
                    geoProcessor.expandGeo(targetingsFixed);
                });
        changesAppliedStep.getAppliedChangesForValidModelChanges().stream()
                .filter(isChanged(PricePackage.TARGETINGS_CUSTOM))
                .forEach(ac -> {
                    var targetingsFixed = ac.getNewValue(PricePackage.TARGETINGS_CUSTOM);
                    geoProcessor.expandGeo(targetingsFixed);
                });
        changesAppliedStep.getAppliedChangesForValidModelChanges().stream()
                .filter(ac -> ac.getModel().isFrontpageVideoPackage())
                .forEach(ac -> ac.modify(PricePackage.ALLOWED_CREATIVE_TEMPLATES, VIDEO_FRONTPAGE_CREATIVE_TEMPLATES));
        changesAppliedStep.getAppliedChangesForValidModelChanges().stream()
                        .filter(ac -> ac.getModel().isFrontpagePackage())
                        .forEach(ac -> ac.modify(PricePackage.IS_FRONTPAGE, true));
        changesAppliedStep.getAppliedChangesForValidModelChanges().stream()
                .filter(isChanged(PricePackage.ALLOWED_TARGET_TAGS))
                .filter(ac -> ac.getModel().isFrontpageVideoPackage())
                .forEach(ac -> ac.modify(PricePackage.ALLOWED_TARGET_TAGS,
                        processVideoFrontpagePackageTag(ac.getNewValue(PricePackage.ALLOWED_TARGET_TAGS))));
        changesAppliedStep.getAppliedChangesForValidModelChanges().stream()
                .filter(isChanged(PricePackage.ALLOWED_ORDER_TAGS))
                .filter(ac -> ac.getModel().isFrontpageVideoPackage())
                .forEach(ac -> ac.modify(PricePackage.ALLOWED_ORDER_TAGS,
                        processVideoFrontpagePackageTag(ac.getNewValue(PricePackage.ALLOWED_ORDER_TAGS))));
    }

    @Override
    protected ValidationResult<List<PricePackage>, Defect> validateAppliedChanges(
            ValidationResult<List<PricePackage>, Defect> validationResult) {
        return updateValidationService.validateAppliedChanges(validationResult, geoTree, operator);
    }

    @Override
    protected void beforeExecution(ExecutionStep<PricePackage> executionStep) {
        var appliedChanges = executionStep.getAppliedChangesForExecution();
        appliedChanges.forEach(setter(PricePackage.LAST_UPDATE_TIME, LocalDateTime.now(ZoneOffset.UTC)));
        appliedChanges.forEach(mapper(PricePackage.TITLE, String::trim));
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<PricePackage>> applicableAppliedChanges) {
        List<Long> applicablePackageIds = applicableAppliedChanges.stream()
                .map(AppliedChanges::getModel)
                .map(PricePackage::getId)
                .collect(toList());
        var grutChanges = applicableAppliedChanges.stream()
                .filter(ch -> ch.changed(PricePackage.AUCTION_PRIORITY))
                .map(ch -> new PricePackageGrut(ch.getModel().getId(), ch.getNewValue(PricePackage.AUCTION_PRIORITY), ch.getModel().getAllowedDomains(), ch.getModel().getAvailableAdGroupTypes()))
                .collect(Collectors.toList());
        if (!grutChanges.isEmpty() && updateAuctionPriorityInGrutProperty.getOrDefault(false)) {
            logger.info("Update price packages in GrUT " + grutChanges);
            grutApiService.getPricePackageGrutApi().createOrUpdatePackages(grutChanges);
        }
        dslContextProvider.ppcdictTransaction(configuration -> {
            Map<Long, PricePackageForLock> lockedPricePackages = repository.lockPricePackagesForUpdate(
                    configuration.dsl(), applicablePackageIds);
            checkPricePackagesLocked(applicablePackageIds, lockedPricePackages);
            checkPricePackagesNotChanged(lockedPricePackages.values(), userTimestampsByPackageId);
            repository.updatePricePackages(configuration.dsl(), applicableAppliedChanges);
        });
        updateCampaignsIfNeeded(applicableAppliedChanges);
        resendActiveCampaignsToBSIfNeeded(applicableAppliedChanges);

        return mapList(applicableAppliedChanges, a -> a.getModel().getId());
    }

    @Override
    protected void afterExecution(ExecutionStep<PricePackage> executionStep) {
        logPricePackagesApprove(executionStep.getAppliedChangesForExecution());
    }

    private void updateCampaignsIfNeeded(Collection<AppliedChanges<PricePackage>> appliedChangesList) {
        var changedPricePackages = listToMap(
                filterList(appliedChangesList, ac -> ac.changed(PricePackage.ALLOWED_PAGE_IDS)),
                ac -> ac.getModel().getId()
        );

        if (changedPricePackages.isEmpty()) {
            return;
        }

        shardHelper.forEachShard(
                shard -> {
                    Map<Long, Long> campaignIdToPackageId =
                            campaignRepository.getCampaignIdsByPricePackageIds(shard, changedPricePackages.keySet());
                    if (campaignIdToPackageId.isEmpty()) {
                        return;
                    }
                    campaignIdToPackageId.forEach(
                            (campaignId, pricePackageId) -> {
                                var appliedChanges = changedPricePackages.get(pricePackageId);
                                if (appliedChanges == null) {
                                    return;
                                }
                                campaignRepository.updateAllowedPageIds(shard, campaignId,
                                        Optional.ofNullable(appliedChanges.getNewValue(PricePackage.ALLOWED_PAGE_IDS))
                                                .orElse(emptyList()));
                            }
                    );
                }
        );

    }

    private void resendActiveCampaignsToBSIfNeeded(Collection<AppliedChanges<PricePackage>> appliedChangesList) {
        List<Long> changedPricePackageIds = filterAndMapList(appliedChangesList,
                ac -> ac.changed(PricePackage.ALLOWED_PAGE_IDS)
                || ac.changed(PricePackage.ALLOWED_DOMAINS) || ac.changed(PricePackage.ALLOWED_SSP),
                ac -> ac.getModel().getId());
        if (changedPricePackageIds.isEmpty()) {
            return;
        }
        shardHelper.forEachShard(
                shard -> {
                    List<Long> campaignIdsForResentToBS =
                            campaignRepository.getCampaignIdsForResendBSByPricePackageIds(shard, changedPricePackageIds);
                    if (campaignIdsForResentToBS.isEmpty()) {
                        return;
                    }
                    campaignRepository.resetBannerSystemSyncStatus(shard, campaignIdsForResentToBS);
                }
        );
    }

    private void logPricePackagesApprove(Collection<AppliedChanges<PricePackage>> appliedChangesList) {
        DecimalFormat df = new DecimalFormat();
        df.setMaximumFractionDigits(2);
        df.setMinimumFractionDigits(2);
        df.setGroupingUsed(false);

        for (var ac : appliedChangesList) {
            if (ac.changed(PricePackage.STATUS_APPROVE)
                    && ac.getNewValue(PricePackage.STATUS_APPROVE) == StatusApprove.YES) {
                var pricePackage = ac.getModel();
                logger.info("Approving price package. Id: \"{}\", Title: \"{}\", CPM: \"{}\", Currency: \"{}\", " +
                                "Tracker URL: \"{}\", Operator UID: \"{}\", Operator login: \"{}\".",
                        pricePackage.getId(), pricePackage.getTitle(),
                        df.format(pricePackage.getPrice()),
                        pricePackage.getCurrency(),
                        pricePackage.getTrackerUrl(),
                        operator.getUid(), operator.getLogin());
            }
        }
    }

}
