package ru.yandex.direct.core.entity.pricepackage.service.validation;

import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.packageNotApproved;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

@Service
public class PricePackageDeleteValidationService {

    private final PricePackageRepository pricePackageRepository;

    @Autowired
    public PricePackageDeleteValidationService(PricePackageRepository pricePackageRepository) {
        this.pricePackageRepository = pricePackageRepository;
    }

    public ValidationResult<List<Long>, Defect> validate(List<Long> pricePackageIds) {
        Map<Long, PricePackage> pricePackages = pricePackageRepository.getPricePackages(pricePackageIds);

        return ListValidationBuilder.of(pricePackageIds, Defect.class)
                .check(notNull())
                .checkEach(notNull())
                .checkEachBy(pricePackageId -> validate(pricePackageId, pricePackages), When.isValid())
                .getResult();

    }

    private ValidationResult<Long, Defect> validate(Long pricePackageId, Map<Long, PricePackage> pricePackages) {
        var pricePackage = pricePackages.get(pricePackageId);
        var existingIds = pricePackages.keySet();

        return ItemValidationBuilder.<Long, Defect>of(pricePackageId)
                .check(inSet(existingIds), CommonDefects.objectNotFound())
                .check(packageNotApproved(pricePackage), When.isValid())
                .getResult();
    }
}
