package ru.yandex.direct.core.entity.pricepackage.service.validation;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.service.validation.PropertyChangeValidator;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.repository.PricePackageRepository;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackagePermissionUtils;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;

import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.forbiddenToChangeFieldsAreNotChanged;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.userTimestampEqualsLastUpdateTime;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.creativeTemplatesCanOnlyExpand;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.dateEndCanOnlyExpand;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.dateStartCanOnlyExpand;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.pricePackageIsExpired;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.retargetingCategoriesAmountLessLowerLimit;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Выполняет проверки списка пакетов специфичные для операции обновления.
 */
@Service
public class PricePackageUpdateValidationService {

    private final PricePackageRepository repository;
    private final PricePackageValidationService validationService;
    private final ModelChangesValidationTool preValidationTool;

    private static final PropertyChangeValidator<PricePackage, PricePackageModelChangesValidationContext>
            PROPERTY_CHANGE_VALIDATOR = PropertyChangeValidator
            .<PricePackage, PricePackageModelChangesValidationContext>newBuilder()
            .add(PricePackage.STATUS_APPROVE, PricePackagePermissionUtils::canChangeStatusApprove)
            .add(PricePackage.DATE_START, PricePackagePermissionUtils::isDateStartOnlyExpand, dateStartCanOnlyExpand())
            .add(PricePackage.DATE_END, PricePackagePermissionUtils::isDateEndOnlyExpand, dateEndCanOnlyExpand())
            .add(PricePackage.CLIENTS, PricePackagePermissionUtils::pricePackageIsNotExpired, pricePackageIsExpired())
            .add(PricePackage.TARGETINGS_CUSTOM,
                    PricePackagePermissionUtils::retargetingCategoriesAmountIsNotLessLowerLimit,
                    retargetingCategoriesAmountLessLowerLimit())
            .add(PricePackage.TARGETINGS_FIXED,
                    PricePackagePermissionUtils::allowExpandedDesktopCreativeAllowedToRemove)
            .add(PricePackage.ALLOWED_CREATIVE_TEMPLATES, PricePackagePermissionUtils::isNewCreativeTemplatesOnlyExpand,
                    creativeTemplatesCanOnlyExpand())
            .add(PricePackage.TARGETING_MARKUPS, PricePackagePermissionUtils::targetingsMarkupsCanBeChanged)
            .add(PricePackage.ALLOWED_PROJECT_PARAM_CONDITIONS, PricePackagePermissionUtils::allowedProjectParamConditionsCanBeChanged)
            .build();

    @Autowired
    public PricePackageUpdateValidationService(PricePackageRepository repository,
                                               PricePackageValidationService validationService) {
        this.repository = repository;
        this.validationService = validationService;
        this.preValidationTool = ModelChangesValidationTool.builder()
                .minSize(1)
                .build();
    }

    public ValidationResult<List<ModelChanges<PricePackage>>, Defect> validateModelChanges(
            List<ModelChanges<PricePackage>> modelChangesList,
            Map<Long, LocalDateTime> userTimestamps,
            User operator) {
        Set<Long> packageIds = listToSet(modelChangesList, ModelChanges::getId);
        Map<Long, PricePackage> pricePackages = repository.getPricePackages(packageIds);
        var context = new PricePackageModelChangesValidationContext(operator);
        return ListValidationBuilder.of(modelChangesList, Defect.class)
                // в DIRECT-103378 вызов в preValidationTool должен быть исправлен
                .checkBy(mcList -> preValidationTool.validateModelChangesList(mcList, pricePackages.keySet()))
                .checkEach(userTimestampEqualsLastUpdateTime(userTimestamps, pricePackages), When.isValid())

                // Для фронта имеет большое значение - висит ошибка на корне или на конкретном поле.
                // В случае, если пакет вообще нельзя редактировать (например, если он заархивирован) или были
                // отредактированы поля, которые запрещено менять (например цена в заапрувленном пакете), то выдается
                // ошибка на уровне всего апдейта.
                .checkEach(forbiddenToChangeFieldsAreNotChanged(pricePackages, context), When.isValid())

                // Если были отредактированы поля, которые можно менять, но отредактированы неправильно (например,
                // дата окончание уменьшается), то новая ошибка пишется на уровне поля.
                .checkEachBy((index, modelChanges) -> {
                    var model = pricePackages.get(modelChanges.getId());
                    return PROPERTY_CHANGE_VALIDATOR.validate(modelChanges, model, context);
                }, When.isValid())

                .getResult();
    }

    public ValidationResult<List<PricePackage>, Defect> validateAppliedChanges(
            ValidationResult<List<PricePackage>, Defect> validationResult,
            GeoTree geoTree, User operator) {
        return new ListValidationBuilder<>(validationResult)
                .checkBy(pricePackages -> validationService.validatePricePackages(pricePackages, geoTree, operator),
                        When.isValid())
                .getResult();
    }

}
