package ru.yandex.direct.core.entity.pricepackage.service.validation;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.crypta.repository.CryptaSegmentRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageClient;
import ru.yandex.direct.core.entity.product.service.ProductService;
import ru.yandex.direct.core.entity.region.validation.RegionIdsValidator;
import ru.yandex.direct.core.entity.sspplatform.repository.SspPlatformsRepository;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;

/**
 * Проверяет список пакетов в целом. Без учета того, какая операция выполняется.
 */
@Service
class PricePackageValidationService {

    private final RegionIdsValidator regionsValidator;
    private final CryptaSegmentRepository cryptaSegmentRepository;
    private final ClientService clientService;
    private final ProductService productService;
    private final SspPlatformsRepository sspPlatformsRepository;
    private final FeatureService featureService;


    @Autowired
    public PricePackageValidationService(RegionIdsValidator regionsValidator,
                                         CryptaSegmentRepository cryptaSegmentRepository,
                                         ClientService clientService, ProductService productService,
                                         SspPlatformsRepository sspPlatformsRepository,
                                         FeatureService featureService) {
        this.regionsValidator = regionsValidator;
        this.cryptaSegmentRepository = cryptaSegmentRepository;
        this.clientService = clientService;
        this.productService = productService;
        this.sspPlatformsRepository = sspPlatformsRepository;
        this.featureService = featureService;

    }

    public ValidationResult<List<PricePackage>, Defect> validatePricePackages(List<PricePackage> pricePackages,
                                                                              GeoTree geoTree, User operator) {
        var vb = ListValidationBuilder.<PricePackage, Defect>of(pricePackages);

        var knownSsp = listToSet(sspPlatformsRepository.getAllSspPlatforms());
        var clientsCurrencies = getClientsCurrencies(pricePackages);

        var validator = new PricePackageValidator(geoTree, regionsValidator, clientsCurrencies,
                productService, knownSsp, featureService.getEnabledForUid(operator.getUid()));
        return vb
                .check(notNull())
                .checkEach(notNull())
                .checkEachBy(validator, When.isValid())
                .getResult();
    }

    private Map<ClientId, Currency> getClientsCurrencies(List<PricePackage> pricePackages) {
        Set<ClientId> clientIds = onlyNotNullPricePackagesSteam(pricePackages)
                .map(PricePackage::getClients).filter(Objects::nonNull)
                .flatMap(Collection::stream).filter(Objects::nonNull)
                .map(PricePackageClient::getClientId).filter(Objects::nonNull)
                .map(ClientId::fromLong)
                .collect(Collectors.toSet());
        return clientService.massGetWorkCurrency(clientIds);
    }

    private static Stream<PricePackage> onlyNotNullPricePackagesSteam(List<PricePackage> pricePackages) {
        return Optional.ofNullable(pricePackages)
                .orElse(emptyList())
                .stream()
                .filter(Objects::nonNull);
    }

}
