package ru.yandex.direct.core.entity.pricepackage.service.validation;

import java.math.BigDecimal;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.pricepackage.model.PriceMarkup;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageCampaignOptions;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageClient;
import ru.yandex.direct.core.entity.pricepackage.model.ShowsFrequencyLimit;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingMarkup;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsCustom;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsFixed;
import ru.yandex.direct.core.entity.pricepackage.model.ViewType;
import ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints;
import ru.yandex.direct.core.entity.product.service.ProductService;
import ru.yandex.direct.core.entity.region.validation.RegionIdsValidator;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.regions.GeoTree;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.DefaultValidator;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupValidationService.MAX_TAG_LENGTH;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupValidationService.MAX_TARGET_TAGS_COUNT;
import static ru.yandex.direct.core.entity.adgroup.service.validation.AdGroupValidationService.TAG_PATTERN;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstants.MAX_DOMAIN_LENGTH;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignConstraints.isValidSsp;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.atLeastOneTemplateIsChosen;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.clientCurrencyEqualsPackageCurrency;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.fixedIsSubsetOfCustom;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.geoExpandedIsNotEmpty;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.isCpdAllowed;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.isMatchingCreativeTemplatesAndAdGroupTypes;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.isMatchingCryptaTypesCount;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.isMatchingFixedRetargetingCategoriesCount;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.isMatchingRetargetingCategoriesCount;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.nonOverlappingMarkups;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.totalAllowedDomainsMaxSize;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.uniqueDomains;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageConstraints.validBidModifiersByFeatures;
import static ru.yandex.direct.core.validation.constraints.Constraints.cpmNotGreaterThan;
import static ru.yandex.direct.core.validation.constraints.Constraints.cpmNotLessThan;
import static ru.yandex.direct.regions.Region.REGION_TYPE_COUNTRY;
import static ru.yandex.direct.regions.Region.REGION_TYPE_DISTRICT;
import static ru.yandex.direct.regions.Region.REGION_TYPE_PROVINCE;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.Predicates.not;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxListSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.maxSetSize;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.notEmptyCollection;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isEqual;
import static ru.yandex.direct.validation.constraint.CommonConstraints.isNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notTrue;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotAfterThan;
import static ru.yandex.direct.validation.constraint.DateConstraints.isNotBeforeThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.greaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;
import static ru.yandex.direct.validation.constraint.NumberConstraints.lessThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notGreaterThan;
import static ru.yandex.direct.validation.constraint.NumberConstraints.notLessThan;
import static ru.yandex.direct.validation.constraint.StringConstraints.admissibleChars;
import static ru.yandex.direct.validation.constraint.StringConstraints.matchPattern;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@ParametersAreNonnullByDefault
public class PricePackageValidator implements DefaultValidator<PricePackage> {

    private static final int TITLE_MAX_LENGTH = 200;
    private static final int TRACKER_URL_MAX_LENGTH = 2048;
    private static final long ORDER_VOLUME_MAX_VALUE = Integer.MAX_VALUE;
    public static final int REGION_TYPE_REGION = 10;
    private static final Set<Integer> ALLOWED_GEO_TYPES = ImmutableSet.of(REGION_TYPE_REGION,
            //для обратной совместимости закопаем geoType в DIRECT-137986
            REGION_TYPE_COUNTRY, REGION_TYPE_DISTRICT, REGION_TYPE_PROVINCE);


    private static final int MAX_FREQUENCY_LIMIT_VALUE = 1000;
    private static final int MAX_FREQUENCY_LIMIT_DAYS_VALUE = 100;

    private static final Map<AdGroupType, Set<CreativeType>> ACCESSIBLE_CREATIVE_TYPES_BY_AD_GROUP_TYPE = Map.of(
            AdGroupType.CPM_VIDEO, Set.of(
                    CreativeType.BANNERSTORAGE, CreativeType.CPM_VIDEO_CREATIVE, CreativeType.CPM_OVERLAY),
            AdGroupType.CPM_BANNER, Set.of(CreativeType.BANNERSTORAGE, CreativeType.CANVAS,
                    CreativeType.HTML5_CREATIVE),
            AdGroupType.CPM_AUDIO, Set.of(CreativeType.BANNERSTORAGE, CreativeType.CPM_AUDIO_CREATIVE)
    );

    private static final Set<Long> INCOME_SEGMENTS = Set.of(
            2499000009L,
            2499000010L,
            2499000012L,
            2499000013L
    );

    private final GeoTree geoTree;
    private final RegionIdsValidator regionsValidator;
    private final Map<Long, CurrencyCode> existingClientIdsWithCurrency;
    private final AvailableCurrencyConstraint notYndFixedCurrency;
    private final ProductService productService;
    private final Set<String> knownSsp;
    private final Set<String> features;

    public PricePackageValidator(GeoTree geoTree, RegionIdsValidator regionsValidator,
                                 Map<ClientId, Currency> existingClientIdsWithCurrency,
                                 ProductService productService, Set<String> knownSsp,
                                 Set<String> features) {
        this.geoTree = geoTree;
        this.regionsValidator = regionsValidator;
        this.existingClientIdsWithCurrency = EntryStream.of(existingClientIdsWithCurrency)
                .mapKeys(ClientId::asLong)
                .mapValues(Currency::getCode)
                .toMap();
        this.productService = productService;
        this.knownSsp = knownSsp;
        this.notYndFixedCurrency = AvailableCurrencyConstraint.allExcept(CurrencyCode.YND_FIXED);
        this.features = features;
    }

    @Override
    public ValidationResult<PricePackage, Defect> apply(PricePackage pricePackage) {
        var vb = ModelItemValidationBuilder.of(pricePackage);

        vb.item(PricePackage.TITLE)
                .check(notNull())
                .check(notBlank())
                .check(maxStringLength(TITLE_MAX_LENGTH))
                .check(admissibleChars());

        vb.item((PricePackage.TRACKER_URL))
                .check(notNull())
                .check(notBlank())
                .check(maxStringLength(TRACKER_URL_MAX_LENGTH))
                .check(validHref());

        var vbDateStart = vb.item(PricePackage.DATE_START)
                .check(notNull());

        vb.item(PricePackage.DATE_END)
                .check(notNull())
                .check(isNotBeforeThan(pricePackage.getDateStart()), When.isValidBoth(vbDateStart));

        var vbCurrency = vb.item(PricePackage.CURRENCY)
                .check(notNull())
                .check(notYndFixedCurrency);

        CurrencyCode currencyCode = pricePackage.getCurrency();
        Currency currency = ifNotNull(currencyCode, CurrencyCode::getCurrency);
        BigDecimal minCpmPrice = ifNotNull(currency, Currency::getMinCpmPrice);
        BigDecimal maxCpmPrice = ifNotNull(currency, Currency::getMaxCpmPrice);
        if (nvl(pricePackage.getIsCpd(), false)) {
            maxCpmPrice = null;//У CPD пакета нет ограничения сверху на цену дня
        }

        // 0 или null в productId - признак отсутствия продукта на пакете
        vb.item(PricePackage.PRODUCT_ID)
                .check(inSet(
                        Set.copyOf(
                                mapList(filterList(productService.getProducts(),
                                        p -> p.getCurrencyCode() == currencyCode), ModelWithId::getId)
                        )), When.notNullAnd(When.valueIs(not(Long.valueOf(0)::equals))));

        vb.item(PricePackage.PRICE)
                .check(notNull())
                .check(cpmNotGreaterThan(maxCpmPrice, currencyCode), When.isValidBoth(vbCurrency))
                .check(cpmNotLessThan(minCpmPrice, currencyCode), When.isValidBoth(vbCurrency));

        var vbOrderVolumeMin = vb.item(PricePackage.ORDER_VOLUME_MIN)
                .check(notNull())
                .check(greaterThan(0L))
                .check(notGreaterThan(ORDER_VOLUME_MAX_VALUE));

        var vbOrderVolumeMax = vb.item(PricePackage.ORDER_VOLUME_MAX);
        vbOrderVolumeMax
                .check(notNull())
                .check(greaterThan(0L))
                .check(notGreaterThan(ORDER_VOLUME_MAX_VALUE));
        if (pricePackage.getOrderVolumeMin() != null) {
            vbOrderVolumeMax.check(notLessThan(pricePackage.getOrderVolumeMin()), When.isValidBoth(vbOrderVolumeMin));
        }

        vb.list(PricePackage.PRICE_MARKUPS)
                .check(nonOverlappingMarkups(), When.notNull())
                .checkEachBy(priceMarkupValidator(pricePackage), When.notNull());

        vb.list(PricePackage.TARGETING_MARKUPS)
                .checkEach(unique(Comparator.comparing(TargetingMarkup::getConditionId)), When.notNull());

        vb.item(PricePackage.IS_PUBLIC)
                .check(notNull());

        vb.item(PricePackage.IS_SPECIAL)
                .check(notNull());

        vb.item(PricePackage.IS_CPD)
                .check(notNull())
                .check(isCpdAllowed(pricePackage), When.isValid());

        // ровно один geo должен быть null, а тот который не null не может быть пустым
        vb.item(PricePackage.TARGETINGS_FIXED)
                .check(notNull())
                .checkBy(targetingsFixedValidator(pricePackage), When.isValid());

        vb.item(PricePackage.TARGETINGS_CUSTOM)
                .check(notNull())
                .checkBy(targetingsCustomValidator(pricePackage), When.isValid());

        vb.list(PricePackage.CLIENTS)
                .check(notNull())
                .checkEach(notNull(), When.isValid())
                .checkEachBy(packageClientValidator(pricePackage), When.isValid());

        vb.item(PricePackage.CAMPAIGN_OPTIONS)
                .checkBy(packageCampaignOptionsValidator(pricePackage), When.notNull());

        vb.item(PricePackage.BID_MODIFIERS)
                .checkBy(bidModifiersValidator(features));

        // AdGroupType.CPM_YNDX_FRONTPAGE нельзя сочетать с любым другим типом групп
        var vbAdGroupTypes = vb.item(PricePackage.AVAILABLE_AD_GROUP_TYPES);
        vbAdGroupTypes
                .check(notNull())
                .check(notEmptyCollection(), When.isValid())
                .check(maxSetSize(1), When.isValidAnd(When.isTrue(isCpmYndxFrontpageAdGroupType(pricePackage))));

        var vbCreativeTemplates = vb.item(PricePackage.ALLOWED_CREATIVE_TEMPLATES);
        vbCreativeTemplates
                .check(atLeastOneTemplateIsChosen(),
                        When.notNullAnd(When.isFalse(isCpmYndxFrontpageAdGroupType(pricePackage))))
                .check(isMatchingCreativeTemplatesAndAdGroupTypes(pricePackage.getAvailableAdGroupTypes(),
                        ACCESSIBLE_CREATIVE_TYPES_BY_AD_GROUP_TYPE),
                        When.notNullAnd(When.isValidBoth(vbAdGroupTypes)));

        vb.list(PricePackage.ALLOWED_DOMAINS)
                .check(totalAllowedDomainsMaxSize(pricePackage.getAllowedSsp()))
                .check(uniqueDomains())
                .checkEach(maxStringLength(MAX_DOMAIN_LENGTH));

        vb.list(PricePackage.ALLOWED_TARGET_TAGS)
                .checkEach(notNull())
                .checkEach(maxStringLength(MAX_TAG_LENGTH), When.isValid())
                .checkEach(unique(), When.isValid())
                .checkEach(matchPattern(TAG_PATTERN), When.isValid())
                .check(maxListSize(MAX_TARGET_TAGS_COUNT), When.isValid());

        vb.list(PricePackage.ALLOWED_ORDER_TAGS)
                .checkEach(notNull())
                .checkEach(maxStringLength(MAX_TAG_LENGTH), When.isValid())
                .checkEach(unique(), When.isValid())
                .checkEach(matchPattern(TAG_PATTERN), When.isValid())
                .check(maxListSize(MAX_TARGET_TAGS_COUNT), When.isValid());

        vb.list(PricePackage.ALLOWED_SSP)
                .check(uniqueDomains())
                .checkEach(isValidSsp(knownSsp));

        return vb.getResult();
    }

    private Validator<PriceMarkup, Defect> priceMarkupValidator(PricePackage pricePackage) {
        return priceMarkup -> {
            var vb = ModelItemValidationBuilder.of(priceMarkup);
            vb.item(PriceMarkup.PERCENT)
                    .check(notNull())
                    .check(inRange(-100, 100), When.isValid());

            var packageDateStart = pricePackage.getDateStart();
            var packageDateEnd = pricePackage.getDateEnd();

            var vbDateStart = vb.item(PriceMarkup.DATE_START)
                    .check(notNull());
            if (packageDateStart != null) {
                vbDateStart
                        .check(isNotBeforeThan(pricePackage.getDateStart()), When.isValid());
            }

            var vbDateEnd = vb.item(PriceMarkup.DATE_END)
                    .check(notNull())
                    .check(isNotBeforeThan(priceMarkup.getDateStart()), When.isValidBoth(vbDateStart));
            if (packageDateEnd != null) {
                vbDateEnd
                        .check(isNotAfterThan(pricePackage.getDateEnd()), When.isValid());
            }

            return vb.getResult();
        };
    }

    private Validator<TargetingsFixed, Defect> targetingsFixedValidator(PricePackage pricePackage) {
        return targetingsFixed -> {
            var vb = ModelItemValidationBuilder.of(targetingsFixed);
            var vbGeoType = vb.item(TargetingsFixed.GEO_TYPE);
            var vbGeo = vb.item(TargetingsFixed.GEO);

            var fixedGeo = targetingsFixed.getGeo();
            var fixedGeoType = targetingsFixed.getGeoType();
            var targetingsCustom = pricePackage.getTargetingsCustom();

            vbGeoType
                    .checkBy(geoTypeValidator(fixedGeo));

            vbGeo
                    .checkBy(geo -> regionsValidator.apply(geo, geoTree), When.notNull());
            if (!vbGeoType.getResult().hasAnyErrors()) {
                vbGeo
                        .check(new PricePackageRegionsConstraint<>(geoTree, fixedGeoType),
                                When.isValidAnd(When.notNull()))
                        .check(geoExpandedIsNotEmpty(targetingsFixed.getGeoExpanded()),
                                When.isValidAnd(When.notNullAnd(When.isTrue(
                                        fixedGeoType != null && fixedGeoType != REGION_TYPE_REGION))))
                        .check(fixedIsSubsetOfCustom(targetingsCustom),
                                When.isValidAnd(When.notNullAnd(When.isTrue(
                                        fixedGeoType != null && fixedGeoType == REGION_TYPE_REGION))));
            }

            vb.list(TargetingsFixed.VIEW_TYPES)
                    .check(notNull())
                    .check(notEmptyCollection(),
                            When.isTrue(pricePackage.isFrontpagePackage() || pricePackage.isFrontpageVideoPackage()))
                    .checkEach(notNull())
                    .checkEach(unique(), When.isValid());

            // ALLOW_EXPANDED_DESKTOP_CREATIVE доступен только для AdGroupType.CPM_YNDX_FRONTPAGE
            var isCpmYndxFrontpage = isCpmYndxFrontpageAdGroupType(pricePackage);
            vb.item(TargetingsFixed.ALLOW_EXPANDED_DESKTOP_CREATIVE)
                    .check(notNull(), When.isTrue(isCpmYndxFrontpage))
                    .check(notTrue(), When.isFalse(isCpmYndxFrontpage));

            vb.item(TargetingsFixed.ALLOW_PREMIUM_DESKTOP_CREATIVE)
                    .check(notTrue(), When.isFalse(isCpmYndxFrontpage))
                    .check(notTrue(),
                            When.isFalse(isOnlySpecificViewType(pricePackage, Set.of(ViewType.DESKTOP, ViewType.MOBILE))))
                    .check(notTrue(),
                            When.isTrue(pricePackage.getTargetingsFixed().getAllowExpandedDesktopCreative() != null
                                    && pricePackage.getTargetingsFixed().getAllowExpandedDesktopCreative())
                    );

            vb.item(TargetingsFixed.HIDE_INCOME_SEGMENT)
                    .check(notTrue(), When.isFalse(isAnyIncomeInFixedCryptaSegments(pricePackage)))
                    .check(notTrue(), When.isFalse(isAllCustomCryptaSegmentsIncomeInFixedCryptaSegments(pricePackage)));

            var retargetingCondition = ifNotNull(targetingsCustom, TargetingsCustom::getRetargetingCondition);
            vb.item(TargetingsFixed.CRYPTA_SEGMENTS)
                    .check(isMatchingFixedRetargetingCategoriesCount(retargetingCondition),
                            When.notNullAnd(When.isTrue(retargetingCondition != null)));

            return vb.getResult();
        };
    }

    private Validator<List<BidModifier>, Defect> bidModifiersValidator(Set<String> features) {
        return bidModifiers -> {
            ItemValidationBuilder<List<BidModifier>, Defect> vb = ModelItemValidationBuilder.of(bidModifiers);
            vb
                    .check(PricePackageConstraints.platformBidModifiers(), When.notNull())
                    .check(validBidModifiersByFeatures(features, bidModifiers), When.isValid());

            return vb.getResult();
        };
    }

    private Validator<TargetingsCustom, Defect> targetingsCustomValidator(PricePackage pricePackage) {
        return targetingsCustom -> {
            var vb = ModelItemValidationBuilder.of(targetingsCustom);
            var vbGeoType = vb.item(TargetingsCustom.GEO_TYPE);
            var vbGeo = vb.item(TargetingsCustom.GEO);

            var targetingsFixed = pricePackage.getTargetingsFixed();
            var fixedGeoIsNull = targetingsFixed == null || targetingsFixed.getGeo() == null;
            var customGeo = targetingsCustom.getGeo();
            var customGeoType = targetingsCustom.getGeoType();

            vbGeoType
                    .checkBy(geoTypeValidator(customGeo));

            vbGeo
                    .check(notNull(), When.isTrue(fixedGeoIsNull))
                    .check(isNull(), When.isTrue(!fixedGeoIsNull &&
                            customGeoType != null && customGeoType != REGION_TYPE_REGION))
                    .checkBy(geo -> regionsValidator.apply(geo, geoTree), When.notNull());
            if (!vbGeoType.getResult().hasAnyErrors()) {
                vbGeo
                        .check(new PricePackageRegionsConstraint<>(geoTree, customGeoType),
                                When.isValidAnd(When.notNull()))
                        .check(geoExpandedIsNotEmpty(targetingsCustom.getGeoExpanded()),
                                When.isValidAnd(When.notNullAnd(When.isFalse(
                                        customGeoType != null && customGeoType == REGION_TYPE_REGION))));
            }

            vb.item(TargetingsCustom.RETARGETING_CONDITION)
                    .check(isMatchingCryptaTypesCount(), When.notNull())
                    .check(isMatchingRetargetingCategoriesCount(), When.isValid());

            return vb.getResult();
        };
    }

    private Validator<Integer, Defect> geoTypeValidator(@Nullable List<Long> geo) {
        return geoType -> ItemValidationBuilder.of(geoType, Defect.class)
                .check(notNull(), When.isTrue(geo != null))
                .check(isNull(), When.isTrue(geo == null))
                .check(inSet(ALLOWED_GEO_TYPES), When.notNull())
                .getResult();
    }

    private Validator<PricePackageClient, Defect> packageClientValidator(PricePackage pricePackage) {
        return packageClient -> {
            ModelItemValidationBuilder<PricePackageClient> vb = ModelItemValidationBuilder.of(packageClient);

            vb.item(PricePackageClient.CLIENT_ID)
                    .check(notNull())
                    .check(inSet(existingClientIdsWithCurrency.keySet()), objectNotFound(), When.isValid())
                    .check(clientCurrencyEqualsPackageCurrency(pricePackage, existingClientIdsWithCurrency),
                            When.isValid());

            if (pricePackage.getIsPublic() != null) {
                vb.item(PricePackageClient.IS_ALLOWED)
                        .check(notNull())
                        // к общедоступным пакетам могут быть привязаны только минус-клиенты
                        // к обычным пакетам могут быть привязаны только плюс-клиенты
                        .check(isEqual(!pricePackage.getIsPublic(), invalidValue()), When.isValid());
            }

            return vb.getResult();
        };
    }

    private Validator<PricePackageCampaignOptions, Defect> packageCampaignOptionsValidator(PricePackage pricePackage) {
        return packageCampaignOptions -> {
            var vb = ModelItemValidationBuilder.of(packageCampaignOptions);
            var vbShowsFrequencyLimit = vb.item(PricePackageCampaignOptions.SHOWS_FREQUENCY_LIMIT);

            vbShowsFrequencyLimit
                    .checkBy(showsFrequencyLimitValidator(), When.notNull());

            return vb.getResult();
        };
    }

    private Validator<ShowsFrequencyLimit, Defect> showsFrequencyLimitValidator() {
        return showsFrequencyLimit -> {
            var vb = ModelItemValidationBuilder.of(showsFrequencyLimit);
            var vbFrequencyLimit = vb.item(ShowsFrequencyLimit.FREQUENCY_LIMIT);
            var vbFrequencyLimitDays = vb.item(ShowsFrequencyLimit.FREQUENCY_LIMIT_DAYS);

            var frequencyLimit = showsFrequencyLimit.getFrequencyLimit();
            var frequencyLimitDays = showsFrequencyLimit.getFrequencyLimitDays();
            var frequencyLimitIsForCampaignTime = showsFrequencyLimit.getFrequencyLimitIsForCampaignTime();
            vbFrequencyLimit
                    .check(notNull(), When.isTrue(frequencyLimitDays != null))
                    .check(greaterThan(0), When.notNull())
                    .check(lessThan(MAX_FREQUENCY_LIMIT_VALUE), When.notNull());
            vb.item(ShowsFrequencyLimit.FREQUENCY_LIMIT_IS_FOR_CAMPAIGN_TIME)
                    .check(notNull());
            vbFrequencyLimitDays
                    .check(notNull(), When.isTrue(frequencyLimit != null && !frequencyLimitIsForCampaignTime))
                    .check(greaterThan(0), When.notNull())
                    .check(lessThan(MAX_FREQUENCY_LIMIT_DAYS_VALUE), When.notNull());

            return vb.getResult();
        };
    }

    private boolean isCpmYndxFrontpageAdGroupType(PricePackage pricePackage) {
        return pricePackage.getAvailableAdGroupTypes() != null
                && pricePackage.isFrontpagePackage();
    }

    private boolean isOnlySpecificViewType(PricePackage pricePackage, Set<ViewType> viewType) {
        var viewTypes = pricePackage.getTargetingsFixed().getViewTypes();
        return viewTypes != null && viewTypes.size() == 1 &&
                StreamEx.of(viewTypes)
                        .filter(Objects::nonNull)
                        .filter(viewType::contains)
                        .distinct().count() == 1;
    }

    private boolean isAnyIncomeInFixedCryptaSegments(PricePackage pricePackage) {
        var cryptaSegments = pricePackage.getTargetingsFixed().getCryptaSegments();
        return cryptaSegments != null && cryptaSegments.stream().anyMatch(INCOME_SEGMENTS::contains);
    }

    private boolean isAllCustomCryptaSegmentsIncomeInFixedCryptaSegments(PricePackage pricePackage) {
        var fixedCryptaSegments = pricePackage
                .getTargetingsFixed()
                .getCryptaSegments();
        var fixedIncome = fixedCryptaSegments == null
                ? null
                : fixedCryptaSegments.stream()
                .filter(INCOME_SEGMENTS::contains)
                .collect(Collectors.toSet());

        var customCryptaSegments = pricePackage.getTargetingsCustom() == null ?
                null :
                pricePackage.getTargetingsCustom()
                        .getRetargetingCondition()
                        .getCryptaSegments();
        return customCryptaSegments == null ||
                customCryptaSegments.stream()
                        .allMatch(s -> !INCOME_SEGMENTS.contains(s) || (fixedIncome != null && fixedIncome.contains(s)));
    }
}
