package ru.yandex.direct.core.entity.pricepackage.service.validation.defects;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.apache.commons.collections4.ListUtils;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifier;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventory;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierInventoryAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierMobile;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.InventoryType;
import ru.yandex.direct.core.entity.creative.model.CreativeType;
import ru.yandex.direct.core.entity.pricepackage.model.PriceMarkup;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.pricepackage.model.PriceRetargetingCondition;
import ru.yandex.direct.core.entity.pricepackage.model.StatusApprove;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsCustom;
import ru.yandex.direct.core.entity.pricepackage.model.TargetingsFixed;
import ru.yandex.direct.core.entity.pricepackage.service.PricePackagePermissionUtils;
import ru.yandex.direct.core.entity.pricepackage.service.validation.PricePackageModelChangesValidationContext;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.validation.ValidationUtils;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.utils.FunctionalUtils;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.defect.CollectionDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.DefectIds;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.function.Function.identity;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.ACTUAL_INVENTORY_WIDE_TO_NARROW;
import static ru.yandex.direct.core.entity.campaign.service.CampaignWithPricePackageUtils.extractBidModifierInventory;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.duplicatedStrings;
import static ru.yandex.direct.core.entity.client.Constants.DEFAULT_DISABLED_PLACES_COUNT_LIMIT;
import static ru.yandex.direct.core.entity.pricepackage.service.PricePackagePermissionUtils.campaignOptionsCanBeChanged;
import static ru.yandex.direct.core.entity.pricepackage.service.PricePackagePermissionUtils.equalsIgnoringGeoExpanded;
import static ru.yandex.direct.core.entity.pricepackage.service.PricePackagePermissionUtils.retargetingCategoriesAmountIsNotGreaterUpperLimit;
import static ru.yandex.direct.core.entity.pricepackage.service.PricePackagePermissionUtils.retargetingCategoriesAmountIsNotLessLowerLimit;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.clientCurrencyNotEqualsPackageCurrency;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.creativeTemplatesAndAdGroupTypesMismatched;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.creativeTemplatesIsEmpty;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.cryptaTypesCountIsMismatched;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.geoExpandedIsEmpty;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.mutualExcusiveModifiersSelected;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.overlappingPriceMarkups;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.retargetingCategoriesAmountGreaterUpperLimit;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.retargetingCategoriesAmountLessLowerLimit;
import static ru.yandex.direct.core.entity.pricepackage.service.validation.defects.PricePackageDefects.userTimestampNotEqualsLastUpdateTime;
import static ru.yandex.direct.core.entity.region.validation.RegionIdDefects.geoIncorrectRegions;
import static ru.yandex.direct.core.validation.defects.RightsDefects.forbiddenToChange;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.StringUtils.joinLongsToString;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentState;
import static ru.yandex.direct.validation.defect.CommonDefects.invalidValue;

@ParametersAreNonnullByDefault
public class PricePackageConstraints {
    // Список пропертей, которые можно менять у заархивированного пакета
    private static final Set<ModelProperty<? super PricePackage, ?>> CAN_CHANGE_IN_ARCHIVED_PACKAGE =
            Set.of(
                    PricePackage.IS_ARCHIVED
            );
    // Список пропертей, которые нельзя менять до того, как пакет заапрувлен
    private static final Set<ModelProperty<? super PricePackage, ?>> CAN_NOT_CHANGE_BEFORE_APPROVE =
            Set.of(
                    PricePackage.CLIENTS);
    // Список пропертей, которые нельзя менять после того, как пакет заапрувлен
    private static final Set<ModelProperty<? super PricePackage, ?>> CAN_NOT_CHANGE_AFTER_APPROVE =
            Set.of(
                    PricePackage.TRACKER_URL,
                    PricePackage.PRICE,
                    PricePackage.CURRENCY,
                    PricePackage.ORDER_VOLUME_MIN,
                    PricePackage.ORDER_VOLUME_MAX,
                    PricePackage.TARGETINGS_FIXED,
                    PricePackage.TARGETINGS_CUSTOM,
                    PricePackage.IS_PUBLIC,
                    PricePackage.CAMPAIGN_AUTO_APPROVE,
                    PricePackage.AVAILABLE_AD_GROUP_TYPES,
                    PricePackage.BID_MODIFIERS,
                    PricePackage.PRODUCT_ID,
                    PricePackage.IS_CPD
            );

    // Мапа для каждой property задающая пермиссию на доступ оператора.
    // Для краткости - если в мапе нет значения, значит используется пермиссия по умолчанию canManagePricePackages
    private static final Map<ModelProperty<? super PricePackage, ?>, Predicate<User>> OPERATOR_PERMISSIONS =
            Map.of(PricePackage.STATUS_APPROVE,
                    operator -> PricePackagePermissionUtils.canManagePricePackages(operator)
                            || PricePackagePermissionUtils.canApprovePricePackages(operator),
                    PricePackage.IS_ARCHIVED, PricePackagePermissionUtils::canApprovePricePackages,
                    PricePackage.CLIENTS, PricePackagePermissionUtils::canManagePricePackageClients);

    private static boolean markupsCanBeChanged(
            PricePackage pricePackage,
            List<PriceMarkup> newMarkups, LocalDate endDate) {

        if (pricePackage.getStatusApprove() != StatusApprove.YES) {
            return true;
        }

        //если дата их: входит в диапазон продукта + дата старта "завтрашний" день (при условии что завтра тоже
        // входит в даты продукта)

        Set<PriceMarkup> existedMarkups = new HashSet<>(pricePackage.getPriceMarkups());

        for (PriceMarkup markup : newMarkups) {
            if (existedMarkups.contains(markup)) {
                continue;
            }

            if (markup.getDateStart().compareTo(pricePackage.getDateStart()) < 0
                    || markup.getDateEnd().compareTo(endDate) > 0
                    || LocalDate.now().plusDays(1).compareTo(markup.getDateStart()) > 0) {
                return false;
            }
        }

        return true;
    }

    public static Constraint<ModelChanges<PricePackage>, Defect> forbiddenToChangeFieldsAreNotChanged(
            Map<Long, PricePackage> dbPricePackages,
            PricePackageModelChangesValidationContext context) {

        return Constraint.fromPredicate(modelChanges -> {
                    PricePackage pricePackage = dbPricePackages.get(modelChanges.getId());
                    List<ModelProperty<? super PricePackage, ?>> changedProperties =
                            ValidationUtils.getActuallyChangedProperties(pricePackage, modelChanges);
                    changedProperties.removeIf(property ->
                            hasGeoExpandedAndEquals(property, pricePackage, modelChanges));

                    if (changedProperties.contains(PricePackage.PRICE_MARKUPS)) {
                        var endDate = changedProperties.contains(PricePackage.DATE_END) ?
                                modelChanges.getChangedProp(PricePackage.DATE_END) : pricePackage.getDateEnd();

                        if (!markupsCanBeChanged(pricePackage,
                                modelChanges.getChangedProp(PricePackage.PRICE_MARKUPS), endDate)) {
                            return false;
                        }
                    }

                    if (changedProperties.contains(PricePackage.CAMPAIGN_OPTIONS)
                        && !campaignOptionsCanBeChanged(pricePackage,
                            modelChanges.getChangedProp(PricePackage.CAMPAIGN_OPTIONS))) {
                        return false;
                    }

                    return changedProperties.stream().allMatch(modelProperty ->
                            canChangeProperty(modelProperty, pricePackage, context.getOperator()));
                },
                forbiddenToChange());
    }

    public static Constraint<List<BidModifier>, Defect> validBidModifiersByFeatures(
            Set<String> enabledFeatures,
            List<BidModifier> bidModifiers) {

        return Constraint.fromPredicate(modelChanges -> {
                BidModifierInventory bidModifierInventory = extractBidModifierInventory(bidModifiers);
                if (bidModifierInventory == null) {
                    return true;
                }
                Set<InventoryType> allowedOnPackageInventoryTypes = bidModifierInventory.getInventoryAdjustments()
                        .stream()
                        .map(BidModifierInventoryAdjustment::getInventoryType)
                        .collect(Collectors.toSet());
                boolean usingExpandedInventoryBidModifiers = allowedOnPackageInventoryTypes.stream().anyMatch(
                        el -> FunctionalUtils.flatMap(ACTUAL_INVENTORY_WIDE_TO_NARROW.values(), identity()).contains(el));
                return !usingExpandedInventoryBidModifiers ||
                        enabledFeatures.contains(FeatureName.EXPAND_INVENTORY_BID_MODIFIERS.getName());
            },
            invalidValue());
    }

    private static boolean hasGeoExpandedAndEquals(
            ModelProperty<? super PricePackage, ?> property,
            PricePackage pricePackage,
            ModelChanges<PricePackage> modelChanges) {
        if (property == PricePackage.TARGETINGS_FIXED) {
            return equalsIgnoringGeoExpanded(pricePackage.getTargetingsFixed(),
                    modelChanges.getChangedProp(PricePackage.TARGETINGS_FIXED));
        }
        if (property == PricePackage.TARGETINGS_CUSTOM) {
            return equalsIgnoringGeoExpanded(pricePackage.getTargetingsCustom(),
                    modelChanges.getChangedProp(PricePackage.TARGETINGS_CUSTOM));
        }
        return false;
    }

    private static boolean canChangeProperty(ModelProperty<? super PricePackage, ?> modelProperty,
                                             PricePackage pricePackage,
                                             User operator) {
        if (pricePackage.getIsArchived() && !CAN_CHANGE_IN_ARCHIVED_PACKAGE.contains(modelProperty)) {
            return false;
        }
        if (pricePackage.getStatusApprove() == StatusApprove.YES
                && CAN_NOT_CHANGE_AFTER_APPROVE.contains(modelProperty)) {
            return false;
        }
        if (pricePackage.getStatusApprove() != StatusApprove.YES
                && CAN_NOT_CHANGE_BEFORE_APPROVE.contains(modelProperty)) {
            return false;
        }
        return OPERATOR_PERMISSIONS.getOrDefault(modelProperty, PricePackagePermissionUtils::canManagePricePackages).
                test(operator);
    }

    public static <T> Constraint<T, Defect> packageNotApproved(PricePackage pricePackage) {
        return ignored -> pricePackage.getStatusApprove() == StatusApprove.YES
                ? inconsistentState()
                : null;
    }

    public static <T> Constraint<T, Defect> userTimestampEqualsLastUpdateTime(
            LocalDateTime userTimestamp,
            PricePackage pricePackage) {
        return ignored -> {
            var pricePackageLastUpdateTime = pricePackage.getLastUpdateTime();
            return pricePackageLastUpdateTime.equals(userTimestamp)
                    ? null
                    : userTimestampNotEqualsLastUpdateTime(pricePackageLastUpdateTime);
        };
    }

    public static Constraint<ModelChanges<PricePackage>, Defect> userTimestampEqualsLastUpdateTime(
            Map<Long, LocalDateTime> userTimestamps,
            Map<Long, PricePackage> dbPricePackages) {
        return modelChanges -> {
            var pricePackageId = modelChanges.getId();
            var userTimestamp = userTimestamps.get(pricePackageId);
            var pricePackage = dbPricePackages.get(pricePackageId);
            return userTimestampEqualsLastUpdateTime(userTimestamp, pricePackage).apply(pricePackageId);
        };
    }

    public static Constraint<List<BidModifier>, Defect> platformBidModifiers() {
        return modifiers -> {
            if (modifiers == null) {
                return null;
            }
            long cnt = modifiers.stream().filter(modifier -> modifier.getType() == BidModifierType.MOBILE_MULTIPLIER &&
                    nvl(((BidModifierMobile)modifier).getMobileAdjustment().getIsRequiredInPricePackage(), true))
                    .count();

            return cnt > 1 ? mutualExcusiveModifiersSelected() : null;
        };
    }

    public static Constraint<Long, Defect> clientCurrencyEqualsPackageCurrency(
            PricePackage pricePackage,
            Map<Long, CurrencyCode> clientCurrencies) {
        return clientId -> {
            var clientCurrency = clientCurrencies.get(clientId);
            var packageCurrency = pricePackage.getCurrency();
            return packageCurrency != null && clientCurrency != packageCurrency
                    ? clientCurrencyNotEqualsPackageCurrency()
                    : null;
        };
    }

    public static Constraint<PriceRetargetingCondition, Defect> isMatchingCryptaTypesCount() {
        return retargetingCondition -> {
            var lowerCryptaTypesCount = retargetingCondition.getLowerCryptaTypesCount();
            var upperCryptaTypesCount = retargetingCondition.getUpperCryptaTypesCount();
            return upperCryptaTypesCount != 0 && lowerCryptaTypesCount > upperCryptaTypesCount
                    ? cryptaTypesCountIsMismatched()
                    : null;
        };
    }

    public static Constraint<List<Long>, Defect> fixedIsSubsetOfCustom(@Nullable TargetingsCustom targetingsCustom) {
        var customGeo = targetingsCustom == null || targetingsCustom.getGeo() == null ?
                emptySet() : listToSet(targetingsCustom.getGeo());
        return fixedGeo -> customGeo.containsAll(fixedGeo) ?
                null :
                geoIncorrectRegions(joinLongsToString(
                        StreamEx.of(fixedGeo)
                                .filter(geo -> !customGeo.contains(geo))
                                .toList()
                ));
    }

    public static Constraint<PriceRetargetingCondition, Defect> isMatchingRetargetingCategoriesCount() {
        return retargetingCondition ->
                retargetingCategoriesAmountIsNotLessLowerLimit(retargetingCondition)
                        ? null
                        : retargetingCategoriesAmountLessLowerLimit();
    }

    public static Constraint<List<Long>, Defect> isMatchingFixedRetargetingCategoriesCount(
            PriceRetargetingCondition retargetingCondition) {
        return cryptaGoalIds ->
                retargetingCategoriesAmountIsNotGreaterUpperLimit(cryptaGoalIds, retargetingCondition)
                        ? null
                        : retargetingCategoriesAmountGreaterUpperLimit();
    }

    public static Constraint<List<Long>, Defect> geoExpandedIsNotEmpty(@Nullable List<Long> geoExpanded) {
        return ignored -> geoExpanded == null || geoExpanded.size() == 0 ?
                geoExpandedIsEmpty() :
                null;
    }

    public static Constraint<Map<CreativeType, List<Long>>, Defect> atLeastOneTemplateIsChosen() {
        return allowedCreativeTemplates -> {
            var creativeTemplates = allowedCreativeTemplates.values().stream()
                    .flatMap(List::stream)
                    .collect(Collectors.toList());
            return creativeTemplates.isEmpty()
                    ? creativeTemplatesIsEmpty()
                    : null;
        };
    }

    public static Constraint<Map<CreativeType, List<Long>>, Defect> isMatchingCreativeTemplatesAndAdGroupTypes(
            Set<AdGroupType> adGroupTypes,
            Map<AdGroupType, Set<CreativeType>> accessibleCreativeTypesByAdGroupType) {
        var adGroupTypesWithCreativeTypesRestrictions = accessibleCreativeTypesByAdGroupType.keySet();
        boolean skipCreativeTypesRestrictions = adGroupTypes.stream()
                .anyMatch(adGroupType -> !adGroupTypesWithCreativeTypesRestrictions.contains(adGroupType));
        Set<CreativeType> accessibleCreativeTypes = skipCreativeTypesRestrictions
                ? Set.of()
                : adGroupTypes.stream()
                .map(adGroupType -> accessibleCreativeTypesByAdGroupType.getOrDefault(adGroupType, Set.of()))
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());

        return fromPredicate(creativeTemplates -> {
                    var isAnyTemplateMismatched = creativeTemplates.entrySet().stream()
                            .anyMatch(entry -> {
                                var creativeType = entry.getKey();
                                var creativeIds = entry.getValue();
                                return creativeIds != null
                                        && !creativeIds.isEmpty()
                                        && !skipCreativeTypesRestrictions
                                        && !accessibleCreativeTypes.contains(creativeType);
                            });
                    return !isAnyTemplateMismatched;
                },
                creativeTemplatesAndAdGroupTypesMismatched());
    }

    public static Constraint<List<String>, Defect> uniqueDomains() {
        return domains -> {
            if (domains == null) {
                return null;
            }
            Set<String> duplicated = domains.stream().filter(i -> Collections.frequency(domains, i) > 1)
                    .collect(Collectors.toSet());
            if (duplicated.isEmpty()) {
                return null;
            }
            return duplicatedStrings(StreamEx.of(duplicated).toList());
        };
    }

    public static Constraint<List<String>, Defect> totalAllowedDomainsMaxSize(List<String> ssp) {
        return Constraint.fromPredicate(
                domains -> ListUtils.sum(nvl(domains, emptyList()), nvl(ssp, emptyList()))
                        .size() <= DEFAULT_DISABLED_PLACES_COUNT_LIMIT,
                CollectionDefects.maxCollectionSize(DEFAULT_DISABLED_PLACES_COUNT_LIMIT));
    }

    public static Constraint<List<PriceMarkup>, Defect> nonOverlappingMarkups() {
        return priceMarkups -> {
            if (priceMarkups == null) {
                return null;
            }
            var sortedMarkups = priceMarkups.stream()
                    .sorted(Comparator.comparing(PriceMarkup::getDateStart))
                    .toArray(PriceMarkup[]::new);
            for (var i = 1; i < sortedMarkups.length; i++) {
                if (!sortedMarkups[i].getDateStart().isAfter(sortedMarkups[i - 1].getDateEnd())) {
                    return overlappingPriceMarkups();
                }
            }
            return null;
        };
    }


    public static Constraint<Boolean, Defect> isCpdAllowed(PricePackage pricePackage) {
        return isCpd -> {
            if (isCpd == null) {
                return null;
            }
            var isCpdCondition = pricePackage.isFrontpagePackage()
                    && Optional.ofNullable(pricePackage.getTargetingsFixed())
                        .map(TargetingsFixed::getCryptaSegments)
                        .map(List::isEmpty)
                        .orElse(true)
                    && Optional.ofNullable(pricePackage.getTargetingsCustom())
                        .map(TargetingsCustom::getRetargetingCondition)
                        .map(PriceRetargetingCondition::getCryptaSegments)
                        .map(List::isEmpty)
                        .orElse(true);
            if ((!isCpd) || isCpdCondition) {
                return null;
            } else {
                return new Defect<>(DefectIds.INVALID_VALUE);
            }
        };
    }
}
