package ru.yandex.direct.core.entity.product;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.product.model.ProductType;
import ru.yandex.direct.currency.CurrencyCode;

import static java.util.Map.entry;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.TEXT_CAMPAIGN_IN_BALANCE;

@ParametersAreNonnullByDefault
public class ProductConstants {
    private ProductConstants() {
    }

    private static final Map<CampaignType, ProductType> TEXT_PRODUCT_TYPE_BY_CAMPAIGN_TYPES =
            StreamEx.of(TEXT_CAMPAIGN_IN_BALANCE)
                    .mapToEntry(campaignType -> ProductType.TEXT)
                    .toMap();

    /**
     * Возвращает дополнительные типы продуктов, которые могут показываться в кампаниях определенного типа с определенной валютой
     * @param currencyCode - валюта, для которой проверяются типы продуктов
     * @return дополнительные типы продуктов для типов разных кампании.
     */
    public static Map<CampaignType, List<ProductType>> getSpecialProductTypesByCampaignType(CurrencyCode currencyCode) {
        return SPECIAL_PRODUCT_TYPES_BY_CAMPAIGN_TYPE_BY_CURRENCY
                .getOrDefault(currencyCode, DEFAULT_SPECIAL_PRODUCT_TYPES_BY_CAMPAIGN_TYPE);
    }

    /**
     * Direct::BillingAggregates %SPECIAL_PRODUCT_TYPES_BY_CAMP_TYPE
     * <p>
     * Дополнительные типы продуктов, которые могут показываться в кампаниях определенного типа.
     * Например, при создании кампании типа cpm_banner, нужно создать биллинговый агрегат с типом
     * cpm_banner, и, дополнительно, с типом cpm_video, т.к. в этой кампании могут появиться
     * рекламные материалы этого типа.
     * <p>
     * При добавлении новых типов продукта в уже существующие типы кампаний, нужно запустить
     * <TODO: некоторый скрипт>, который досоздаст биллинговые агрегаты всем клиентам, у которых
     * уже есть кампании этого типа.
     **/
    private static final Map<CampaignType, List<ProductType>> DEFAULT_SPECIAL_PRODUCT_TYPES_BY_CAMPAIGN_TYPE =
            Map.of(CampaignType.CPM_BANNER, List.of(
                    ProductType.CPM_VIDEO,
                    ProductType.CPM_OUTDOOR,
                    ProductType.CPM_INDOOR,
                    ProductType.CPM_AUDIO
            ));

    private static final Map<CurrencyCode, Map<CampaignType, List<ProductType>>> SPECIAL_PRODUCT_TYPES_BY_CAMPAIGN_TYPE_BY_CURRENCY =
            Map.of(CurrencyCode.GBP, Map.of(CampaignType.CPM_BANNER, List.of(
                    ProductType.CPM_VIDEO,
                    ProductType.CPM_OUTDOOR,
                    ProductType.CPM_INDOOR
            )));

    /**
     * Settings::BILLING_AGGREGATES_AUTO_CREATE_TYPES
     * Типы продуктов, для которых мы хотим автоматически создавать биллинговые агрегаты
     * при создании кампании.
     */
    public static final Set<ProductType> BILLING_AGGREGATES_AUTO_CREATE_TYPES = Set.of(
            ProductType.CPM_BANNER,
            ProductType.CPM_VIDEO,
            ProductType.CPM_AUDIO,
            ProductType.CPM_OUTDOOR,
            ProductType.CPM_INDOOR
    );

    private static final Map<CampaignType, ProductType> NOT_TEXT_PRODUCT_TYPE_BY_CAMPAIGN_TYPES = Map.ofEntries(
            entry(CampaignType.CPM_BANNER, ProductType.CPM_BANNER),
            entry(CampaignType.CPM_DEALS, ProductType.CPM_DEALS),
            entry(CampaignType.CPM_YNDX_FRONTPAGE, ProductType.CPM_YNDX_FRONTPAGE),
            entry(CampaignType.INTERNAL_AUTOBUDGET, ProductType.INTERNAL_AUTOBUDGET),
            entry(CampaignType.INTERNAL_DISTRIB, ProductType.INTERNAL_DISTRIB),
            entry(CampaignType.INTERNAL_FREE, ProductType.INTERNAL_FREE),
            entry(CampaignType.GEO, ProductType.GEO),
            entry(CampaignType.CPM_PRICE, ProductType.CPM_PRICE)
    );

    public static final Map<CampaignType, ProductType> PRODUCT_TYPE_BY_CAMPAIGN_TYPES =
            EntryStream.of(TEXT_PRODUCT_TYPE_BY_CAMPAIGN_TYPES)
                    .append(NOT_TEXT_PRODUCT_TYPE_BY_CAMPAIGN_TYPES)
                    .toMap();

    /**
     * Direct::BillingAggregates::AGGREGATE_NAME_BY_PRODUCT_TYPE
     * Названия заказов-агрегатов в Биллинге
     **/
    public static final Map<ProductType, String> AGGREGATE_NAME_BY_PRODUCT_TYPE = Map.ofEntries(
            entry(ProductType.TEXT, "Общий счет: покликовый Директ"),
            entry(ProductType.CPM_BANNER, "Общий счет: медийный баннер"),
            entry(ProductType.CPM_DEALS, "Общий счет: частные сделки"),
            entry(ProductType.CPM_VIDEO, "Общий счет: медийное видео"),
            entry(ProductType.CPM_OUTDOOR, "Общий счет: outdoor видео"),
            entry(ProductType.CPM_YNDX_FRONTPAGE, "Общий счет: баннер на главной"),
            entry(ProductType.CPM_INDOOR, "Общий счет: indoor видео"),
            entry(ProductType.CPM_AUDIO, "Общий счет: медийное аудио"),
            entry(ProductType.CPM_PRICE, "Общий счет: прайсовые продажи")
    );

}
