package ru.yandex.direct.core.entity.product.repository;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.product.model.Product;
import ru.yandex.direct.core.entity.product.model.ProductCalcType;
import ru.yandex.direct.core.entity.product.model.ProductRestriction;
import ru.yandex.direct.core.entity.product.model.ProductType;
import ru.yandex.direct.core.entity.product.model.ProductUnit;
import ru.yandex.direct.currency.Currencies;
import ru.yandex.direct.dbschema.ppcdict.enums.ProductsCurrency;
import ru.yandex.direct.dbschema.ppcdict.enums.ProductsType;
import ru.yandex.direct.dbschema.ppcdict.tables.records.ProductRestrictionsRecord;
import ru.yandex.direct.dbschema.ppcdict.tables.records.ProductsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.dbschema.ppcdict.tables.BusinessUnitsInfo.BUSINESS_UNITS_INFO;
import static ru.yandex.direct.dbschema.ppcdict.tables.ProductRestrictions.PRODUCT_RESTRICTIONS;
import static ru.yandex.direct.dbschema.ppcdict.tables.Products.PRODUCTS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.CollectionUtils.isEmpty;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Репозиторий для работы с продуктами из таблицы ppcdict.products
 */
@Repository
@ParametersAreNonnullByDefault
public class ProductRepository {
    private final DslContextProvider databaseWrapperProvider;

    public final JooqMapperWithSupplier<Product> productJooqMapper;
    private final JooqMapperWithSupplier<ProductRestriction> productRestrictionJooqMapper;
    private static final Set<ProductsCurrency> supportedCurrencies = Arrays.stream(ProductsCurrency.values())
            .filter(t -> Currencies.currencyExists(t.getLiteral()))
            .collect(Collectors.toSet());

    @Autowired
    public ProductRepository(DslContextProvider databaseWrapperProvider) {
        this.databaseWrapperProvider = databaseWrapperProvider;

        productJooqMapper = JooqMapperWithSupplierBuilder.builder(Product::new)
                .map(property(Product.ID, PRODUCTS.PRODUCT_ID))
                .map(property(Product.PRODUCT_NAME, PRODUCTS.PRODUCT_NAME))
                .map(property(Product.PUBLIC_NAME_KEY, PRODUCTS.PUBLIC_NAME_KEY))
                .map(property(Product.PUBLIC_DESCRIPTION_KEY, PRODUCTS.PUBLIC_DESCRIPTION_KEY))
                .map(property(Product.THEME_ID, PRODUCTS.THEME_ID))
                .map(convertibleProperty(Product.TYPE, PRODUCTS.TYPE,
                        ProductType::fromSource, ProductType::toSource))
                .map(property(Product.PRICE, PRODUCTS.PRICE))
                .map(convertibleProperty(Product.CURRENCY_CODE, PRODUCTS.CURRENCY,
                        ProductMapping::currencyFromDb, ProductMapping::currencyToDb))
                .map(booleanProperty(Product.VAT, PRODUCTS.NDS))
                .map(convertibleProperty(Product.UNIT, PRODUCTS.UNIT,
                        ProductUnit::fromSource, ProductUnit::toSource))
                .map(property(Product.UNIT_NAME, PRODUCTS.UNIT_NAME))
                .map(property(Product.UNIT_SCALE, PRODUCTS.UNIT_SCALE))
                .map(property(Product.ENGINE_ID, PRODUCTS.ENGINE_ID))
                .map(property(Product.RATE, PRODUCTS.RATE))
                .map(property(Product.DAILY_SHOWS, PRODUCTS.DAILY_SHOWS))
                .map(property(Product.PACKET_SIZE, PRODUCTS.PACKET_SIZE))
                .map(property(Product.BUSINESS_UNIT, PRODUCTS.BUSINESS_UNIT))
                .map(property(Product.BUSINESS_UNIT_NAME, BUSINESS_UNITS_INFO.NAME))
                .map(convertibleProperty(Product.CALC_TYPE, PRODUCTS.CALC_TYPE,
                        ProductCalcType::fromSource, ProductCalcType::toSource))
                .build();

        productRestrictionJooqMapper = JooqMapperWithSupplierBuilder.builder(ProductRestriction::new)
                .map(property(ProductRestriction.ID, PRODUCT_RESTRICTIONS.ID))
                .map(property(ProductRestriction.PRODUCT_ID, PRODUCT_RESTRICTIONS.PRODUCT_ID))
                .map(convertibleProperty(ProductRestriction.GROUP_TYPE, PRODUCT_RESTRICTIONS.ADGROUP_TYPE,
                        fromDb -> AdGroupType.valueOf(fromDb.toUpperCase()), toDb -> toDb.name().toLowerCase()))
                .map(property(ProductRestriction.PUBLIC_NAME_KEY, PRODUCT_RESTRICTIONS.PUBLIC_NAME_KEY))
                .map(property(ProductRestriction.PUBLIC_DESCRIPTION_KEY, PRODUCT_RESTRICTIONS.PUBLIC_DESCRIPTION_KEY))
                .map(property(ProductRestriction.CONDITION_JSON, PRODUCT_RESTRICTIONS.CONDITION_JSON))
                .map(property(ProductRestriction.UNIT_COUNT_MIN, PRODUCT_RESTRICTIONS.UNIT_COUNT_MIN))
                .map(property(ProductRestriction.UNIT_COUNT_MAX, PRODUCT_RESTRICTIONS.UNIT_COUNT_MAX))
                .build();
    }

    /**
     * Получить все записи о продуктах из таблицы ppcdict.products
     */
    public List<Product> getAllProducts() {
        return databaseWrapperProvider.ppcdict()
                .select(PRODUCTS.PRODUCT_ID, PRODUCTS.PRODUCT_NAME, PRODUCTS.PUBLIC_NAME_KEY,
                        PRODUCTS.PUBLIC_DESCRIPTION_KEY, PRODUCTS.THEME_ID, PRODUCTS.TYPE, PRODUCTS.PRICE,
                        PRODUCTS.CURRENCY, PRODUCTS.NDS, PRODUCTS.UNIT, PRODUCTS.UNIT_NAME, PRODUCTS.UNIT_SCALE,
                        PRODUCTS.ENGINE_ID, PRODUCTS.RATE, PRODUCTS.DAILY_SHOWS, PRODUCTS.PACKET_SIZE,
                        PRODUCTS.CALC_TYPE, PRODUCTS.BUSINESS_UNIT, BUSINESS_UNITS_INFO.NAME)
                .from(PRODUCTS)
                .leftJoin(BUSINESS_UNITS_INFO)
                .on(PRODUCTS.BUSINESS_UNIT.eq(BUSINESS_UNITS_INFO.ID))
                .where(PRODUCTS.CURRENCY.in(supportedCurrencies))
                .fetch()
                .map(productJooqMapper::fromDb);
    }

    /**
     * Получить все записи о продуктах из таблицы ppcdict.products по списку id продуктов
     */
    public List<Product> getProductsById(Collection<Long> productIds) {
        if (isEmpty(productIds)) {
            return Collections.emptyList();
        }

        return databaseWrapperProvider.ppcdict()
                .select(PRODUCTS.PRODUCT_ID, PRODUCTS.PRODUCT_NAME, PRODUCTS.PUBLIC_NAME_KEY,
                        PRODUCTS.PUBLIC_DESCRIPTION_KEY, PRODUCTS.THEME_ID, PRODUCTS.TYPE, PRODUCTS.PRICE,
                        PRODUCTS.CURRENCY, PRODUCTS.NDS, PRODUCTS.UNIT, PRODUCTS.UNIT_NAME, PRODUCTS.UNIT_SCALE,
                        PRODUCTS.ENGINE_ID, PRODUCTS.RATE, PRODUCTS.DAILY_SHOWS, PRODUCTS.PACKET_SIZE,
                        PRODUCTS.CALC_TYPE, PRODUCTS.BUSINESS_UNIT, BUSINESS_UNITS_INFO.NAME)
                .from(PRODUCTS)
                .leftJoin(BUSINESS_UNITS_INFO)
                .on(PRODUCTS.BUSINESS_UNIT.eq(BUSINESS_UNITS_INFO.ID))
                .where(PRODUCTS.PRODUCT_ID.in(productIds))
                .fetch()
                .map(productJooqMapper::fromDb);
    }

    public Map<Long, String> getBusinessUnitNames() {
        return databaseWrapperProvider.ppcdict()
                .select(BUSINESS_UNITS_INFO.ID, BUSINESS_UNITS_INFO.NAME)
                .from(BUSINESS_UNITS_INFO)
                .fetchMap(BUSINESS_UNITS_INFO.ID, BUSINESS_UNITS_INFO.NAME);
    }

    /**
     * Получить все записи об ограничениях на продуктах из таблицы ppcdict.product_restrictions
     */
    public List<ProductRestriction> getAllProductRestrictions() {
        return databaseWrapperProvider.ppcdict()
                .select(productRestrictionJooqMapper.getFieldsToRead())
                .from(PRODUCT_RESTRICTIONS).fetch().map(productRestrictionJooqMapper::fromDb);
    }

    /**
     * Получить все записи об ограничениях на продуктах и залочить внутри транзакции для обновления
     */
    public List<ProductRestriction> getAllProductRestrictionsForUpdate(DSLContext dslContext) {
        return dslContext
                .select(productRestrictionJooqMapper.getFieldsToRead())
                .from(PRODUCT_RESTRICTIONS)
                .forUpdate()
                .fetch().map(productRestrictionJooqMapper::fromDb);
    }

    public void addProductRestrictions(DSLContext dslContext, Collection<ProductRestriction> productRestrictions) {
        InsertHelper<ProductRestrictionsRecord> insertHelper =
                new InsertHelper<>(dslContext, PRODUCT_RESTRICTIONS);

        insertHelper.addAll(productRestrictionJooqMapper, productRestrictions);
        insertHelper.execute();
    }

    private void insertNewProductsInTransaction(DSLContext dslContext, Collection<Product> products) {
        List<Long> ids = mapList(products, Product::getId);

        Set<Long> existingProducts = dslContext.select(PRODUCTS.PRODUCT_ID)
                .from(PRODUCTS)
                .where(PRODUCTS.PRODUCT_ID.in(ids))
                .fetchSet(PRODUCTS.PRODUCT_ID);

        InsertHelper<ProductsRecord> insertHelper =
                new InsertHelper<>(dslContext, PRODUCTS);

        insertHelper.addAll(productJooqMapper,
                products.stream().filter(e -> !existingProducts.contains(e.getId())).collect(Collectors.toList()));

        insertHelper.executeIfRecordsAdded();
    }

    public void updateBusinessUnit(DSLContext dslContext, long productId, long businessUnitId) {
        dslContext.update(PRODUCTS).set(PRODUCTS.BUSINESS_UNIT, businessUnitId)
                .where(
                        PRODUCTS.PRODUCT_ID.eq(productId),
                        PRODUCTS.TYPE.eq(ProductsType.auto_import)
                )
                .execute();
    }

    public void insertNewProducts(DSLContext dslContext, Collection<Product> products) {

        dslContext.transaction(
                configuration ->
                        insertNewProductsInTransaction(configuration.dsl(), products)
        );

    }

    /**
     * Обновление записей об ограничениях продукта
     */
    public void updateProductRestrictions(
            DSLContext dslContext, Collection<AppliedChanges<ProductRestriction>> changes
    ) {
        JooqUpdateBuilder<ProductRestrictionsRecord, ProductRestriction> ub =
                new JooqUpdateBuilder<>(PRODUCT_RESTRICTIONS.ID, changes);

        ub.processProperty(ProductRestriction.PRODUCT_ID, PRODUCT_RESTRICTIONS.PRODUCT_ID);
        ub.processProperty(
                ProductRestriction.GROUP_TYPE,
                PRODUCT_RESTRICTIONS.ADGROUP_TYPE, toDb -> toDb.name().toLowerCase()
        );
        ub.processProperty(ProductRestriction.PUBLIC_NAME_KEY, PRODUCT_RESTRICTIONS.PUBLIC_NAME_KEY);
        ub.processProperty(ProductRestriction.PUBLIC_DESCRIPTION_KEY, PRODUCT_RESTRICTIONS.PUBLIC_DESCRIPTION_KEY);
        ub.processProperty(ProductRestriction.CONDITION_JSON, PRODUCT_RESTRICTIONS.CONDITION_JSON);
        ub.processProperty(ProductRestriction.UNIT_COUNT_MIN, PRODUCT_RESTRICTIONS.UNIT_COUNT_MIN);
        ub.processProperty(ProductRestriction.UNIT_COUNT_MAX, PRODUCT_RESTRICTIONS.UNIT_COUNT_MAX);

        dslContext.update(PRODUCT_RESTRICTIONS)
                .set(ub.getValues())
                .where(PRODUCT_RESTRICTIONS.ID.in(ub.getChangedIds()))
                .execute();
    }
}
