package ru.yandex.direct.core.entity.product.repository;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Suppliers;
import com.google.common.collect.ImmutableMap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.product.model.Product;
import ru.yandex.direct.core.entity.product.model.ProductRestriction;
import ru.yandex.direct.core.entity.product.model.ProductSimple;
import ru.yandex.direct.core.entity.product.model.ProductType;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.groupingBy;

/**
 * Кэш для элементов таблицы продуктов.
 * Является надстройкой над {@link ProductRepository},
 * Отдаем продукт по интерфейсу для защиты объекта от изменений, чтобы не испортили кеш
 */
@Service
@ParametersAreNonnullByDefault
public class ProductsCache {

    static final long YND_FIXED_FOR_TEXT_TYPE_PRODUCT_ID = 1475L;
    private static final Duration RELOAD_INTERVAL = Duration.ofMinutes(5);

    private final ProductRepository productRepository;

    private volatile Supplier<Map<Long, Product>> productsSupplier;
    private volatile Supplier<Map<Long, List<ProductRestriction>>> productRestrictionsSupplier;

    @Autowired
    public ProductsCache(ProductRepository productRepository) {
        this.productRepository = productRepository;
        invalidate();
    }

    /**
     * Инвалидировать кэш. Нужно в случае добавления новых продуктов в базу данных.
     */
    public void invalidate() {
        productsSupplier =
                Suppliers.memoizeWithExpiration(this::getData, RELOAD_INTERVAL.toMillis(), TimeUnit.MILLISECONDS);
        this.productRestrictionsSupplier =
                Suppliers.memoizeWithExpiration(
                        () -> productRepository.getAllProductRestrictions().stream()
                                .collect(groupingBy(ProductRestriction::getProductId)),
                        RELOAD_INTERVAL.toMillis(), TimeUnit.MILLISECONDS);
    }

    Map<Long, Product> getData() {
        return productRepository.getAllProducts().stream()
                .filter(this::isNotYndFixedProductForGeo)
                .collect(ImmutableMap.toImmutableMap(ProductSimple::getId, identity()));
    }

    /**
     * Продукт не фишковый и не для Геоконтекста
     * В проде с ProductID = {@link #YND_FIXED_FOR_TEXT_TYPE_PRODUCT_ID} есть две записи.
     * Отличаются только названиеми типом
     * Вторую запись игнорим, чтобы поиск по productId был однозначным
     * В светлом будущем, когда из базы можно будет удалить продукт для гео, то это проверка будет не нужна
     */
    private boolean isNotYndFixedProductForGeo(Product product) {
        return !(product.getId() == YND_FIXED_FOR_TEXT_TYPE_PRODUCT_ID && product.getType() == ProductType.GEO);
    }

    /**
     * Получить кэшированную запись продукта из базы.
     *
     * @see #isNotYndFixedProductForGeo(Product)
     */
    public ProductSimple getProductById(Long productId) {
        return productsSupplier.get().get(productId);
    }

    /**
     * Кешированый список продуктов из базы
     */
    public List<ProductSimple> getSimpleProducts() {
        return new ArrayList<>(productsSupplier.get().values());
    }

    /**
     * Кешированый список продуктов из базы
     */
    public List<Product> getProducts() {
        return new ArrayList<>(productsSupplier.get().values());
    }

    /**
     * Кешированый список ограничений на продуктах из базы
     */
    public Map<Long, List<ProductRestriction>> getProductRestrictions() {
        return new HashMap<>(productRestrictionsSupplier.get());
    }

}
