package ru.yandex.direct.core.entity.product.service;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.apache.commons.lang.NotImplementedException;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.CriterionType;
import ru.yandex.direct.core.entity.adgroup.model.ProductRestrictionKey;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CommonCampaign;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackage;
import ru.yandex.direct.core.entity.product.model.Product;
import ru.yandex.direct.core.entity.product.model.ProductRestriction;
import ru.yandex.direct.core.entity.product.model.ProductSimple;
import ru.yandex.direct.core.entity.product.model.ProductType;
import ru.yandex.direct.core.entity.product.repository.ProductRepository;
import ru.yandex.direct.core.entity.product.repository.ProductsCache;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.liveresource.LiveResourceFactory;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.utils.JsonUtils;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Maps.uniqueIndex;
import static java.util.Comparator.comparing;
import static java.util.Comparator.nullsFirst;
import static ru.yandex.direct.core.entity.adgroup.model.CriterionType.CONTENT_CATEGORY;
import static ru.yandex.direct.core.entity.adgroup.model.CriterionType.KEYWORD;
import static ru.yandex.direct.core.entity.adgroup.model.CriterionType.USER_PROFILE;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.CPM;
import static ru.yandex.direct.core.entity.campaign.model.CampaignTypeKinds.TEXT_CAMPAIGN_IN_BALANCE;
import static ru.yandex.direct.core.entity.product.ProductConstants.PRODUCT_TYPE_BY_CAMPAIGN_TYPES;
import static ru.yandex.direct.core.entity.product.ProductConstants.getSpecialProductTypesByCampaignType;

/**
 * Сервис для работы с продуктами из таблицы ppcdict.products
 * <p>
 * При первом читающем обращении получает все продукты из базы и кеширует их локально, дальше все значения отдает из
 * кеша
 *
 * @see ProductsCache
 */
@Service
@ParametersAreNonnullByDefault
public class ProductService {
    public static final String QUASI_CURRENCY = "QuasiCurrency";
    public static final String BUCKS = "Bucks";
    public static final long ZEN_PRODUCT_ID = 513389L;

    private static final Set<ProductType> CAMPAIGN_PRODUCT_TYPES_WITH_CURRENCY = Set.of(
            ProductType.TEXT,
            ProductType.CPM_BANNER,
            ProductType.CPM_VIDEO,
            ProductType.CPM_AUDIO,
            ProductType.CPM_INDOOR,
            ProductType.CPM_OUTDOOR,
            ProductType.CPM_DEALS,
            ProductType.INTERNAL_AUTOBUDGET,
            ProductType.INTERNAL_DISTRIB,
            ProductType.INTERNAL_FREE,
            ProductType.CPM_YNDX_FRONTPAGE,
            ProductType.CPM_PRICE
    );

    private static final Map<String, CriterionType> criterionTypeByPublicNameKeyMap =
            Map.of("banner_text", KEYWORD,
                    "banner", USER_PROFILE,
                    "video", USER_PROFILE,
                    "banner_content_category", CONTENT_CATEGORY);

    private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(ProductService.class);

    public static final String PRODUCT_RESTRICTIONS_FILE = "classpath:///core/product-restrictions.json";
    public static final Comparator<ProductRestrictionKey> PRODUCT_RESTRICTION_KEY_COMPARATOR =
            comparing(ProductRestrictionKey::getAdGroupType)
                    .thenComparing(ProductRestrictionKey::getCriterionType, nullsFirst(Comparator.naturalOrder()))
                    .thenComparingLong(ProductRestrictionKey::getProductId);

    public static final Long NOT_BUSINESS_UNIT_ID = -1L;
    public static final String NOT_BUSINESS_UNIT_NAME = "не бизнес юнит";

    private final ProductsCache productsCache;
    private final ProductRepository productRepository;
    private final DslContextProvider dslContextProvider;

    // не используем читалку JSON из JsonTools, т.к. там разрешается иметь в JSON неизвестные поля.
    // тут же следует падать при неизвестных полях, т.к. это скорее всего следствие опечатки
    private static ObjectMapper productRestrictionsJsonFileMapper = new ObjectMapper().enable(
            JsonParser.Feature.ALLOW_COMMENTS,
            JsonParser.Feature.ALLOW_YAML_COMMENTS,
            JsonParser.Feature.ALLOW_TRAILING_COMMA
    );

    @Autowired
    public ProductService(
            ProductsCache productsCache,
            ProductRepository productRepository,
            DslContextProvider dslContextProvider
    ) {
        this.productsCache = productsCache;
        this.productRepository = productRepository;
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получить модель продукта по его идентификатору
     */
    public ProductSimple getProductById(Long productId) {
        ProductSimple product = productsCache.getProductById(productId);
        if (product == null) {
            throw new NoSuchElementException(String.format("No product found with id %s", productId));
        }
        return product;
    }

    public List<Product> getProducts() {
        return productsCache.getProducts();
    }

    public Map<Long, List<ProductRestriction>> getProductRestrictions() {
        return productsCache.getProductRestrictions();
    }

    /**
     * Получить справочник бизнес-юнитов, добавив фиктивный "не бизнес юнит" с id=-1, означающий все не БЮ продукты.
     */
    public Map<Long, String> getBusinessUnits() {
        Map<Long, String> businessUnits = productRepository.getBusinessUnitNames();
        businessUnits.put(NOT_BUSINESS_UNIT_ID, NOT_BUSINESS_UNIT_NAME);
        return businessUnits;
    }

    /**
     * для прайсовой кампании продукт зависит от настроек пакета
     */
    public Long calculateProductIdByPackage(Client client, PricePackage pricePackage) {
        // 0 или null в productId - признак отсутствия продукта на пакете
        if (pricePackage.getProductId() != null && pricePackage.getProductId() != 0) {
            return pricePackage.getProductId();
        }
        boolean quasiCurrencyFlag =
                client.getWorkCurrency().equals(CurrencyCode.KZT) && client.getUsesQuasiCurrency();
        ProductType productType = pricePackage.isFrontpagePackage() ? ProductType.CPM_PRICE : ProductType.CPM_VIDEO;
        Product product = calculateProduct(CampaignType.CPM_PRICE, productType,
                client.getWorkCurrency(), quasiCurrencyFlag);
        return product.getId();
    }

    public Long calculateProductId(Client client, CommonCampaign campaign) {
        if (campaign.getSource() != null && campaign.getSource() == CampaignSource.ZEN) {
            return ZEN_PRODUCT_ID;
        }
        boolean quasiCurrencyFlag = getClientQuasiCurrencyFlag(client);
        return calculateProductForCampaign(
                campaign.getType(),
                client.getWorkCurrency(),
                quasiCurrencyFlag).getId();
    }

    /**
     * Возвращаем описание продукта для кампаний
     */
    public Set<Product> calculateProductsForCampaigns(
            Set<CampaignType> campaignTypes,
            CurrencyCode currencyCode,
            Boolean quasiCurrencyFlag) {

        Map<ProductType, Set<CampaignType>> productTypeToCampaignTypes = StreamEx.of(campaignTypes)
                .mapToEntry(this::calculateProductType, Function.identity())
                .grouping(Collectors.toSet());

        var specialProductTypesByCampaignType = getSpecialProductTypesByCampaignType(currencyCode);

        //Добавляем типы продуктов которые могут получится при создании групп
        EntryStream.of(specialProductTypesByCampaignType)
                .filterKeys(campaignTypes::contains)
                .forKeyValue((campaignType, specialProductTypes) -> {
                    specialProductTypes.forEach(productType ->
                            productTypeToCampaignTypes.computeIfAbsent(productType, x -> new HashSet<>())
                                    .add(campaignType));
                });

        return calculateProducts(productTypeToCampaignTypes, currencyCode, quasiCurrencyFlag);
    }

    /**
     * Возвращаем описание продукта для кампании
     */
    public ProductSimple calculateProductForCampaign(
            CampaignType campaignType,
            CurrencyCode currencyCode,
            Boolean quasiCurrencyFlag) {

        ProductType productType = calculateProductType(campaignType);

        if (!CAMPAIGN_PRODUCT_TYPES_WITH_CURRENCY.contains(productType)) {
            currencyCode = CurrencyCode.YND_FIXED;
        }

        return calculateProduct(campaignType, productType, currencyCode, quasiCurrencyFlag);
    }

    private Product calculateProduct(CampaignType campaignType, ProductType productType, CurrencyCode currencyCode,
                                     Boolean quasiCurrencyFlag) {
        Set<Product> productSet = calculateProducts(Map.of(productType, Set.of(campaignType)), currencyCode,
                quasiCurrencyFlag);
        //calculateProducts бросит исключение, если продукт не найдется
        checkState(productSet.iterator().hasNext());
        return productSet.iterator().next();
    }

    private Set<Product> calculateProducts(Map<ProductType, Set<CampaignType>> productTypeToCampaignTypes,
                                           CurrencyCode currencyCode,
                                           Boolean quasiCurrencyFlag) {
        List<Product> products = getProducts();
        return EntryStream.of(productTypeToCampaignTypes)
                .mapToValue((productType, campaignTypes) -> calculateProducts(products, campaignTypes,
                        productType,
                        currencyCode, quasiCurrencyFlag))
                .values()
                .flatMap(Collection::stream)
                .toSet();
    }

    private Set<Product> calculateProducts(List<Product> products, Set<CampaignType> campaignTypes,
                                           ProductType productType, CurrencyCode currencyCode,
                                           Boolean quasiCurrencyFlag) {
        return StreamEx.of(campaignTypes)
                .map(campaignType -> calculateProduct(products, campaignType, productType, currencyCode,
                        quasiCurrencyFlag))
                .toSet();
    }

    private Product calculateProduct(List<Product> products, CampaignType campaignType,
                                     ProductType productType, CurrencyCode currencyCode,
                                     Boolean quasiCurrencyFlag) {
        return StreamEx.of(products)
                .filter(product -> productType.equals(product.getType()))
                .filter(product -> currencyCode.equals(product.getCurrencyCode()))
                .filter(product -> !canUseQuasiCurrency(campaignType, currencyCode) ||
                        checkUnitNameOnQuasiCurrency(quasiCurrencyFlag, product))
                .findFirst()
                .orElseThrow(() -> new IllegalStateException("Can not find product info by type: " + productType.name() +
                        " and currencyCode: " + currencyCode.name() +
                        " and quasiCurrencyFlag: " + quasiCurrencyFlag));
    }

    public static boolean checkUnitNameOnQuasiCurrency(Boolean quasiCurrencyFlag, Product product) {
        return quasiCurrencyFlag ?
                product.getUnitName().equals(QUASI_CURRENCY) :
                product.getUnitName().equals(BUCKS);
    }

    private static boolean canUseQuasiCurrency(CampaignType campaignType, CurrencyCode currencyCode) {
        return currencyCode != CurrencyCode.YND_FIXED &&
                (TEXT_CAMPAIGN_IN_BALANCE.contains(campaignType) ||
                        CPM.contains(campaignType));
    }

    public static boolean getClientQuasiCurrencyFlag(Client client) {
        return client.getWorkCurrency().equals(CurrencyCode.KZT) && client.getUsesQuasiCurrency();
    }

    private ProductType calculateProductType(CampaignType campaignType) {
        ProductType productType = PRODUCT_TYPE_BY_CAMPAIGN_TYPES.get(campaignType);
        checkState(productType != null, "Can not calculate product type for campaign type: " + campaignType.name());
        return productType;
    }

    public List<ProductRestriction> getProductRestrictionsNoCache() {
        return productRepository.getAllProductRestrictions();
    }

    /**
     * Вычисляет уникальный ключ ограничения продукта на основе полей productID, groupType, publicNameKey
     */
    public static ProductRestrictionKey calculateUniqueProductRestrictionKey(ProductRestriction pr) {
        return new ProductRestrictionKey()
                .withProductId(pr.getProductId())
                .withAdGroupType(pr.getGroupType())
                .withCriterionType(getCriterionTypeByPublicNameKey(pr.getPublicNameKey()));
    }

    private static CriterionType getCriterionTypeByPublicNameKey(String publicNameKey) {
        if (Strings.isNullOrEmpty(publicNameKey)) {
            return null;
        }

        return criterionTypeByPublicNameKeyMap.get(publicNameKey);
    }

    /**
     * Заменяет все записи ограничений продукта в базе на переданные.
     * <p>
     * Пока не поддерживает добавление и удаление записей
     */
    public void replaceProductRestrictions(Collection<ProductRestriction> newRestrictions) {
        List<ProductRestriction> toAdd = new ArrayList<>();
        List<AppliedChanges<ProductRestriction>> toUpdate = new ArrayList<>();

        dslContextProvider.ppcdict().transaction(tx -> {
            DSLContext dsl = tx.dsl();
            List<ProductRestriction> productRestrictions = productRepository.getAllProductRestrictionsForUpdate(dsl);
            Map<ProductRestrictionKey, ProductRestriction> oldRestrictionsMap =
                    uniqueIndex(productRestrictions, ProductService::calculateUniqueProductRestrictionKey);
            Map<ProductRestrictionKey, ProductRestriction> newRestrictionsMap =
                    uniqueIndex(newRestrictions, ProductService::calculateUniqueProductRestrictionKey);

            ArrayList<ProductRestrictionKey> sortedKeys =
                    new ArrayList<>(Sets.union(oldRestrictionsMap.keySet(), newRestrictionsMap.keySet()));
            sortedKeys.sort(PRODUCT_RESTRICTION_KEY_COMPARATOR);
            for (ProductRestrictionKey key : sortedKeys) {
                var oldPr = oldRestrictionsMap.get(key);
                var newPr = newRestrictionsMap.get(key);

                checkModification(oldPr, newPr);

                if (oldPr == null) {
                    // добавление нового рестрикшена
                    prepareRestrictionForSave(newPr);
                    toAdd.add(newPr);
                } else if (newPr == null) {
                    // удаление существующего рестрикшена
                    // TODO: добавить код удаления рестрикшена
                } else {
                    // обновление существующего рестрикшена
                    if (newPr.getId() == null) {
                        newPr.setId(oldPr.getId());
                    }
                    prepareRestrictionForSave(newPr);

                    AppliedChanges<ProductRestriction> changes = new ModelChanges<>(oldPr.getId(),
                            ProductRestriction.class)
                            .process(newPr.getProductId(), ProductRestriction.PRODUCT_ID)
                            .process(newPr.getGroupType(), ProductRestriction.GROUP_TYPE)
                            .process(newPr.getPublicNameKey(), ProductRestriction.PUBLIC_NAME_KEY)
                            .process(newPr.getPublicDescriptionKey(), ProductRestriction.PUBLIC_DESCRIPTION_KEY)
                            .process(newPr.getConditionJson(), ProductRestriction.CONDITION_JSON)
                            .process(newPr.getUnitCountMin(), ProductRestriction.UNIT_COUNT_MIN)
                            .process(newPr.getUnitCountMax(), ProductRestriction.UNIT_COUNT_MAX)
                            .applyTo(oldPr);

                    // если поменялось поле conditionJson, пробуем привести старое и новое значение к
                    // канонизированному виду
                    // чтобы понять, действительно ли поменялись значения, а не порядок следования внутренних полей
                    if (changes.changed(ProductRestriction.CONDITION_JSON)) {
                        String oldJson = changes.getOldValue(ProductRestriction.CONDITION_JSON);
                        checkState(oldJson != null);
                        String newJson = changes.getNewValue(ProductRestriction.CONDITION_JSON);
                        checkState(newJson != null);
                        if (canonizeConditionJson(oldJson).equals(canonizeConditionJson(newJson))) {
                            changes.modify(ProductRestriction.CONDITION_JSON, oldJson);
                        }
                    }
                    if (changes.hasActuallyChangedProps()) {
                        toUpdate.add(changes);
                        logger.warn("changes to {}: {}", key, changes.toString());
                    }
                }
            }

            if (!toAdd.isEmpty()) {
                productRepository.addProductRestrictions(dsl, toAdd);
            }
            if (!toUpdate.isEmpty()) {
                productRepository.updateProductRestrictions(dsl, toUpdate);
            }
        });
    }

    private void prepareRestrictionForSave(ProductRestriction restriction) {
        if (restriction.getConditionJson() == null && restriction.getConditions() != null) {
            restriction.withConditionJson(JsonUtils.toDeterministicJson(restriction.getConditions()));
        }
    }

    private void checkModification(@Nullable ProductRestriction oldPr, @Nullable ProductRestriction newPr) {

        if (newPr == null) {
            throw new NotImplementedException("Deletion of restrictions is not implemented yet");
        }

        if (oldPr != null && newPr.getId() != null && !newPr.getId().equals(oldPr.getId())) {
            throw new RuntimeException("ID change is forbidden");
        }

        // не верим, что вызывающий код может держать обе проперти в консистентном состоянии
        checkState(newPr.getConditions() == null || newPr.getConditionJson() == null,
                "either conditions or conditionJson must be set in new ProductRestriction"
        );
    }

    private String canonizeConditionJson(String oldJson) {
        List<Map<String, Object>> condition = JsonUtils.fromJson(oldJson, new TypeReference<>() {
        });
        return JsonUtils.toDeterministicJson(condition);
    }

    /**
     * Парсит JSON файл с ограничениями продуктов из ресурсов и возвращает список ядровых моделей-ограничений
     */
    public static List<ProductRestriction> readProductRestrictionFile() {
        String json = LiveResourceFactory.get(PRODUCT_RESTRICTIONS_FILE).getContent();
        ProductRestriction[] prList;
        try {
            prList = productRestrictionsJsonFileMapper.readValue(json, ProductRestriction[].class);
        } catch (IOException e) {
            throw new RuntimeException("Error parsing product restrictions file", e);
        }
        return Arrays.asList(prList);
    }

}
